package com.vcc.playerads.manager;

import android.content.Context;
import android.text.TextUtils;
import android.view.ViewGroup;
import android.widget.FrameLayout;

import com.vcc.playerads.events.AdsEvent;
import com.vcc.playerads.events.IPlayer;
import com.vcc.playerads.events.OnAdsEventListener;
import com.vcc.playerads.models.AdsModel;

import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;

public class SHAdsManager implements ISHAdsManager {
    private Context context;
    private String urlRequest;
    private FrameLayout adsGroupContainer;
    private IPlayer iPlayer;
    private VASTLayout vastLayout;
    private OnAdsEventListener onAdsEventListener;

    /**
     * Contructer SHAdsManger can use by interface ISHAdsManager
     *
     * @param context
     * @param iPlayer            get data, state from playerView
     * @param onAdsEventListener set listen event from Ads
     */

    public SHAdsManager(Context context, IPlayer iPlayer, OnAdsEventListener onAdsEventListener) {
        this(context, "", iPlayer, onAdsEventListener);
    }

    /**
     * Contructer SHAdsManger can use by interface ISHAdsManager can add Url Ads
     *
     * @param context
     * @param url                url ads (example vast 2, vast 3)
     * @param iPlayer            get data, state from playerView
     * @param onAdsEventListener listen event from Ads
     */
    private SHAdsManager(Context context, String url, IPlayer iPlayer, OnAdsEventListener onAdsEventListener) {
        try {
            System.loadLibrary("APIRequest");
        } catch (Exception | UnsatisfiedLinkError e) {
            if (onAdsEventListener != null) {
                AdsEvent adsEvent = new AdsEvent() {
                    @Override
                    public AdsEventType getType() {
                        return AdsEventType.loadError;
                    }

                    @Override
                    public AdsModel getAds() {
                        return null;
                    }
                };
                onAdsEventListener.onAdsEventListener(adsEvent);
            }
            return;
        }
        if (iPlayer != null)
            this.adsGroupContainer = iPlayer.getAdsContainer();
        this.iPlayer = iPlayer;
        this.context = context;
        this.urlRequest = url;
        setOnAdsEventListener(onAdsEventListener);
        initAdsView();
    }

    /**
     * set UrlRequest for Ads
     *
     * @param urlRequest urlRequest
     */
    @Override
    public void setUrlRequest(String urlRequest) {
        this.urlRequest = urlRequest;
    }

    private void initAdsView() {
        if (adsGroupContainer == null)
            return;
        if (vastLayout == null) {
            vastLayout = new VASTLayout(context);
            vastLayout.setIPlayerContent(iPlayer);
            vastLayout.setLayoutParams(new FrameLayout.LayoutParams(
                    ViewGroup.LayoutParams.MATCH_PARENT,
                    ViewGroup.LayoutParams.MATCH_PARENT
            ));

            adsGroupContainer.addView(vastLayout);
            if (onAdsEventListener != null)
                vastLayout.setAdsEventListener(onAdsEventListener);
        }
    }

    /**
     * request Ads default with resetAds after request Ads and skip each Ad
     */
    @Override
    public void requestAds() {
        requestAds(true);
    }

    /**
     * request Ads
     *
     * @param resetAds reset Ad after request
     */
    @Override
    public void requestAds(boolean resetAds) {
        if (TextUtils.isEmpty(urlRequest)) {
            if (onAdsEventListener != null) {
                AdsEvent adsEvent = new AdsEvent() {
                    @Override
                    public AdsEventType getType() {
                        return AdsEventType.linkRequestEmpty;
                    }

                    @Override
                    public AdsModel getAds() {
                        return null;
                    }
                };
                onAdsEventListener.onAdsEventListener(adsEvent);
            }
            return;
        }

        if (vastLayout != null)
            vastLayout.requestAds(urlRequest, resetAds);

    }
    /**
     *
     * @param skipAllAds skip each Ad or All Ads after click Skip Ads
     */

    /**
     * resume Ads when comeback to  Activity
     */
    @Override
    public void resumeAds() {
        if (vastLayout != null)
            vastLayout.resumeAds();
    }

    /**
     * pause Ads when onStop Activity
     */
    @Override
    public void pauseAds() {
        if (vastLayout != null)
            vastLayout.pauseAds();
    }

    @Override
    public void skipAllAds(boolean isSkip) {
        vastLayout.setSkipAllAds(isSkip);
    }

    /**
     * skip Ads
     */
    @Override
    public void skipAds() {
        if (vastLayout != null)
            vastLayout.clickSkipAds();
    }

    /**
     * release Ads
     */
    @Override
    public void releaseAds() {
        if (vastLayout != null)
            vastLayout.resetAllAds();
    }

    /**
     * get Marker (time when show Ads)
     *
     * @return List time can show Ads
     */
    @Override
    public List<String> getMarkerShowAds() {
        if (vastLayout != null) {
            return vastLayout.getListMarkerAds();
        }
        return new CopyOnWriteArrayList<>();
    }

    @Override
    public String getType() {
        if (vastLayout != null) {
            return vastLayout.getTypeOfAds();
        }
        return "";
    }

    @Override
    public String getDurationAds() {
        if (vastLayout != null) {
            return vastLayout.getDurationAds();
        }
        return "";
    }

    @Override
    public int getProgressAds() {
        if (vastLayout != null) {
            return vastLayout.getProgressAds();
        }
        return 0;
    }

    @Override
    public String getAdsId() {
        if (vastLayout != null) {
            return vastLayout.getAdsId();
        }
        return "";
    }

    @Override
    public float getVolumeAds() {
        if (vastLayout != null) {
            return vastLayout.getMusicVolumeLevel();
        }
        return 0.0f;
    }

    /**
     * get State of Ads
     *
     * @return
     */
    @Override
    public boolean isPlayingAds() {
        return vastLayout != null && vastLayout.isPlayingAds();
    }

    public OnAdsEventListener getOnAdsEventListener() {
        return onAdsEventListener;
    }

    /**
     * set Listener from Ads
     *
     * @param onAdsEventListener
     */
    @Override
    public void setOnAdsEventListener(OnAdsEventListener onAdsEventListener) {
        if (vastLayout != null)
            vastLayout.setAdsEventListener(onAdsEventListener);
        this.onAdsEventListener = onAdsEventListener;
    }

    /**
     * mute Ads
     *
     * @param mute or unmute
     */
    @Override
    public void muteAds(boolean mute) {
        if (vastLayout != null)
            vastLayout.muteAds(mute);
    }
}
