package me.lake.librestreaming.customview;

import android.Manifest;
import android.app.Activity;
import android.content.Context;
import android.content.pm.PackageManager;
import android.content.res.Configuration;
import android.graphics.SurfaceTexture;
import android.hardware.Camera;
import android.os.Handler;
import android.os.Message;
import android.support.v4.content.ContextCompat;
import android.support.v7.app.AlertDialog;
import android.util.AttributeSet;
import android.util.Log;
import android.view.TextureView;
import android.view.View;
import android.widget.Toast;

import com.sohalive.http.LibApiSecure;

import java.util.List;

import me.lake.librestreaming.client.RESClient;
import me.lake.librestreaming.core.listener.RESConnectionListener;
import me.lake.librestreaming.filterbase.hardvideofilter.BaseHardVideoFilter;
import me.lake.librestreaming.filterbase.softaudiofilter.BaseSoftAudioFilter;
import me.lake.librestreaming.filterbase.softvideofilter.BaseSoftVideoFilter;
import me.lake.librestreaming.filterpackage.FilterManagers;
import me.lake.librestreaming.filterpackage.audiofilter.SetVolumeAudioFilter;
import me.lake.librestreaming.filterpackage.hardfilter.FilterItemHard;
import me.lake.librestreaming.filterpackage.softfilter.FilterItemSoft;
import me.lake.librestreaming.model.RESConfig;
import me.lake.librestreaming.model.Size;

/**
 * Created by lake on 24/06/16.
 * Librestreaming project.
 */
public class AspectTextureView extends TextureView implements RESConnectionListener,
        TextureView.SurfaceTextureListener {
    //region Initial & Constructor
    public static final int MODE_FITXY = 0;
    public static final int MODE_INSIDE = 1;
    public static final int MODE_OUTSIDE = 2;
    private static final long TIME_DELAY = 3000;
    private static final float MIN_SCALE = 1 / 3.5f;
    private static final String TAG = "aspectexture";
    private float minScale = MIN_SCALE;
    private float xOld;
    private double targetAspect = -1;
    private int aspectMode = MODE_OUTSIDE;
    private CallBackStream callBack;
    private Handler mainHandler;
    private int currentFilter = 0;
    private RESClient resClient;
    private int filterMode = RESConfig.FilterMode.SOFT;
    private Activity mActivity;
    private boolean isStarted = false;
    private AlertDialog alertDialog;
    private String appKey;
    private String secretKey;
    private String linkUp;
    private boolean isNotCreated = false;

    public AspectTextureView(Context context) {
        super(context);
        setKeepScreenOn(true);
        setSurfaceTextureListener(this);
    }

    public AspectTextureView(Context context, AttributeSet attrs) {
        super(context, attrs);
        setKeepScreenOn(true);
        setSurfaceTextureListener(this);
    }

    public AspectTextureView(Context context, AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);
        setKeepScreenOn(true);
        setSurfaceTextureListener(this);
    }

    public float getMinScale() {
        return minScale;
    }

    public void setMinScale(float minScale) {
        this.minScale = minScale;
    }

    public AspectTextureView with(Activity activity) {
        mActivity = activity;
        return this;
    }


    /**
     * @param mode        {@link #MODE_FITXY},{@link #MODE_INSIDE},{@link #MODE_OUTSIDE}
     * @param aspectRatio width/height
     */
    public void setAspectRatio(int mode, double aspectRatio) {
        if (mode != MODE_INSIDE && mode != MODE_OUTSIDE && mode != MODE_FITXY) {
            throw new IllegalArgumentException("illegal mode");
        }
        if (aspectRatio < 0) {
            throw new IllegalArgumentException("illegal aspect ratio");
        }
        if (targetAspect != aspectRatio || aspectMode != mode) {
            targetAspect = aspectRatio;
            aspectMode = mode;
            requestLayout();
        }
    }

    @Override
    protected void onMeasure(int widthMeasureSpec, int heightMeasureSpec) {
        if (targetAspect > 0) {
            int initialWidth = MeasureSpec.getSize(widthMeasureSpec);
            int initialHeight = MeasureSpec.getSize(heightMeasureSpec);

            double viewAspectRatio = (double) initialWidth / initialHeight;
            double aspectDiff = targetAspect / viewAspectRatio - 1;

            if (Math.abs(aspectDiff) > 0.01 && aspectMode != MODE_FITXY) {
                if (aspectMode == MODE_INSIDE) {
                    if (aspectDiff > 0) {
                        initialHeight = (int) (initialWidth / targetAspect);
                    } else {
                        initialWidth = (int) (initialHeight * targetAspect);
                    }
                } else if (aspectMode == MODE_OUTSIDE) {
                    if (aspectDiff > 0) {
                        initialWidth = (int) (initialHeight * targetAspect);
                    } else {
                        initialHeight = (int) (initialWidth / targetAspect);
                    }
                }
                widthMeasureSpec = MeasureSpec.makeMeasureSpec(initialWidth, MeasureSpec.EXACTLY);
                heightMeasureSpec = MeasureSpec.makeMeasureSpec(initialHeight, MeasureSpec.EXACTLY);
            }
        }
        super.onMeasure(widthMeasureSpec, heightMeasureSpec);
    }

    @Override
    public void layout(int l, int t, int r, int b) {
        View p = (View) getParent();
        if (p != null) {
            int pw = p.getMeasuredWidth();
            int ph = p.getMeasuredHeight();
            int w = getMeasuredWidth();
            int h = getMeasuredHeight();
            t = (ph - h) / 2;
            l = (pw - w) / 2;
            r += l;
            b += t;
        }
        super.layout(l, t, r, b);
    }
    //endregion

    //region GPU streaming
    @Override
    public void onOpenConnectionResult(int result) {
        if (result == 0) {
            Log.i(TAG, "server IP = " + resClient.getServerIpAddr());
        }
        /**
         * result==0 success
         * result!=0 failed
         */
//        tv_rtmp.setText("open=" + result);
    }

    @Override
    public void onWriteError(int error) {
        if (error == 100) {
            resClient.stop();
            resClient.start();
        }
        /**
         * failed to write data,maybe restart.
         */
        Log.e(TAG, "writeError=" + error);
    }

    @Override
    public void onCloseConnectionResult(int result) {
        /**
         * result==0 success
         * result!=0 failed
         */
        Log.i(TAG, "close=" + result);
    }

    @Override
    public void onSurfaceTextureAvailable(SurfaceTexture surface, int width, int height) {
        if (resClient != null) {
            try {
                resClient.createPreview(surface, width, height);
            } catch (Exception ex) {
                Log.e(TAG, "onSurfaceTextureAvailable: " + ex.getMessage());
            }
        } else {
            isNotCreated = true;
        }
    }

    @Override
    public void onSurfaceTextureSizeChanged(SurfaceTexture surface, int width, int height) {
        if (resClient != null) {
            try {
                if (isNotCreated)
                    resClient.createPreview(surface, width, height);
                else
                    resClient.updatePreview(width, height);
            } catch (Exception ex) {
                Log.e(TAG, "onSurfaceTextureSizeChanged: " + ex.getMessage());
            }
        }
    }

    @Override
    public boolean onSurfaceTextureDestroyed(SurfaceTexture surface) {
        if (resClient != null) {
            try {
                resClient.destroyPreview();
            } catch (Exception ex) {
                Log.e(TAG, ex.toString());
            }
        }
        return true;
    }

    @Override
    public void onSurfaceTextureUpdated(SurfaceTexture surface) {

    }

    public boolean toggleFlashLight() {
        return resClient != null && resClient.toggleFlashLight();
    }

    //endregion

    //region Config stream video and camera
    public void switchCamera() {
        if (isStarted()) {
            if (Camera.getNumberOfCameras() > 1) {
                //switch camera
                resClient.swapCamera();
            }
        } else {
            Toast.makeText(mActivity, "Camera is switchable when streaming!", Toast.LENGTH_SHORT).show();
        }
    }

    public void zoomVideo(float progress) {
        resClient.setZoomByPercent(progress / 100.0f);
    }

    public void changeVolume(float progress) {
        BaseSoftAudioFilter audioFilter = resClient.acquireSoftAudioFilter();
        if (audioFilter != null) {
            if (audioFilter instanceof SetVolumeAudioFilter) {
                SetVolumeAudioFilter blackWhiteFilter = (SetVolumeAudioFilter) audioFilter;
                blackWhiteFilter.setVolumeScale((float) (progress / 10.0));
            }
        }
        resClient.releaseSoftAudioFilter();
    }


    public AspectTextureView setCallBack(CallBackStream callBack) {
        this.callBack = callBack;
        return this;
    }

    public void destroyStream() {
        try {
            resClient.stop();
        } catch (Exception ex) {
            Log.e(TAG, ex.toString());
        }
        if (mainHandler != null) {
            mainHandler.removeCallbacksAndMessages(null);
        }
        if (resClient != null) {
            resClient.destroy();
        }
    }

    public void nextFilter() {
        if (!isStarted())
            return;
        switch (filterMode) {
            case RESConfig.FilterMode.HARD:

                if (FilterManagers.getInstance(mActivity)
                        .nextFilterHard(resClient, currentFilter))
                    currentFilter++;
                else currentFilter = 0;
                break;
            case RESConfig.FilterMode.SOFT:
                if (FilterManagers.getInstance(mActivity)
                        .nextFilterSoft(resClient, currentFilter))
                    if (currentFilter < FilterManagers.getInstance(mActivity).getFilterSoft().size() - 1)
                        currentFilter++;
                    else
                        currentFilter = 0;
                else currentFilter = 0;
                break;
        }
    }

    /**
     * @param filterHard
     * @deprecated
     */
    public void setFilterHard(List<FilterItemHard> filterHard) {
        FilterManagers.getInstance(mActivity).setFilterHard(filterHard);
    }

    public void setFilterSoft(List<FilterItemSoft> filterSoft) {

        FilterManagers.getInstance(mActivity).setFilterSoft(filterSoft);
    }

    /**
     * @param baseHardVideoFilter
     * @deprecated
     */
    public void setFilterEffect(BaseHardVideoFilter baseHardVideoFilter, int progress) {
        FilterManagers.getInstance(mActivity)
                .setFilterHard(resClient, baseHardVideoFilter, progress);
    }

    public void setFilterEffect(BaseSoftVideoFilter baseSoftVideoFilter, int progress) {
        FilterManagers.getInstance(mActivity)
                .setFilterSoft(resClient, baseSoftVideoFilter, progress);
    }

    public boolean isStarted() {
        return isStarted;
    }

    public void setStarted(boolean started) {
        isStarted = started;
    }

    public boolean startStream() {
        try {
            resClient.start();
            setStarted(true);
            return true;
        } catch (Exception ex) {
            return false;
        }
    }

    public boolean stopStream() {
        try {
            resClient.stop();
            setStarted(false);
            return true;
        } catch (Exception ex) {
            return false;
        }
    }

    public void handleMotion(int position, float positionOffset, int positionOffsetPixels) {
        if (position == 0) {
            float scaleFactor = getMinScale()
                    + (1 - getMinScale()) * positionOffset;
            setPivotX(getRight() - Utils.getInstance(mActivity).convertDpToPx(10.0f));
            setPivotY(getBottom() - Utils.getInstance(mActivity).convertDpToPx(10.0f));
            if (scaleFactor <= 1) {
                setScaleX(scaleFactor);
                setScaleY(scaleFactor);
            }
        }
    }

    public String getAppKey() {
        return appKey;
    }

    public AspectTextureView setAppKey(String appKey) {
        this.appKey = appKey;
        return this;
    }

    public String getSecretKey() {
        return secretKey;
    }

    public AspectTextureView setSecretKey(String secretKey) {
        this.secretKey = secretKey;
        return this;
    }

    public String getLinkUp() {
        return linkUp;
    }

    public AspectTextureView setLinkUp(String linkUp) {
        this.linkUp = linkUp;
        return this;
    }

    public void build() {
        if (!checkPermission(Manifest.permission.INTERNET)
                || !checkPermission(Manifest.permission.CAMERA)
                || !checkPermission(Manifest.permission.RECORD_AUDIO)
                || !checkPermission(Manifest.permission.MODIFY_AUDIO_SETTINGS)
                ) {
            if (callBack == null) {
                Log.e("Player sdk: ", "OnPlayerCallback cannot null");
            } else {
                callBack.configError("You have not permission internet. Please check again!");
            }
        } else {
            LibApiSecure.Builder(mActivity).setAppKey(appKey)
                    .setSecretKey(secretKey)
                    .setOnCallbackPermission(new LibApiSecure.OnCallbackPermission() {
                        @Override
                        public void permissionCallBack(final boolean isValid) {
                            post(new Runnable() {
                                @Override
                                public void run() {
                                    if (isValid) {
                                        initial(linkUp);
                                        Log.d(TAG, "Permission granted!");
                                    } else callBack.configError("You have not permission.");
                                }
                            });
                        }
                    })
                    .build();
        }
    }

    private void initial(String rtmpAddr) {
        try {
            filterMode = RESConfig.FilterMode.SOFT;
            resClient = new RESClient();

            RESConfig resConfig = RESConfig.obtain();
            resConfig.setFilterMode(filterMode);
            resConfig.setTargetVideoSize(new Size(1280, 720));
            resConfig.setBitRate(1000 * 1024);
            resConfig.setRenderingMode(RESConfig.RenderingMode.OpenGLES);
            resConfig.setDefaultCamera(Camera.CameraInfo.CAMERA_FACING_FRONT);

            int frontDirection, backDirection;
            Camera.CameraInfo cameraInfo = new Camera.CameraInfo();

            Camera.getCameraInfo(Camera.CameraInfo.CAMERA_FACING_FRONT, cameraInfo);
            frontDirection = cameraInfo.orientation;

            Camera.getCameraInfo(Camera.CameraInfo.CAMERA_FACING_BACK, cameraInfo);
            backDirection = cameraInfo.orientation;
            if (getResources().getConfiguration().orientation == Configuration.ORIENTATION_PORTRAIT) {
                resConfig.setFrontCameraDirectionMode((frontDirection == 90 ? RESConfig.DirectionMode.FLAG_DIRECTION_ROATATION_270 : RESConfig.DirectionMode.FLAG_DIRECTION_ROATATION_90) | RESConfig.DirectionMode.FLAG_DIRECTION_FLIP_HORIZONTAL);
                resConfig.setBackCameraDirectionMode((backDirection == 90 ? RESConfig.DirectionMode.FLAG_DIRECTION_ROATATION_90 : RESConfig.DirectionMode.FLAG_DIRECTION_ROATATION_270));
            } else {
                resConfig.setBackCameraDirectionMode((backDirection == 90 ? RESConfig.DirectionMode.FLAG_DIRECTION_ROATATION_0 : RESConfig.DirectionMode.FLAG_DIRECTION_ROATATION_180));
                resConfig.setFrontCameraDirectionMode((frontDirection == 90 ? RESConfig.DirectionMode.FLAG_DIRECTION_ROATATION_180 : RESConfig.DirectionMode.FLAG_DIRECTION_ROATATION_0) | RESConfig.DirectionMode.FLAG_DIRECTION_FLIP_HORIZONTAL);
            }
            resConfig.setRtmpAddr(rtmpAddr);

            if (!resClient.prepare(resConfig)) {
                resClient = null;
                callBack.configError("Config Failed!");
                Toast.makeText(mActivity, "Cannot initial config video!", Toast.LENGTH_SHORT).show();
                return;
            }

            post(new Runnable() {
                @Override
                public void run() {
                    Size s = resClient.getVideoSize();
                    AspectTextureView.this.setAspectRatio(AspectTextureView.MODE_INSIDE, ((double) s.getWidth()) / s.getHeight());
                    Log.d(TAG, "version=" + resClient.getVertion());

                    mainHandler = new Handler() {
                        @Override
                        public void handleMessage(Message msg) {
                            Log.i(TAG, "byteSpeed=" + (resClient.getAVSpeed() / 1024) + ";drawFPS=" + resClient.getDrawFrameRate() + ";sendFPS=" + resClient.getSendFrameRate());
                            sendEmptyMessageDelayed(0, TIME_DELAY);
                        }
                    };
                    mainHandler.sendEmptyMessageDelayed(0, TIME_DELAY);
                }
            });

            resClient.setConnectionListener(this);

            resClient.setSoftAudioFilter(new SetVolumeAudioFilter());

            if (isAvailable())
                onSurfaceTextureSizeChanged(getSurfaceTexture(), getWidth(), getHeight());
            post(new Runnable() {
                @Override
                public void run() {
                    callBack.configSuccess("Config successful! Now, you can start your live session.");
                }
            });
        } catch (Exception ex) {
            callBack.configError("Config Failed!");
            post(new Runnable() {
                @Override
                public void run() {
                    Toast.makeText(mActivity, "Cannot initial config video!", Toast.LENGTH_SHORT).show();
                    mActivity.finish();
                }
            });
        }
    }

    private boolean checkPermission(String permissionStr) {
        // Here, thisActivity is the current activity
        if (ContextCompat.checkSelfPermission(mActivity, permissionStr)
                != PackageManager.PERMISSION_GRANTED) {
            return false;
        }
        return true;
    }

    //endregion
}
