/*
 * Copyright 2015 The original authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.vaadin.spring.boot.internal;

import javax.servlet.http.HttpServlet;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.NoSuchBeanDefinitionException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.context.embedded.ServletRegistrationBean;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import com.vaadin.server.Constants;
import com.vaadin.spring.server.SpringVaadinServlet;

/**
 * Spring configuration that sets up a
 * {@link com.vaadin.spring.server.SpringVaadinServlet}. If you want to
 * customize the servlet, extend it and make it available as a Spring bean.
 *
 * @author Petter Holmström (petter@vaadin.com)
 * @author Henri Sara (hesara@vaadin.com)
 */
@Configuration
@EnableConfigurationProperties(VaadinServletConfigurationProperties.class)
public class VaadinServletConfiguration implements InitializingBean {

    public static final String DEFAULT_SERVLET_URL_MAPPING = "/*";

    /**
     * Mapping for static resources that is used in case a non-default mapping
     * is used as the primary mapping.
     */
    public static final String STATIC_RESOURCES_URL_MAPPING = "/VAADIN/*";

    private static Logger logger = LoggerFactory
            .getLogger(VaadinServletConfiguration.class);

    @Autowired
    protected ApplicationContext applicationContext;
    @Autowired
    protected VaadinServletConfigurationProperties configurationProperties;

    protected Class<? extends HttpServlet> getServletClass() {
        return SpringVaadinServlet.class;
    }

    protected Logger getLogger() {
        return logger;
    }

    protected String[] getUrlMappings() {
        String mapping = configurationProperties.getUrlMapping();
        if (mapping == null || "".equals(mapping.trim())) {
            mapping = DEFAULT_SERVLET_URL_MAPPING;
        }
        if (!DEFAULT_SERVLET_URL_MAPPING.equals(mapping)) {
            return new String[] { mapping, STATIC_RESOURCES_URL_MAPPING };
        } else {
            return new String[] { mapping };
        }
    }

    @Bean
    ServletRegistrationBean vaadinServletRegistration() {
        return createServletRegistrationBean();
    }

    @Override
    public void afterPropertiesSet() throws Exception {
        getLogger().debug("{} initialized", getClass().getName());
    }

    private HttpServlet newServletInstance() {
        try {
            return getServletClass().newInstance();
        } catch (Exception e) {
            throw new RuntimeException("Could not create new servlet instance",
                    e);
        }
    }

    protected HttpServlet createServlet() {
        HttpServlet servlet;
        try {
            servlet = applicationContext.getBean(getServletClass());
            getLogger()
                    .info("Using servlet instance [{}] found in the application context",
                            servlet);
        } catch (NoSuchBeanDefinitionException ex) {
            getLogger()
                    .info("Servlet was not found in the application context, using default");
            servlet = newServletInstance();
        }
        return servlet;
    }

    protected ServletRegistrationBean createServletRegistrationBean() {
        getLogger().info("Registering servlet of type [{}]",
                getServletClass().getCanonicalName());
        final String[] urlMappings = getUrlMappings();
        getLogger().info("Servlet will be mapped to URLs {}",
                (Object[]) urlMappings);
        final HttpServlet servlet = createServlet();
        final ServletRegistrationBean registrationBean = new ServletRegistrationBean(
                servlet, urlMappings);
        addInitParameters(registrationBean);
        return registrationBean;
    }

    protected void addInitParameters(
            ServletRegistrationBean servletRegistrationBean) {
        getLogger().info("Setting servlet init parameters");

        addInitParameter(servletRegistrationBean,
                Constants.SERVLET_PARAMETER_PRODUCTION_MODE,
                String.valueOf(configurationProperties.isProductionMode()));
        addInitParameter(servletRegistrationBean,
                Constants.SERVLET_PARAMETER_RESOURCE_CACHE_TIME,
                String.valueOf(configurationProperties.getResourceCacheTime()));
        addInitParameter(servletRegistrationBean,
                Constants.SERVLET_PARAMETER_HEARTBEAT_INTERVAL,
                String.valueOf(configurationProperties.getHeartbeatInterval()));
        addInitParameter(servletRegistrationBean,
                Constants.SERVLET_PARAMETER_CLOSE_IDLE_SESSIONS,
                String.valueOf(configurationProperties.isCloseIdleSessions()));

        addInitParameter(servletRegistrationBean,
                Constants.PARAMETER_VAADIN_RESOURCES,
                configurationProperties.getResources());
    }

    private void addInitParameter(
            ServletRegistrationBean servletRegistrationBean, String paramName,
            String propertyValue) {
        if (propertyValue != null) {
            getLogger().info("Set servlet init parameter [{}] = [{}]",
                    paramName, propertyValue);
            servletRegistrationBean.addInitParameter(paramName, propertyValue);
        }
    }

}
