/*
 * Copyright (C) 2000-2023 Vaadin Ltd
 *
 * This program is available under Vaadin Commercial License and Service Terms.
 *
 * See <https://vaadin.com/commercial-license-and-service-terms> for the full
 * license.
 */
package com.vaadin.shared.ui;

import java.io.Serializable;
import java.util.HashSet;
import java.util.List;

import com.vaadin.shared.AbstractComponentState;
import com.vaadin.shared.Registration;
import com.vaadin.shared.communication.SharedState;

public final class ComponentStateUtil implements Serializable {

    private ComponentStateUtil() {
        // Util class is not instantiable
    }

    public static final boolean isUndefinedWidth(AbstractComponentState state) {
        return state.width == null || state.width.isEmpty();
    }

    public static final boolean isUndefinedHeight(
            AbstractComponentState state) {
        return state.height == null || state.height.isEmpty();
    }

    public static final boolean hasDescription(AbstractComponentState state) {
        return state.description != null && !state.description.isEmpty();
    }

    public static final boolean hasStyles(AbstractComponentState state) {
        return hasStyles(state.styles);
    }

    public static final boolean hasStyles(List<String> styles) {
        return styles != null && !styles.isEmpty();
    }

    public static final boolean isRelativeWidth(AbstractComponentState state) {
        return state.width != null && state.width.endsWith("%");
    }

    public static final boolean isRelativeHeight(AbstractComponentState state) {
        return state.height != null && state.height.endsWith("%");
    }

    /**
     * Removes an event listener id.
     *
     * @param state
     *            shared state
     * @param eventIdentifier
     *            The event identifier to remove
     * @deprecated Use a {@link Registration} object returned by
     *             {@link #addRegisteredEventListener(SharedState, String)} to
     *             remove a listener
     */
    @Deprecated
    public static final void removeRegisteredEventListener(SharedState state,
            String eventIdentifier) {
        if (state.registeredEventListeners == null) {
            return;
        }
        state.registeredEventListeners.remove(eventIdentifier);
        if (state.registeredEventListeners.size() == 0) {
            state.registeredEventListeners = null;
        }
    }

    /**
     * Adds an event listener id.
     *
     * @param eventListenerId
     *            The event identifier to add
     * @return a registration object for removing the listener
     * @since 8.0
     */
    public static final Registration addRegisteredEventListener(
            SharedState state, String eventListenerId) {
        if (state.registeredEventListeners == null) {
            state.registeredEventListeners = new HashSet<>();
        }
        state.registeredEventListeners.add(eventListenerId);
        return () -> removeRegisteredEventListener(state, eventListenerId);
    }
}
