/*
 * Copyright (C) 2000-2024 Vaadin Ltd
 *
 * This program is available under Vaadin Commercial License and Service Terms.
 *
 * See <https://vaadin.com/commercial-license-and-service-terms> for the full
 * license.
 */

package com.vaadin.data.converter;

import java.util.Locale;

import com.vaadin.data.Converter;
import com.vaadin.data.ErrorMessageProvider;
import com.vaadin.data.Result;
import com.vaadin.data.ValueContext;

/**
 * A converter that converts from {@link String} to {@link Boolean} and back.
 * The String representation is given by {@link Boolean#toString()} or provided
 * in constructor
 * {@link StringToBooleanConverter#StringToBooleanConverter(String, String, String)}.
 * <p>
 * Leading and trailing white spaces are ignored when converting from a String.
 * </p>
 * <p>
 * For language-dependent representation, subclasses should overwrite
 * {@link #getFalseString(Locale)} and {@link #getTrueString(Locale)}
 * </p>
 *
 * @author Vaadin Ltd
 * @since 8.0
 */
public class StringToBooleanConverter implements Converter<String, Boolean> {

    private final String trueString;

    private final String falseString;

    private ErrorMessageProvider errorMessageProvider;

    /**
     * Creates converter with default string representations - "true" and
     * "false".
     *
     * @param errorMessage
     *            the error message to use if conversion fails
     */
    public StringToBooleanConverter(String errorMessage) {
        this(errorMessage, Boolean.TRUE.toString(), Boolean.FALSE.toString());
    }

    /**
     * Creates a new converter instance with the given error message provider.
     * Empty strings are converted to <code>null</code>.
     *
     * @param errorMessageProvider
     *            the error message provider to use if conversion fails
     *
     * @since 8.4
     */
    public StringToBooleanConverter(ErrorMessageProvider errorMessageProvider) {
        this(Boolean.TRUE.toString(), Boolean.FALSE.toString(),
                errorMessageProvider);
    }

    /**
     * Creates converter with custom string representation.
     *
     * @param errorMessage
     *            the error message to use if conversion fails
     * @param falseString
     *            string representation for <code>false</code>
     * @param trueString
     *            string representation for <code>true</code>
     */
    public StringToBooleanConverter(String errorMessage, String trueString,
            String falseString) {
        this(trueString, falseString, ctx -> errorMessage);
    }

    /**
     * Creates converter with custom string representation.
     *
     * @param falseString
     *            string representation for <code>false</code>
     * @param trueString
     *            string representation for <code>true</code>
     * @param errorMessageProvider
     *            the error message provider to use if conversion fails
     *
     * @since 8.4
     */
    public StringToBooleanConverter(String trueString, String falseString,
            ErrorMessageProvider errorMessageProvider) {
        this.errorMessageProvider = errorMessageProvider;
        this.trueString = trueString;
        this.falseString = falseString;
    }

    @Override
    public Result<Boolean> convertToModel(String value, ValueContext context) {
        if (value == null) {
            return Result.ok(null);
        }

        // Remove leading and trailing white space
        value = value.trim();

        Locale locale = context.getLocale().orElse(null);
        if (getTrueString(locale).equals(value)) {
            return Result.ok(true);
        } else if (getFalseString(locale).equals(value)) {
            return Result.ok(false);
        } else if (value.isEmpty()) {
            return Result.ok(null);
        } else {
            return Result.error(errorMessageProvider.apply(context));
        }
    }

    @Override
    public String convertToPresentation(Boolean value, ValueContext context) {
        if (value == null) {
            return null;
        }
        Locale locale = context.getLocale().orElse(null);
        if (value) {
            return getTrueString(locale);
        } else {
            return getFalseString(locale);
        }
    }

    /**
     * Gets the locale-depended string representation for false. Default is
     * locale-independent value {@code false}
     *
     * @param locale
     *            to be used
     * @return the string representation for false
     */
    protected String getFalseString(Locale locale) {
        return falseString;
    }

    /**
     * Gets the locale-depended string representation for true. Default is
     * locale-independent value {@code true}
     *
     * @param locale
     *            to be used
     * @return the string representation for true
     */
    protected String getTrueString(Locale locale) {
        return trueString;
    }

}
