/*
 * Copyright (C) 2000-2024 Vaadin Ltd
 *
 * This program is available under Vaadin Commercial License and Service Terms.
 *
 * See <https://vaadin.com/commercial-license-and-service-terms> for the full
 * license.
 */
package com.vaadin.event;

import com.vaadin.ui.AbstractField;
import com.vaadin.ui.Component.Focusable;

/**
 * A ready-made {@link ShortcutListener} that focuses the given
 * {@link Focusable} (usually an {@link AbstractField}) when 
 * the keyboard shortcut is invoked.
 *
 * @author Vaadin Ltd
 * @since 8.7
 */
public class FocusShortcut extends ShortcutListener {
    protected Focusable focusable;

    /**
     * Creates a keyboard shortcut for focusing the given {@link Focusable}
     * using the shorthand notation defined in {@link ShortcutAction}.
     *
     * @param focusable
     *            to focused when the shortcut is invoked
     * @param shorthandCaption
     *            caption with keycode and modifiers indicated
     */
    public FocusShortcut(Focusable focusable, String shorthandCaption) {
        super(shorthandCaption);
        this.focusable = focusable;
    }

    /**
     * Creates a keyboard shortcut for focusing the given {@link Focusable}.
     *
     * @param focusable
     *            to focused when the shortcut is invoked
     * @param keyCode
     *            keycode that invokes the shortcut
     * @param modifiers
     *            modifiers required to invoke the shortcut
     */
    public FocusShortcut(Focusable focusable, int keyCode, int... modifiers) {
        super(null, keyCode, modifiers);
        this.focusable = focusable;
    }

    /**
     * Creates a keyboard shortcut for focusing the given {@link Focusable}.
     *
     * @param focusable
     *            to focused when the shortcut is invoked
     * @param keyCode
     *            keycode that invokes the shortcut
     */
    public FocusShortcut(Focusable focusable, int keyCode) {
        this(focusable, keyCode, null);
    }

    @Override
    public void handleAction(Object sender, Object target) {
        focusable.focus();
    }
}
