/*
 * Copyright (C) 2000-2023 Vaadin Ltd
 *
 * This program is available under Vaadin Commercial License and Service Terms.
 *
 * See <https://vaadin.com/commercial-license-and-service-terms> for the full
 * license.
 */

package com.vaadin.annotations;

import java.lang.annotation.Documented;
import java.lang.annotation.ElementType;
import java.lang.annotation.Inherited;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;

import com.vaadin.server.Constants;
import com.vaadin.server.DefaultDeploymentConfiguration;
import com.vaadin.server.DeploymentConfiguration;
import com.vaadin.server.VaadinServlet;
import com.vaadin.server.VaadinSession;
import com.vaadin.ui.UI;

/**
 * Annotation for configuring subclasses of {@link VaadinServlet}. For a
 * {@link VaadinServlet} class that has this annotation, the defined values are
 * read during initialization and will be available using
 * {@link DeploymentConfiguration#getApplicationOrSystemProperty(String, String)}
 * as well as from specific methods in {@link DeploymentConfiguration}. Init
 * params defined in <code>web.xml</code> or the <code>@WebServlet</code>
 * annotation take precedence over values defined in this annotation.
 *
 * @since 7.1
 * @author Vaadin Ltd
 */
@Retention(RetentionPolicy.RUNTIME)
@Target(ElementType.TYPE)
@Inherited
public @interface VaadinServletConfiguration {
    /**
     * Defines the init parameter name for methods in
     * {@link VaadinServletConfiguration}.
     *
     * @since 7.1
     * @author Vaadin Ltd
     */
    @Retention(RetentionPolicy.RUNTIME)
    @Target(ElementType.METHOD)
    @Documented
    public @interface InitParameterName {
        /**
         * The name of the init parameter that the annotated method controls.
         *
         * @return the parameter name
         */
        public String value();
    }

    /**
     * Whether Vaadin is in production mode.
     *
     * @return true if in production mode, false otherwise.
     *
     * @see DeploymentConfiguration#isProductionMode()
     */
    @InitParameterName(Constants.SERVLET_PARAMETER_PRODUCTION_MODE)
    public boolean productionMode();

    /**
     * Gets the default UI class to use for the servlet.
     *
     * @return the default UI class
     */
    @InitParameterName(VaadinSession.UI_PARAMETER)
    public Class<? extends UI> ui();

    /**
     * Gets the priority of the default UI class for the servlet.
     * When faced with multiple UI providers, higher priority providers
     * are served first.
     *
     * @return the priority of the default UI class (defaults to 0)
     * 
     * @see DeploymentConfiguration#getUIProviderPriority()
     * 
     * @since 8.21
     */
    @InitParameterName(VaadinSession.UI_PRIORITY_PARAMETER)
    public int uiPriority() default 0;

    /**
     * The time resources can be cached in the browser, in seconds. The default
     * value is 3600 seconds, i.e. one hour.
     *
     * @return the resource cache time
     *
     * @see DeploymentConfiguration#getResourceCacheTime()
     */
    @InitParameterName(Constants.SERVLET_PARAMETER_RESOURCE_CACHE_TIME)
    public int resourceCacheTime() default DefaultDeploymentConfiguration.DEFAULT_RESOURCE_CACHE_TIME;

    /**
     * The number of seconds between heartbeat requests of a UI, or a
     * non-positive number if heartbeat is disabled. The default value is 300
     * seconds, i.e. 5 minutes.
     *
     * @return the time between heartbeats
     *
     * @see DeploymentConfiguration#getHeartbeatInterval()
     */
    @InitParameterName(Constants.SERVLET_PARAMETER_HEARTBEAT_INTERVAL)
    public int heartbeatInterval() default DefaultDeploymentConfiguration.DEFAULT_HEARTBEAT_INTERVAL;

    /**
     * Whether a session should be closed when all its open UIs have been idle
     * for longer than its configured maximum inactivity time. The default value
     * is <code>false</code>.
     *
     * @return true if UIs and sessions receiving only heartbeat requests are
     *         eventually closed; false if heartbeat requests extend UI and
     *         session lifetime indefinitely
     *
     * @see DeploymentConfiguration#isCloseIdleSessions()
     */
    @InitParameterName(Constants.SERVLET_PARAMETER_CLOSE_IDLE_SESSIONS)
    public boolean closeIdleSessions() default DefaultDeploymentConfiguration.DEFAULT_CLOSE_IDLE_SESSIONS;

    /**
     * The default widgetset to use for the servlet. The default value is
     * <code>""</code>, which will cause
     * <code>com.vaadin.DefaultWidgetSet</code> to be used unless overridden by
     * an init parameter or unless an automatically generated
     * <code>AppWidgetset</code> is used.
     *
     * @return the default widgetset name
     */
    @InitParameterName(VaadinServlet.PARAMETER_WIDGETSET)
    public String widgetset() default "";

}
