/*
 * Copyright (C) 2000-2023 Vaadin Ltd
 *
 * This program is available under Vaadin Commercial License and Service Terms.
 *
 * See <https://vaadin.com/commercial-license-and-service-terms> for the full
 * license.
 */

package com.vaadin.ui;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Objects;

import org.jsoup.nodes.Element;

import com.vaadin.shared.ui.ValueChangeMode;
import com.vaadin.shared.ui.richtextarea.RichTextAreaClientRpc;
import com.vaadin.shared.ui.richtextarea.RichTextAreaServerRpc;
import com.vaadin.shared.ui.richtextarea.RichTextAreaState;
import com.vaadin.shared.ui.richtextarea.Snippet;
import com.vaadin.ui.declarative.DesignContext;

import elemental.json.Json;

/**
 * A simple RichTextArea to edit HTML format text.
 */
public class RichTextArea extends AbstractField<String>
        implements HasValueChangeMode {

    private class RichTextAreaServerRpcImpl implements RichTextAreaServerRpc {
        @Override
        public void setText(String text) {
            updateDiffstate("value", Json.create(text));
            if (!setValue(text, true)) {
                // The value was not updated, this could happen if the field has
                // been set to readonly on the server and the client does not
                // know about it yet. Must re-send the correct state back.
                markAsDirty();
            }
        }
    }

    /**
     * Constructs an empty <code>RichTextArea</code> with no caption.
     */
    public RichTextArea() {
        super();
        registerRpc(new RichTextAreaServerRpcImpl());
        setValue("");
    }

    /**
     * Constructs an empty <code>RichTextArea</code> with the given caption.
     *
     * @param caption
     *            the caption for the editor.
     */
    public RichTextArea(String caption) {
        this();
        setCaption(caption);
    }

    /**
     * Constructs a new <code>RichTextArea</code> with the given caption and
     * initial text contents.
     *
     * @param caption
     *            the caption for the editor.
     * @param value
     *            the initial text content of the editor, not {@code null}
     */
    public RichTextArea(String caption, String value) {
        this(caption);
        setValue(value);
    }

    /**
     * Constructs a new {@code RichTextArea} with a value change listener.
     * <p>
     * The listener is called when the value of this {@code TextField} is
     * changed either by the user or programmatically.
     *
     * @param valueChangeListener
     *            the value change listener, not {@code null}
     * @since 8.0
     */
    public RichTextArea(ValueChangeListener<String> valueChangeListener) {
        addValueChangeListener(valueChangeListener);
    }

    /**
     * Constructs a new {@code RichTextArea} with the given caption and a value
     * change listener.
     * <p>
     * The listener is called when the value of this {@code TextField} is
     * changed either by the user or programmatically.
     *
     * @param caption
     *            the caption for the field
     * @param valueChangeListener
     *            the value change listener, not {@code null}
     * @since 8.0
     */
    public RichTextArea(String caption,
            ValueChangeListener<String> valueChangeListener) {
        this(valueChangeListener);
        setCaption(caption);
    }

    /**
     * Constructs a new {@code RichTextArea} with the given caption, initial
     * text contents and a value change listener.
     * <p>
     * The listener is called when the value of this {@code RichTextArea} is
     * changed either by the user or programmatically.
     *
     * @param caption
     *            the caption for the field
     * @param value
     *            the value for the field, not {@code null}
     * @param valueChangeListener
     *            the value change listener, not {@code null}
     * @since 8.0
     */
    public RichTextArea(String caption, String value,
            ValueChangeListener<String> valueChangeListener) {
        this(caption, value);
        addValueChangeListener(valueChangeListener);
    }

    @Override
    public void readDesign(Element design, DesignContext designContext) {
        super.readDesign(design, designContext);
        setValue(design.html());
    }

    @Override
    public void writeDesign(Element design, DesignContext designContext) {
        super.writeDesign(design, designContext);
        design.html(getValue());
    }

    @Override
    protected RichTextAreaState getState() {
        return (RichTextAreaState) super.getState();
    }

    @Override
    protected RichTextAreaState getState(boolean markAsDirty) {
        return (RichTextAreaState) super.getState(markAsDirty);
    }

    /**
     * Sets the value of this object. If the new value is not equal to
     * {@code getValue()}, fires a {@link ValueChangeEvent}. Throws
     * {@code NullPointerException} if the value is null.
     *
     * @param value
     *            the new value, not {@code null}
     * @throws NullPointerException
     *             if {@code value} is {@code null}
     */
    @Override
    public void setValue(String value) {
        Objects.requireNonNull(value, "value cannot be null");
        setValue(value, false);
    }

    @Override
    public String getValue() {
        return getState(false).value;
    }

    @Override
    public String getEmptyValue() {
        return "";
    }

    @Override
    protected void doSetValue(String value) {
        getState().value = value;
    }

    /**
     * Selects all text in the rich text area. As a side effect, focuses the
     * rich text area.
     *
     * @since 6.5
     */
    public void selectAll() {
        getRpcProxy(RichTextAreaClientRpc.class).selectAll();
        focus();
    }

    @Override
    public void setValueChangeMode(ValueChangeMode mode) {
        getState().valueChangeMode = mode;
    }

    @Override
    public ValueChangeMode getValueChangeMode() {
        return getState(false).valueChangeMode;
    }

    @Override
    public void setValueChangeTimeout(int timeout) {
        getState().valueChangeTimeout = timeout;

    }

    @Override
    public int getValueChangeTimeout() {
        return getState(false).valueChangeTimeout;
    }

    /**
     * Add a predefined content snippet to the menu. HTML formatting is allowed.
     * Drop-down menu is visible after the first snippet is added. The title of
     * the snippet is automatically generated from the first 20 characters of
     * the snippet.
     * 
     * @param snippet
     *            snippet text, can be HTML formatted
     * @since 8.16
     */
    public void addSnippet(String snippet) {
        addSnippet(new Snippet(snippet));
    }

    /**
     * Add a predefined content snippet to the menu. HTML formatting is allowed
     * in the snippet text. Drop down menu is visible after the first snippet is
     * added. This method allows assigning a custom title to the added snippet.
     * 
     * @param title
     *            string to display in drop-down menu
     * @param snippet
     *            snippet text, can be HTML formatted
     * @since 8.16
     */
    public void addSnippet(String title, String snippet) {
        addSnippet(new Snippet(title, snippet));
    }

    /**
     * Add a {@link Snippet} object to the menu. Drop-down menu is visible after
     * the first snippet is added.
     * 
     * @param snippet
     *            a {@link Snippet} object
     * @since 8.16
     */
    public void addSnippet(Snippet snippet) {
        List<Snippet> snippets = getState().snippets;
        if (snippets == null) {
            snippets = getState().snippets = new ArrayList<>();
        }
        snippets.add(snippet);
    }

    /**
     * Add predefined content snippets to the menu. HTML formatting is allowed.
     * Drop down menu is visible after the first snippet is added. The titles of
     * snippets in the drop-down menu are automatically generated from the first
     * 20 characters of the snippet text.
     * 
     * @param snippets
     *            array of snippet body text, can be HTML formatted
     * @since 8.16
     */
    public void addSnippets(String[] snippets) {
        List<Snippet> oldSnippets = getState().snippets; 
        List<Snippet> newSnippets = new ArrayList<>();
        for (int i = 0; i < snippets.length; ++i) {
            newSnippets.add(new Snippet(snippets[i]));
        }
        if (oldSnippets != null) {
        	oldSnippets.addAll(newSnippets);
        } else {
        	getState().snippets = newSnippets;
        }
    }

    /**
     * Add predefined content snippets to the menu. HTML formatting is allowed
     * in the snippet text. Drop-down menu is visible after the first snippet is
     * added. Each snippet is assigned a title from the titles array, or has one
     * automatically generated from the snippet text if the corresponding value
     * in the titles array is null. The array sizes must match exactly.
     * 
     * @param titles
     *            array of strings to display as snippet titles. This array may
     *            include nulls, in which case the title is automatically
     *            generated from the first 20 characters in the snippet text.
     * @param snippets
     *            array of snippet text, can be HTML formatted
     * @since 8.16
     */
    public void addSnippets(String[] titles, String[] snippets) {
        assert (titles.length == snippets.length);
        if (titles.length != snippets.length) {
            throw new IllegalArgumentException(
                    "Mismatched snippet title and value arrays");
        }

        List<Snippet> oldSnippets = getState().snippets;
        List<Snippet> newSnippets = new ArrayList<>();
        for (int i = 0; i < snippets.length; ++i) {
            newSnippets.add(new Snippet(titles[i], snippets[i]));
        }
        if (oldSnippets != null) {
        	oldSnippets.addAll(newSnippets);
        } else {
        	getState().snippets = newSnippets;
        }
    }

    /**
     * Add {@link Snippet} objects to the menu. Drop-down menu is visible after
     * the first snippet has been added.
     * 
     * @param snippets
     *            a {@link Collection} of Snippet objects
     * @since 8.16
     */
    public void addSnippets(Collection<Snippet> snippets) {
        List<Snippet> oldSnippets = getState().snippets;
        if (oldSnippets == null) {
            getState().snippets = new ArrayList<Snippet>(snippets);
        } else {
            oldSnippets.addAll(snippets);
        }
    }

    /**
     * Remove all snippets - this also hides the snippets drop-down unless new
     * snippets are added.
     * 
     * @since 8.16
     */
    public void clearSnippets() {
        getState().snippets = null;
    }
}
