/*
 * Copyright (C) 2000-2023 Vaadin Ltd
 *
 * This program is available under Vaadin Commercial License and Service Terms.
 *
 * See <https://vaadin.com/commercial-license-and-service-terms> for the full
 * license.
 */
package com.vaadin.event;

import java.lang.reflect.Method;

import com.vaadin.ui.Component;
import com.vaadin.util.ReflectTools;

/**
 * An event that is fired when an item is collapsed in a listing component that
 * displays hierarchical data. Note that expanded subtrees of the collapsed item
 * will not trigger collapse events.
 *
 * @author Vaadin Ltd
 * @since 8.1
 * @param <T>
 *            collapsed item type
 */
public class CollapseEvent<T> extends Component.Event
        implements HasUserOriginated {

    private final T collapsedItem;

    private final boolean userOriginated;

    /**
     * Construct a collapse event.
     *
     * @param source
     *            the hierarchical component this event originated from
     * @param collapsedItem
     *            the item that was collapsed
     * @param userOriginated
     *            whether the collapse was triggered by a user interaction or
     *            the server
     */
    public CollapseEvent(Component source, T collapsedItem,
            boolean userOriginated) {
        super(source);
        this.collapsedItem = collapsedItem;
        this.userOriginated = userOriginated;
    }

    /**
     * Get the collapsed item that triggered this event.
     *
     * @return the collapsed item
     */
    public T getCollapsedItem() {
        return collapsedItem;
    }

    @Override
    public boolean isUserOriginated() {
        return userOriginated;
    }

    /**
     * Item collapse event listener.
     *
     * @param <T>
     *            the collapsed item's type
     * @since 8.1
     */
    @FunctionalInterface
    public interface CollapseListener<T> extends SerializableEventListener {

        public static final Method COLLAPSE_METHOD = ReflectTools.findMethod(
                CollapseListener.class, "itemCollapse", CollapseEvent.class);

        /**
         * Callback method for when an item has been collapsed.
         *
         * @param event
         *            the collapse event
         */
        public void itemCollapse(CollapseEvent<T> event);
    }
}
