/*
 * Copyright 2000-2026 Vaadin Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.vaadin.flow.data.renderer;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.UUID;
import java.util.regex.Pattern;

import com.vaadin.flow.component.UI;
import com.vaadin.flow.component.dependency.JsModule;
import com.vaadin.flow.data.provider.DataGenerator;
import com.vaadin.flow.data.provider.DataKeyMapper;
import com.vaadin.flow.dom.Element;
import com.vaadin.flow.function.SerializableBiConsumer;
import com.vaadin.flow.function.SerializableConsumer;
import com.vaadin.flow.function.ValueProvider;
import com.vaadin.flow.internal.JacksonSerializer;
import com.vaadin.flow.internal.JacksonUtils;
import com.vaadin.flow.internal.StateTree;
import com.vaadin.flow.internal.UsageStatistics;
import com.vaadin.flow.internal.nodefeature.ReturnChannelMap;
import com.vaadin.flow.internal.nodefeature.ReturnChannelRegistration;
import com.vaadin.flow.shared.Registration;

import tools.jackson.databind.node.ArrayNode;
import tools.jackson.databind.node.ObjectNode;

/**
 * LitRenderer is a {@link Renderer} that uses a Lit-based template literal to
 * render given model objects in the components that support the JS renderer
 * functions API. Mainly it's intended for use with {@code Grid},
 * {@code ComboBox} and {@code VirtualList}, but is not limited to these.
 *
 * @author Vaadin Ltd
 * @since 22.0.
 *
 * @param <SOURCE>
 *            the type of the model object used inside the template expression
 *
 * @see #of(String)
 * @see <a href=
 *      "https://lit.dev/docs/templates/overview/">https://lit.dev/docs/templates/overview/</a>
 * @see <a href=
 *      "https://cdn.vaadin.com/vaadin-web-components/20.0.0/#/elements/vaadin-combo-box"><code>&lt;vaadin-combo-box&gt;.renderer</code></a>
 */
@JsModule("./lit-renderer.ts")
public class LitRenderer<SOURCE> extends Renderer<SOURCE> {

    static {
        UsageStatistics.markAsUsed("flow-components/LitRenderer", null);
    }

    private final String templateExpression;

    private final String propertyNamespace;

    private final Map<String, ValueProvider<SOURCE, ?>> valueProviders = new HashMap<>();
    private final Map<String, SerializableBiConsumer<SOURCE, ArrayNode>> clientCallables = new HashMap<>();

    private final String ALPHANUMERIC_REGEX = "^[a-zA-Z0-9]+$";

    private LitRenderer(String templateExpression) {
        this.templateExpression = templateExpression;

        propertyNamespace = String.format("lr_%s_",
                UUID.randomUUID().toString().replace("-", "").substring(0, 16));
    }

    LitRenderer() {
        this("");
    }

    /**
     * Creates a new LitRenderer based on the provided template expression. The
     * expression accepts content that is allowed inside JS template literals,
     * and works with the Lit data binding syntax.
     * <p>
     * The template expression has access to:
     * <ul>
     * <li>{@code item} the model item being rendered</li>
     * <li>{@code index} the index of the current item (when rendering a
     * list)</li>
     * <li>{@code item.property} any property of the model item exposed via
     * {@link #withProperty(String, ValueProvider)}</li>
     * <li>any function exposed via
     * {@link #withFunction(String, SerializableConsumer)}</li>
     * </ul>
     * <p>
     * Examples:
     *
     * <pre>
     * {@code
     * // Prints the `name` property of a person
     * LitRenderer.<Person> of("<div>Name: ${item.name}</div>")
     *         .withProperty("name", Person::getName);
     *
     * // Prints the index of the item inside a repeating list
     * LitRenderer.of("${index}");
     * }
     * </pre>
     *
     * @param <SOURCE>
     *            the type of the input object used inside the template
     *
     * @param templateExpression
     *            the template expression used to render items, not
     *            <code>null</code>
     * @return an initialized LitRenderer
     * @see LitRenderer#withProperty(String, ValueProvider)
     * @see LitRenderer#withFunction(String, SerializableConsumer)
     */
    public static <SOURCE> LitRenderer<SOURCE> of(String templateExpression) {
        Objects.requireNonNull(templateExpression);
        return new LitRenderer<>(templateExpression);
    }

    @Override
    public Rendering<SOURCE> render(Element container,
            DataKeyMapper<SOURCE> keyMapper, String rendererName) {
        DataGenerator<SOURCE> dataGenerator = createDataGenerator();
        Registration registration = createJsRendererFunction(container,
                keyMapper, rendererName);

        return new Rendering<SOURCE>() {
            @Override
            public Optional<DataGenerator<SOURCE>> getDataGenerator() {
                return Optional.of(dataGenerator);
            }

            @Override
            public Registration getRegistration() {
                return registration;
            }
        };
    }

    private UI getElementUI(Element element) {
        return ((StateTree) element.getNode().getOwner()).getUI();
    }

    private void setElementRenderer(Element container, String rendererName,
            String templateExpression, ReturnChannelRegistration returnChannel,
            ArrayNode clientCallablesArray, String propertyNamespace) {
        assert container.getNode().isAttached() : "Container must be attached";

        String appId = getElementUI(container).getInternals().getAppId();

        container.executeJs(
                "window.Vaadin.setLitRenderer(this, $0, $1, $2, $3, $4, $5)",
                rendererName, templateExpression, returnChannel,
                clientCallablesArray, propertyNamespace, appId);
    }

    /**
     * Returns the Lit template expression used to render items.
     *
     * @return the template expression
     */
    protected String getTemplateExpression() {
        return templateExpression;
    }

    /**
     * Returns the namespace used to prefix property names when sending them to
     * the client as part of an item.
     *
     * @return the property namespace
     */
    String getPropertyNamespace() {
        return propertyNamespace;
    }

    private Registration createJsRendererFunction(Element container,
            DataKeyMapper<SOURCE> keyMapper, String rendererName) {
        ReturnChannelRegistration returnChannel = container.getNode()
                .getFeature(ReturnChannelMap.class)
                .registerChannel(arguments -> {
                    // Invoked when the client calls one of the client callables
                    String handlerName = arguments.get(0).asString();
                    String itemKey = arguments.get(1).asString();
                    ArrayNode args = (ArrayNode) arguments.get(2);

                    SerializableBiConsumer<SOURCE, ArrayNode> handler = clientCallables
                            .get(handlerName);
                    SOURCE item = keyMapper.get(itemKey);
                    if (item != null) {
                        handler.accept(item, args);
                    }
                });

        ArrayNode clientCallablesArray = JacksonUtils
                .listToJson(new ArrayList<>(clientCallables.keySet()));

        List<Registration> registrations = new ArrayList<>();

        // Since the renderer is set manually on the client-side, an attach
        // listener for the host component is required so that the renderer gets
        // applied even when the host component gets a new Web Component
        // instance (for example on detach + reattach).
        //
        // The attach listener needs to be released when the Renderer instance
        // is no longer used so the registration is cleared by the renderer
        // registration.
        registrations.add(container.addAttachListener(e -> {
            setElementRenderer(container, rendererName, getTemplateExpression(),
                    returnChannel, clientCallablesArray, propertyNamespace);
        }));
        // Call once initially
        if (container.getNode().isAttached()) {
            setElementRenderer(container, rendererName, getTemplateExpression(),
                    returnChannel, clientCallablesArray, propertyNamespace);
        }

        // Get the renderer function cleared when the LitRenderer is
        // unregistered
        registrations.add(() -> container.executeJs(
                "window.Vaadin.unsetLitRenderer(this, $0, $1)", rendererName,
                propertyNamespace));

        return () -> registrations.forEach(Registration::remove);
    }

    private DataGenerator<SOURCE> createDataGenerator() {
        // Use an anonymous class instead of Lambda to prevent potential
        // deserialization issues when used with Grid
        // see https://github.com/vaadin/flow-components/issues/6256
        return new DataGenerator<SOURCE>() {
            @Override
            public void generateData(SOURCE item, ObjectNode jsonObject) {
                valueProviders.forEach((key, provider) -> {
                    jsonObject.set(
                            // Prefix the property name with a LitRenderer
                            // instance specific namespace to avoid property
                            // name clashes.
                            // Fixes https://github.com/vaadin/flow/issues/8629
                            // in LitRenderer
                            propertyNamespace + key,
                            JacksonSerializer.toJson(provider.apply(item)));
                });
            }
        };
    }

    /**
     * Makes a property available to the template expression. Each property is
     * referenced inside the template by using the {@code ${item.property}}
     * syntax.
     * <p>
     * Examples:
     *
     * <pre>
     * {@code
     * // Regular property
     * LitRenderer.<Person> of("<div>Name: ${item.name}</div>")
     *         .withProperty("name", Person::getName);
     *
     * // Property that uses a bean. Note that in this case the entire "Address"
     * // object will be sent to the template.
     * // Note that even properties of the bean which are not used in the
     * // template are sent to the client, so use
     * // this feature with caution.
     * LitRenderer.<Person> of("<span>Street: ${item.address.street}</span>")
     *         .withProperty("address", Person::getAddress);
     *
     * // In this case only the street field inside the Address object is sent
     * LitRenderer.<Person> of("<span>Street: ${item.street}</span>")
     *         .withProperty("street",
     *                 person -> person.getAddress().getStreet());
     * }
     * </pre>
     *
     * Any Jackson types supported are valid for LitRenderer.
     *
     * @param property
     *            the name of the property used inside the template expression,
     *            not <code>null</code>
     *
     * @param provider
     *            a {@link ValueProvider} that provides the actual value for the
     *            property, not <code>null</code>
     * @return this instance for method chaining
     */
    public LitRenderer<SOURCE> withProperty(String property,
            ValueProvider<SOURCE, ?> provider) {
        Objects.requireNonNull(property);
        Objects.requireNonNull(provider);
        valueProviders.put(property, provider);
        return this;
    }

    /**
     * Adds a function that can be called from within the template expression.
     * <p>
     * Examples:
     *
     * <pre>
     * {@code
     * // Standard event
     * LitRenderer.of("<button @click=${handleClick}>Click me</button>")
     *         .withFunction("handleClick", object -> doSomething());
     * }
     * </pre>
     *
     * The name of the function used in the template expression should be the
     * name used at the functionName parameter. This name must be a valid
     * JavaScript function name.
     *
     * @param functionName
     *            the name of the function used inside the template expression,
     *            must be alphanumeric and not <code>null</code>, must not be
     *            one of the JavaScript reserved words
     *            (https://www.w3schools.com/js/js_reserved.asp)
     * @param handler
     *            the handler executed when the function is called, not
     *            <code>null</code>
     * @return this instance for method chaining
     * @see <a href=
     *      "https://lit.dev/docs/templates/expressions/#event-listener-expressions">https://lit.dev/docs/templates/expressions/#event-listener-expressions</a>
     */
    public LitRenderer<SOURCE> withFunction(String functionName,
            SerializableConsumer<SOURCE> handler) {
        return withFunction(functionName,
                (item, ignore) -> handler.accept(item));
    }

    /**
     * Adds a function that can be called from within the template expression.
     * The function accepts arguments that can be consumed by the given handler.
     *
     * <p>
     * Examples:
     *
     * <pre>
     * {@code
     * // Standard event
     * LitRenderer.of("<button @click=${handleClick}>Click me</button>")
     *         .withFunction("handleClick", item -> doSomething());
     *
     * // Function invocation with arguments
     * LitRenderer.of("<input @keypress=${(e) => handleKeyPress(e.key)}>")
     *         .withFunction("handleKeyPress", (item, args) -> {
     *             System.out.println("Pressed key: " + args.getString(0));
     *         });
     * }
     * </pre>
     *
     * The name of the function used in the template expression should be the
     * name used at the functionName parameter. This name must be a valid
     * Javascript function name.
     *
     * @param functionName
     *            the name of the function used inside the template expression,
     *            must be alphanumeric and not <code>null</code>, must not be
     *            one of the JavaScript reserved words
     *            (https://www.w3schools.com/js/js_reserved.asp)
     * @param handler
     *            the handler executed when the function is called, not
     *            <code>null</code>
     * @return this instance for method chaining
     * @see <a href=
     *      "https://lit.dev/docs/templates/expressions/#event-listener-expressions">https://lit.dev/docs/templates/expressions/#event-listener-expressions</a>
     */
    public LitRenderer<SOURCE> withFunction(String functionName,
            SerializableBiConsumer<SOURCE, ArrayNode> handler) {
        Objects.requireNonNull(functionName);
        Objects.requireNonNull(handler);

        if (!Pattern.matches(ALPHANUMERIC_REGEX, functionName)) {
            throw new IllegalArgumentException(
                    "Function name must be alphanumeric");
        }
        clientCallables.put(functionName, handler);
        return this;
    }

    /**
     * Gets the property mapped to {@link ValueProvider}s in this renderer. The
     * returned map is immutable.
     *
     * @return the mapped properties, never <code>null</code>
     */
    public Map<String, ValueProvider<SOURCE, ?>> getValueProviders() {
        return Collections.unmodifiableMap(valueProviders);
    }
}
