/*
 * Copyright (C) 2000-2024 Vaadin Ltd
 *
 * This program is available under Vaadin Commercial License and Service Terms.
 *
 * See <https://vaadin.com/commercial-license-and-service-terms> for the full
 * license.
 */
package com.vaadin.client.connectors.grid;

import com.google.gwt.safehtml.shared.SafeHtmlUtils;
import com.google.gwt.user.client.Element;
import com.vaadin.client.annotations.OnStateChange;
import com.vaadin.client.renderers.HtmlRenderer;
import com.vaadin.client.renderers.Renderer;
import com.vaadin.client.widget.grid.RendererCellReference;
import com.vaadin.shared.ui.Connect;
import com.vaadin.shared.ui.tree.TreeRendererState;
import com.vaadin.ui.Tree.TreeRenderer;

import elemental.json.JsonObject;

/**
 * Connector for TreeRenderer.
 *
 * @author Vaadin Ltd
 * @since 8.1
 */
@SuppressWarnings("deprecation")
@Connect(TreeRenderer.class)
public class TreeRendererConnector
        extends AbstractGridRendererConnector<String> {

    @Override
    public Renderer<String> createRenderer() {
        return new HtmlRenderer() {

            @Override
            public void render(RendererCellReference cell, String htmlString) {
                String content = "<span class=\"v-captiontext\">"
                        + getContentString(htmlString) + "</span>";

                JsonObject row = getParent().getParent().getDataSource()
                        .getRow(cell.getRowIndex());
                if (row != null && row.hasKey("itemIcon")) {
                    String resourceId = row.getString("itemIcon");
                    Element element = getConnection()
                            .getIcon(getResourceUrl(resourceId)).getElement();
                    content = element.getString() + content;
                }
                super.render(cell, content);
            }

            private String getContentString(String htmlString) {
                switch (getState().mode) {
                case HTML:
                    return htmlString;
                case PREFORMATTED:
                    return "<pre>" + SafeHtmlUtils.htmlEscape(htmlString)
                            + "</pre>";
                default:
                    return SafeHtmlUtils.htmlEscape(htmlString);
                }
            }
        };
    }

    @OnStateChange("mode")
    void updateContentMode() {
        // Redraw content
        getParent().getParent().getWidget().requestRefreshBody();

        // Some pre-formatted content might change size of content.
        getParent().getParent().getWidget().recalculateColumnWidths();
    }

    @Override
    public ColumnConnector getParent() {
        return (ColumnConnector) super.getParent();
    }

    @Override
    public TreeRendererState getState() {
        return (TreeRendererState) super.getState();
    }
}
