/*
 * Copyright (C) 2000-2024 Vaadin Ltd
 *
 * This program is available under Vaadin Commercial License and Service Terms.
 *
 * See <https://vaadin.com/commercial-license-and-service-terms> for the full
 * license.
 */
/**
 * This plugin handles adding/deleting theme resources events and triggers
 * theme meta data re-generation and application theme update on the fly.
 */
class ThemeLiveReloadPlugin {
  /**
   * Create a new instance of ThemeLiveReloadPlugin
   * @param processThemeResourcesCallback callback which is called on
   * adding/deleting of theme resource files to re-generate theme meta
   * data and apply theme changes to application.
   */
  constructor(processThemeResourcesCallback) {
    if (!processThemeResourcesCallback || typeof processThemeResourcesCallback !== 'function') {
      throw new Error(
        "Couldn't instantiate a ThemeLiveReloadPlugin" +
          ' instance, because theme resources process callback is not set' +
          ' and, thus, no information provided what to do upon' +
          ' adding/deleting theme resource files. Please provide this' +
          ' callback as a ThemeLiveReloadPlugin constructor parameter.'
      );
    }
    this.processThemeResourcesCallback = processThemeResourcesCallback;
    // Component style sheet might be deleted from parent theme folder, so
    // the regexp does not contain the exact theme name
    this.componentStyleFileRegexp = /(\\|\/)themes\1([\s\S]*)\1components\1(.*)\.css$/;
    // There might be several theme generated files in the generated
    // folder, so the regexp does not contain the exact theme name
    this.themeGeneratedFileRegexp = /theme-[\s\S]*?\.generated\.js$/;
  }

  apply(compiler) {
    // Adds a hook for theme files change event
    compiler.hooks.watchRun.tapAsync('ThemeLiveReloadPlugin', (compilation, callback) => {
      const logger = compiler.getInfrastructureLogger('ThemeLiveReloadPlugin');
      const changedFilesPaths = compiler.modifiedFiles;
      const removedFilesPaths = compiler.removedFiles;
      if (changedFilesPaths && changedFilesPaths.length > 0) {
        let themeName = undefined;
        let themeGeneratedFileChanged = false;
        let themeGeneratedFileDeleted = false;
        let deletedComponentStyleFile = undefined;

        logger.debug('Detected changes in the following files ' + changedFilesPaths);
        changedFilesPaths.forEach((changedFilePath) => {
          const themeGeneratedFileChangedNow = changedFilePath.match(this.themeGeneratedFileRegexp);

          if (themeGeneratedFileChangedNow) {
            themeGeneratedFileChanged = true;
          }
        });

        if (removedFilesPaths && removedFilesPaths.length > 0) {
          removedFilesPaths.forEach(removedFilePath => {
            const themeGeneratedFileChangedNow = removedFilePath.match(this.themeGeneratedFileRegexp);
            if (themeGeneratedFileChangedNow) {
              themeGeneratedFileDeleted = true;
            } else {
              const matchResult = removedFilePath.match(this.componentStyleFileRegexp);
              if (matchResult) {
                themeName = matchResult[2];
                deletedComponentStyleFile = file;
              }
            }
          });
        }

        // This is considered as a workaround for
        // https://github.com/vaadin/flow/issues/9948: delete component
        // styles and theme generated file in one run to not have webpack
        // compile error
        if (deletedComponentStyleFile && !themeGeneratedFileDeleted) {
          logger.warn(
            "Custom theme component style sheet '" +
              deletedComponentStyleFile +
              "' has been deleted.\n\n" +
              "You should also delete './frontend/generated/theme-" +
              themeName +
              ".generated.js' (simultaneously) with the component stylesheet'.\n" +
              "Otherwise it will cause a webpack compilation error 'no such file or directory', as component style sheets are referenced from " +
              "'./frontend/generated/theme-" +
              themeName +
              ".generated.js'.\n\n" +
              "If you encounter a 'no such file or directory' error in your application, just click on the overlay (or refresh the browser page), and it should disappear.\n\n" +
              'It should then be possible to continue working on the application and theming.\n' +
              "If it doesn't help, you need to restart the application."
          );
        }

        // Webpack watches to the changes in theme-[my-theme].generated.js
        // because it is referenced from theme.js. Changes in this file
        // should not trigger the theme handling callback (which
        // re-generates theme-[my-theme].generated.js),
        // otherwise it will get into infinite re-compilation loop.
        if (themeGeneratedFileDeleted || !themeGeneratedFileChanged) {
          this.processThemeResourcesCallback(logger);
        }
      }
      callback();
    });
  }
}

module.exports = ThemeLiveReloadPlugin;
