/*
 * Decompiled with CFR 0.152.
 */
package com.vaadin.flow.server;

import com.vaadin.flow.function.DeploymentConfiguration;
import com.vaadin.flow.internal.ResponseWriter;
import com.vaadin.flow.server.HandlerHelper;
import com.vaadin.flow.server.StaticFileHandler;
import com.vaadin.flow.server.VaadinServletService;
import com.vaadin.flow.server.osgi.OSGiAccess;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.net.URLConnection;
import java.nio.file.FileSystem;
import java.nio.file.FileSystemAlreadyExistsException;
import java.nio.file.FileSystems;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.regex.Pattern;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class StaticFileServer
implements StaticFileHandler {
    static final String PROPERTY_FIX_INCORRECT_WEBJAR_PATHS = "vaadin.fixIncorrectWebjarPaths";
    private static final Pattern INCORRECT_WEBJAR_PATH_REGEX = Pattern.compile("^/frontend[-\\w/]*/webjars/");
    private final ResponseWriter responseWriter;
    private final VaadinServletService servletService;
    private DeploymentConfiguration deploymentConfiguration;
    protected static final Pattern APP_THEME_PATTERN = Pattern.compile("^\\/themes\\/[\\s\\S]+?\\/");
    static final Map<URI, Integer> openFileSystems = new HashMap<URI, Integer>();
    private static final Object fileSystemLock = new Object();

    public StaticFileServer(VaadinServletService servletService) {
        this.servletService = servletService;
        this.deploymentConfiguration = servletService.getDeploymentConfiguration();
        this.responseWriter = new ResponseWriter(this.deploymentConfiguration);
    }

    @Override
    public boolean isStaticResourceRequest(HttpServletRequest request) {
        String requestFilename = this.getRequestFilename(request);
        if (requestFilename.endsWith("/")) {
            return false;
        }
        if (APP_THEME_PATTERN.matcher(requestFilename).find() || requestFilename.startsWith("/VAADIN/static/") || requestFilename.startsWith("/VAADIN/build/")) {
            return true;
        }
        URL resource = this.getStaticResource(requestFilename);
        if (resource != null && this.resourceIsDirectory(resource)) {
            return false;
        }
        if (resource == null && this.shouldFixIncorrectWebjarPaths() && this.isIncorrectWebjarPath(requestFilename)) {
            return true;
        }
        return resource != null;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean resourceIsDirectory(URL resource) {
        if (resource.getPath().endsWith("/")) {
            return true;
        }
        URI resourceURI = null;
        try {
            resourceURI = resource.toURI();
        }
        catch (URISyntaxException e) {
            StaticFileServer.getLogger().debug("Syntax error in uri from getStaticResource", (Throwable)e);
            return false;
        }
        if ("jar".equals(resource.getProtocol())) {
            String pathInJar = resource.getPath().substring(resource.getPath().indexOf(33) + 1);
            try {
                FileSystem fileSystem = this.getFileSystem(resourceURI);
                Path path = fileSystem.getPath(pathInJar, new String[0]);
                boolean bl = Files.isDirectory(path, new LinkOption[0]);
                return bl;
            }
            catch (IOException e) {
                StaticFileServer.getLogger().debug("failed to read jar file contents", (Throwable)e);
            }
            finally {
                this.closeFileSystem(resourceURI);
            }
        }
        return "file".equals(resource.getProtocol()) && Files.isDirectory(Paths.get(resourceURI), new LinkOption[0]);
    }

    private URI getFileURI(URI resourceURI) {
        if (!"jar".equals(resourceURI.getScheme())) {
            return resourceURI;
        }
        try {
            String scheme = resourceURI.getRawSchemeSpecificPart();
            int jarPartIndex = scheme.indexOf("!/");
            if (jarPartIndex != -1) {
                scheme = scheme.substring(0, jarPartIndex);
            }
            return new URI(scheme);
        }
        catch (URISyntaxException syntaxException) {
            throw new IllegalArgumentException(syntaxException.getMessage(), syntaxException);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    FileSystem getFileSystem(URI resourceURI) throws IOException {
        Object object = fileSystemLock;
        synchronized (object) {
            URI fileURI = this.getFileURI(resourceURI);
            if (!fileURI.getScheme().equals("file")) {
                throw new IOException("Can not read scheme '" + fileURI.getScheme() + "' for resource " + resourceURI + " and will determine this as not a folder");
            }
            if (openFileSystems.computeIfPresent(fileURI, (key, value) -> value + 1) != null) {
                return FileSystems.getFileSystem(resourceURI);
            }
            FileSystem fileSystem = this.getNewOrExistingFileSystem(resourceURI);
            openFileSystems.put(fileURI, 1);
            return fileSystem;
        }
    }

    private FileSystem getNewOrExistingFileSystem(URI resourceURI) throws IOException {
        try {
            return FileSystems.newFileSystem(resourceURI, Collections.emptyMap());
        }
        catch (FileSystemAlreadyExistsException fsaee) {
            StaticFileServer.getLogger().trace("Tried to get new filesystem, but it already existed for target uri.", (Throwable)fsaee);
            return FileSystems.getFileSystem(resourceURI);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void closeFileSystem(URI resourceURI) {
        Object object = fileSystemLock;
        synchronized (object) {
            try {
                URI fileURI = this.getFileURI(resourceURI);
                Integer locks = openFileSystems.computeIfPresent(fileURI, (key, value) -> value - 1);
                if (locks != null && locks == 0) {
                    openFileSystems.remove(fileURI);
                    FileSystems.getFileSystem(resourceURI).close();
                }
            }
            catch (IOException ioe) {
                StaticFileServer.getLogger().error("Failed to close FileSystem for '{}'", (Object)resourceURI);
                StaticFileServer.getLogger().debug("Exception closing FileSystem", (Throwable)ioe);
            }
        }
    }

    @Override
    public boolean serveStaticResource(HttpServletRequest request, HttpServletResponse response) throws IOException {
        String filenameWithPath = this.getRequestFilename(request);
        if (HandlerHelper.isPathUnsafe(filenameWithPath)) {
            StaticFileServer.getLogger().info("Blocked attempt to access file: {}", (Object)filenameWithPath);
            response.setStatus(403);
            return true;
        }
        URL resourceUrl = null;
        if (this.isAllowedVAADINBuildOrStaticUrl(filenameWithPath)) {
            resourceUrl = APP_THEME_PATTERN.matcher(filenameWithPath).find() ? this.servletService.getClassLoader().getResource("META-INF/VAADIN/static" + filenameWithPath) : this.servletService.getClassLoader().getResource("META-INF" + filenameWithPath);
        }
        if (resourceUrl == null) {
            resourceUrl = this.getStaticResource(filenameWithPath);
        }
        if (resourceUrl == null && this.shouldFixIncorrectWebjarPaths() && this.isIncorrectWebjarPath(filenameWithPath)) {
            resourceUrl = this.getStaticResource(this.fixIncorrectWebjarPath(filenameWithPath));
        }
        if (resourceUrl == null) {
            response.sendError(404);
            return true;
        }
        this.writeCacheHeaders(filenameWithPath, response);
        long timestamp = this.writeModificationTimestamp(resourceUrl, request, response);
        if (this.browserHasNewestVersion(request, timestamp)) {
            response.setStatus(304);
            return true;
        }
        this.responseWriter.writeResponseContents(filenameWithPath, resourceUrl, request, response);
        return true;
    }

    protected URL getStaticResource(String path) {
        if (OSGiAccess.getInstance().getOsgiServletContext() == null) {
            return this.servletService.getStaticResource(path);
        }
        return null;
    }

    private boolean shouldFixIncorrectWebjarPaths() {
        return this.deploymentConfiguration.isProductionMode() && this.deploymentConfiguration.getBooleanProperty(PROPERTY_FIX_INCORRECT_WEBJAR_PATHS, false);
    }

    private boolean isIncorrectWebjarPath(String requestFilename) {
        return INCORRECT_WEBJAR_PATH_REGEX.matcher(requestFilename).lookingAt();
    }

    private String fixIncorrectWebjarPath(String requestFilename) {
        return INCORRECT_WEBJAR_PATH_REGEX.matcher(requestFilename).replaceAll("/webjars/");
    }

    private boolean isAllowedVAADINBuildOrStaticUrl(String filenameWithPath) {
        if (this.deploymentConfiguration.isCompatibilityMode()) {
            StaticFileServer.getLogger().trace("Serving from the classpath in legacy mode is not accepted. Letting request for '{}' go to servlet context.", (Object)filenameWithPath);
            return false;
        }
        return filenameWithPath.startsWith("/VAADIN/build/") || filenameWithPath.startsWith("/VAADIN/static/") || APP_THEME_PATTERN.matcher(filenameWithPath).find();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected long writeModificationTimestamp(URL resourceUrl, HttpServletRequest request, HttpServletResponse response) {
        URLConnection connection = null;
        try {
            connection = resourceUrl.openConnection();
            long lastModifiedTime = connection.getLastModified();
            lastModifiedTime -= lastModifiedTime % 1000L;
            response.setDateHeader("Last-Modified", lastModifiedTime);
            long l = lastModifiedTime;
            return l;
        }
        catch (Exception e) {
            StaticFileServer.getLogger().trace("Failed to find out last modified timestamp. Continuing without it.", (Throwable)e);
        }
        finally {
            try {
                InputStream is;
                if (connection != null && (is = connection.getInputStream()) != null) {
                    is.close();
                }
            }
            catch (IOException e) {
                StaticFileServer.getLogger().warn("Error closing URLConnection input stream", (Throwable)e);
            }
        }
        return -1L;
    }

    protected void writeCacheHeaders(String filenameWithPath, HttpServletResponse response) {
        int resourceCacheTime = this.getCacheTime(filenameWithPath);
        String cacheControl = !this.deploymentConfiguration.isProductionMode() ? "no-cache" : (resourceCacheTime > 0 ? "max-age=" + resourceCacheTime : "public, max-age=0, must-revalidate");
        response.setHeader("Cache-Control", cacheControl);
    }

    String getRequestFilename(HttpServletRequest request) {
        if (request.getPathInfo() == null) {
            return request.getServletPath();
        }
        if (request.getPathInfo().startsWith("/VAADIN/") || APP_THEME_PATTERN.matcher(request.getPathInfo()).find()) {
            return request.getPathInfo();
        }
        return request.getServletPath() + request.getPathInfo();
    }

    protected int getCacheTime(String filenameWithPath) {
        if (filenameWithPath.contains(".nocache.")) {
            return 0;
        }
        if (filenameWithPath.contains(".cache.")) {
            return 31536000;
        }
        return 3600;
    }

    protected boolean browserHasNewestVersion(HttpServletRequest request, long resourceLastModifiedTimestamp) {
        assert (resourceLastModifiedTimestamp >= -1L);
        if (resourceLastModifiedTimestamp == -1L) {
            return false;
        }
        try {
            long headerIfModifiedSince = request.getDateHeader("If-Modified-Since");
            if (headerIfModifiedSince >= resourceLastModifiedTimestamp) {
                return true;
            }
        }
        catch (Exception e) {
            StaticFileServer.getLogger().trace("Unable to parse If-Modified-Since", (Throwable)e);
        }
        return false;
    }

    private static Logger getLogger() {
        return LoggerFactory.getLogger((String)StaticFileServer.class.getName());
    }
}

