package com.vaadin.copilot.plugins.propertypanel;

import java.math.BigDecimal;
import java.math.BigInteger;

import com.vaadin.copilot.ComponentPropertyType;
import com.vaadin.copilot.JavaReflectionUtil;

import tools.jackson.databind.JsonNode;
import tools.jackson.databind.node.ArrayNode;

/**
 * Helper util for properties panel to handle parsing and obtaining data from
 * JSON
 */
public class ComponentPropertyHelperUtil {
    private static final String VALUE_JSON_KEY = "value";
    private static final String QUALIFIED_NAME_JSON_KEY = "qualifiedClassName";
    private static final String JAVA_TYPE_JSON_KEY = "javaType";

    private ComponentPropertyHelperUtil() {

    }

    /**
     * Converts object to {@link ComponentPropertyType}
     *
     * @param paramType
     *            Field parameter
     * @return Property field info. Returns null if param is null or type is not
     *         known
     */
    public static PropertyFieldInfo getPropertyType(Class<?> paramType) {
        if (paramType == null) {
            return null;
        }
        if (paramType.isArray()) {
            PropertyFieldInfo propertyType = getPropertyType(paramType.getComponentType());
            if (propertyType != null) {
                return new PropertyFieldInfo(propertyType.type, true, paramType.getComponentType().getName());
            }
        }
        if (paramType.equals(String.class)) {
            return new PropertyFieldInfo(ComponentPropertyType.STRING, paramType.getName());
        } else if (paramType.equals(Integer.class) || paramType.equals(int.class)) {
            return new PropertyFieldInfo(ComponentPropertyType.INTEGER, paramType.getName());
        } else if (paramType.equals(Double.class) || paramType.equals(double.class)) {
            return new PropertyFieldInfo(ComponentPropertyType.DOUBLE, paramType.getName());
        } else if (paramType.equals(Boolean.class) || paramType.equals(boolean.class)) {
            return new PropertyFieldInfo(ComponentPropertyType.BOOLEAN, paramType.getName());
        } else if (paramType.equals(Long.class) || paramType.equals(long.class)) {
            return new PropertyFieldInfo(ComponentPropertyType.LONG, paramType.getName());
        } else if (paramType.equals(Float.class) || paramType.equals(float.class)) {
            return new PropertyFieldInfo(ComponentPropertyType.FLOAT, paramType.getName());
        } else if (paramType.equals(Short.class) || paramType.equals(short.class)) {
            return new PropertyFieldInfo(ComponentPropertyType.SHORT, paramType.getName());
        } else if (paramType.equals(Byte.class) || paramType.equals(byte.class)) {
            return new PropertyFieldInfo(ComponentPropertyType.BYTE, paramType.getName());
        } else if (paramType.equals(Character.class) || paramType.equals(char.class)) {
            return new PropertyFieldInfo(ComponentPropertyType.CHARACTER, paramType.getName());
        } else if (paramType.equals(BigDecimal.class)) {
            return new PropertyFieldInfo(ComponentPropertyType.BIG_DECIMAL, paramType.getName());
        } else if (paramType.equals(BigInteger.class)) {
            return new PropertyFieldInfo(ComponentPropertyType.BIG_INTEGER, paramType.getName());
        } else if (paramType.isEnum()) {
            return new PropertyFieldInfo(ComponentPropertyType.ENUM, paramType.getName());
        }
        return null;
    }

    /**
     * Parses JSON object to obtain Java-typed value. Compares JSON field type with
     * {@link ComponentPropertyType} and returns the value within that type.
     *
     * @param data
     *            data is sent from Properties Panel
     * @return Object that represents the value. e.g. If json has
     *         {@link ComponentPropertyType#LONG} then Object type is Long
     * @throws IllegalArgumentException
     *             is thrown if JSON object is null or type is unknown.
     */
    public static Object extractPropertyValueFromJson(JsonNode data) {
        if (data == null) {
            throw new IllegalArgumentException("JSON object cannot be null");
        }
        JsonNode valueJson = data.get(VALUE_JSON_KEY);
        var qualifiedClassName = data.has(QUALIFIED_NAME_JSON_KEY) ? data.get(QUALIFIED_NAME_JSON_KEY).asString()
                : null;
        var javaTypeStr = data.hasNonNull(JAVA_TYPE_JSON_KEY) ? data.get(JAVA_TYPE_JSON_KEY).asString() : null;
        ComponentPropertyType javaType = ComponentPropertyType.find(javaTypeStr);
        if (valueJson.isArray()) {
            var array = (ArrayNode) valueJson;
            int length = array.size();
            Object[] values = new Object[length];
            for (int i = 0; i < length; i++) {
                values[i] = parseValueExpression(array.get(i), javaType, qualifiedClassName);
            }
            return values;
        }
        return parseValueExpression(valueJson, javaType, qualifiedClassName);
    }

    private static Object parseValueExpression(JsonNode valueJson, ComponentPropertyType javaType,
            String qualifiedClassName) {
        if (valueJson.isTextual() && ComponentPropertyType.STRING == javaType) {
            return valueJson.asString();
        }
        if (valueJson.isTextual() && ComponentPropertyType.CHARACTER == javaType) {
            String strValue = valueJson.asString();
            return strValue.length() == 1 ? strValue.charAt(0) : null;
        }
        if (valueJson.isNull()) {
            return null;
        }

        if (valueJson.isBoolean()) {
            return valueJson.asBoolean();
        }
        if (valueJson.isNumber()) {
            return parseNumberExpression(valueJson, javaType);
        }

        if (ComponentPropertyType.ENUM == javaType && qualifiedClassName != null) {
            Class<?> enumClass = JavaReflectionUtil.getClass(qualifiedClassName);
            return JavaReflectionUtil.getEnumConstant(enumClass, valueJson.asString());
        }
        throw new IllegalArgumentException("Unsupported type " + valueJson.getNodeType());
    }

    private static Object parseNumberExpression(JsonNode valueJson, ComponentPropertyType javaType) {
        if (ComponentPropertyType.INTEGER == javaType) {
            return valueJson.asInt();
        } else if (ComponentPropertyType.DOUBLE == javaType) {
            return valueJson.asDouble();
        } else if (ComponentPropertyType.FLOAT == javaType) {
            return (float) valueJson.asDouble();
        } else if (ComponentPropertyType.SHORT == javaType) {
            return (short) valueJson.asInt();
        } else if (ComponentPropertyType.LONG == javaType) {
            return valueJson.asLong();
        } else if (ComponentPropertyType.BYTE == javaType) {
            return (byte) valueJson.asInt();
        } else if (ComponentPropertyType.BIG_DECIMAL == javaType) {
            return BigDecimal.valueOf(valueJson.asDouble());
        } else if (ComponentPropertyType.BIG_INTEGER == javaType) {
            return BigInteger.valueOf(valueJson.asLong());
        }
        return valueJson.asDouble();
    }

    /**
     * Holds metadata about a component property field, including its type, whether
     * it supports multiple selection, and the fully qualified name of its
     * associated class.
     * <p>
     * This record encapsulates the characteristics of a property used in a
     * component, typically for serialization or tooling purposes.
     *
     * @param type
     *            the {@link ComponentPropertyType} of the property (e.g., STRING,
     *            BOOLEAN)
     * @param multipleSelection
     *            {@code true} if the property allows multiple selections;
     *            {@code false} otherwise
     * @param qualifiedClassName
     *            the fully qualified name of the Java class representing the
     *            property's type, or {@code null} if not specified
     */
    public record PropertyFieldInfo(ComponentPropertyType type, boolean multipleSelection, String qualifiedClassName) {

        /**
         * Constructs a {@code PropertyFieldInfo} with the specified type, assuming no
         * multiple selection and no associated class name.
         *
         * @param type
         *            the {@link ComponentPropertyType} of the property
         */

        public PropertyFieldInfo(ComponentPropertyType type) {
            this(type, false, null);
        }

        /**
         * Constructs a {@code PropertyFieldInfo} with the specified type and qualified
         * class name, assuming no multiple selection.
         *
         * @param type
         *            the {@link ComponentPropertyType} of the property
         * @param qualifiedClassName
         *            the fully qualified name of the Java class for this property
         */
        public PropertyFieldInfo(ComponentPropertyType type, String qualifiedClassName) {
            this(type, false, qualifiedClassName);
        }
    }
}
