package com.vaadin.copilot.javarewriter;

import java.io.File;
import java.io.IOException;
import java.lang.annotation.Annotation;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.util.Collections;
import java.util.Map;

import com.vaadin.copilot.Copilot;
import com.vaadin.copilot.ProjectFileManager;

import com.github.javaparser.ParserConfiguration;
import com.github.javaparser.StaticJavaParser;
import com.github.javaparser.ast.CompilationUnit;
import com.github.javaparser.ast.body.ClassOrInterfaceDeclaration;
import com.github.javaparser.ast.body.TypeDeclaration;
import com.github.javaparser.ast.expr.StringLiteralExpr;
import com.github.javaparser.ast.nodeTypes.modifiers.NodeWithPublicModifier;

/** A utility class for modifying Java source code. */
public class JavaModifier {

    /**
     * Parses the given source code string into a CompilationUnit. This method is
     * configured to understand and parse source code according to the specified
     * language level of Java 17. It prepares the parsed source code for further
     * manipulation or analysis while preserving its original formatting.
     *
     * @param source
     *            The source code of a Java file as a String.
     * @return A CompilationUnit object representing the parsed structure of the
     *         source code, enabling further analysis or modifications to be made
     *         programmatically.
     */
    private CompilationUnit parseSource(String source) {
        ParserConfiguration parserConfiguration = new ParserConfiguration();
        parserConfiguration.setLanguageLevel(Copilot.LANGUAGE_LEVEL);
        StaticJavaParser.setConfiguration(parserConfiguration);
        return StaticJavaParser.parse(source);
    }

    /**
     * A functional interface for operations that modify a CompilationUnit.
     *
     * @param <T>
     *            The type of the object that the operation modifies.
     */
    @FunctionalInterface
    public interface OperationFunction<T> {
        /**
         * Performs the operation on the given object.
         *
         * @param t
         *            The object to modify.
         * @throws IOException
         *             If an error occurs during the operation.
         */
        void accept(T t) throws IOException;
    }

    /**
     * Modifies the source code of the given class using the provided operations.
     *
     * @param cls
     *            The class to modify.
     * @param operations
     *            The operations to perform on the class.
     * @return A map of modified files and their new content.
     * @throws IOException
     *             If an error occurs
     */
    public Map<File, String> modify(Class<?> cls, OperationFunction<CompilationUnitOperations> operations)
            throws IOException {
        File javaFile = ProjectFileManager.get().getFileForClass(cls);

        String source = Files.readString(javaFile.toPath(), StandardCharsets.UTF_8);
        CompilationUnit cu = parseSource(source);

        operations.accept(new CompilationUnitOperations(cu, cls));

        String result = cu.toString();
        return Collections.singletonMap(javaFile, result);
    }

    /** A class for performing operations on a CompilationUnit. */
    public static class CompilationUnitOperations {

        private final CompilationUnit cu;
        private final Class<?> cls;

        /**
         * Creates a new operations object for the given CompilationUnit and class.
         *
         * @param cu
         *            The CompilationUnit to operate on.
         * @param cls
         *            The class that the CompilationUnit represents.
         */
        public CompilationUnitOperations(CompilationUnit cu, Class<?> cls) {
            this.cu = cu;
            this.cls = cls;
        }

        /**
         * Adds a specified annotation with a value to the public type in the class
         * represented by the provided Class object. This is primarily used to
         * dynamically modify the source code of a class by adding annotations that may
         * influence its behavior at runtime or during further processing.
         *
         * @param annotation
         *            The annotation class to add to the class.
         * @param value
         *            The value to assign to the annotation.
         * @throws IOException
         *             If an error occurs during file operations or if the class does
         *             not contain an application type where the annotation can be
         *             added.
         */
        public void addClassAnnotation(Class<? extends Annotation> annotation, String value) throws IOException {
            TypeDeclaration<?> type = getPublicType(cu);
            type.addSingleMemberAnnotation(annotation, new StringLiteralExpr(value));
        }

        /**
         * Checks if the class has the given annotation.
         * 
         * @param annotationClass
         *            The annotation class to check for.
         * @return true if the class has the annotation, false otherwise.
         */
        public boolean hasClassAnnotation(Class<? extends Annotation> annotationClass) {
            TypeDeclaration<?> type = getPublicType(cu);
            return type.getAnnotationByName(annotationClass.getSimpleName()).isPresent();
        }

        /** Returns the (first) public type in the given CompilationUnit. */
        private TypeDeclaration<?> getPublicType(CompilationUnit cu) {
            return cu.getTypes().stream().filter(NodeWithPublicModifier::isPublic).findFirst()
                    .orElseThrow(() -> new IllegalArgumentException("No public type found in " + cls.getName()));
        }

        /**
         * Adds an interface to the public type in the class.
         *
         * @param interfaceClass
         *            The interface class to add to the class.
         */
        public void addInterface(Class<?> interfaceClass) {
            ClassOrInterfaceDeclaration classOrInterfaceDecl = getPublicType(cu).asClassOrInterfaceDeclaration();
            if (!hasImplementedType(classOrInterfaceDecl, interfaceClass)) {
                classOrInterfaceDecl.addImplementedType(interfaceClass);
            }
        }

        /**
         * Returns whether the class has implemented the given interface.
         *
         * @param classOrInterfaceDecl
         *            The class or interface declaration to check.
         * @param interfaceClass
         *            The interface class to check for.
         * @return True if the class has implemented the interface, false otherwise.
         */
        private boolean hasImplementedType(ClassOrInterfaceDeclaration classOrInterfaceDecl, Class<?> interfaceClass) {
            return classOrInterfaceDecl.getImplementedTypes().stream()
                    .anyMatch(type -> type.getNameAsString().equals(interfaceClass.getSimpleName()));
        }
    }
}
