package com.vaadin.copilot.javarewriter;

import java.util.Objects;

import com.github.javaparser.Range;
import com.github.javaparser.ast.CompilationUnit;
import com.github.javaparser.ast.body.ClassOrInterfaceDeclaration;
import com.github.javaparser.ast.stmt.BlockStmt;
import com.github.javaparser.ast.stmt.Statement;

/**
 * Represents a point in the source code where new code can be inserted.
 */
public class InsertionPoint {
    private final CompilationUnit compilationUnit;
    private final BlockStmt block;
    private final ClassOrInterfaceDeclaration declaration;
    private int index;

    /**
     * Creates a new InsertionPoint instance.
     *
     * @param block
     *            the block where to insert the code
     * @param index
     *            the index where to insert the code
     */
    public InsertionPoint(BlockStmt block, int index) {
        this.block = block;
        this.index = index;
        this.declaration = null;
        this.compilationUnit = JavaRewriterUtil.findAncestorOrThrow(block, CompilationUnit.class);
    }

    public InsertionPoint(ClassOrInterfaceDeclaration declaration, int index) {
        this.declaration = declaration;
        this.index = index;
        this.block = null;
        this.compilationUnit = JavaRewriterUtil.findAncestorOrThrow(declaration, CompilationUnit.class);

    }

    /**
     * Returns a free variable name based on the given base name, available in the
     * scope where code will be inserted.
     *
     * @param baseName
     *            the base name for the variable
     * @return a free variable name
     */
    public String getFreeVariableName(String baseName) {
        if (getBlock() != null) {
            return JavaRewriterUtil.findFreeVariableName(baseName, getBlock());
        } else if (getDeclaration() != null) {
            return JavaRewriterUtil.findFreeVariableName(baseName, getDeclaration());
        }
        return null;
    }

    /**
     * Adds a statement to the insertion point.
     *
     * @param statement
     *            the statement to add
     */
    public void add(Statement statement) {
        if (block != null) {
            block.addStatement(index++, statement);
        } else {
            throw new IllegalArgumentException("Cannot add statement to a class or interface declaration");
        }
    }

    /**
     * Increment index by given amount
     *
     * @param increment
     *            value to add
     */
    public void incrementIndex(int increment) {
        this.index += increment;
    }

    /**
     * Increment index by adding 1
     */
    public void incrementIndexByOne() {
        incrementIndex(1);
    }

    public BlockStmt getBlock() {
        return block;
    }

    public int getIndex() {
        return index;
    }

    public ClassOrInterfaceDeclaration getDeclaration() {
        return declaration;
    }

    /**
     * Checks if this location is inside the same block as the reference location
     * but after it.
     *
     * @param referenceLocation
     *            the reference location
     * @return true if this location is inside the same block as the reference
     *         location but after it, false otherwise
     */
    public boolean isAfter(InsertionPoint referenceLocation) {
        return Objects.equals(this.block, referenceLocation.block) && this.index > referenceLocation.index;
    }

    public CompilationUnit getCompilationUnit() {
        return compilationUnit;
    }

    /**
     * Checks if an insert point is in the same range
     *
     * @param range
     *            the range to check
     * @return True if the insert point is in the same range
     */
    public boolean containsRange(Range range) {
        if (block != null) {
            Range blockRange = block.getRange().orElse(null);
            if (blockRange != null) {
                return blockRange.contains(range);
            }
        } else if (declaration != null) {
            Range declRange = declaration.getRange().orElse(null);
            if (declRange != null) {
                return declRange.contains(range);
            }
        }
        return false;
    }
}
