package com.vaadin.copilot;

import java.lang.reflect.Method;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import com.vaadin.base.devserver.DevToolsInterface;
import com.vaadin.flow.internal.JacksonUtils;
import com.vaadin.flow.internal.hilla.EndpointRequestUtil;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import tools.jackson.databind.JsonNode;
import tools.jackson.databind.node.ArrayNode;
import tools.jackson.databind.node.ObjectNode;

/**
 * Provides endpoint information to the client.
 */
public class UiServiceHandler extends CopilotCommand {

    @Override
    public boolean handleMessage(String command, JsonNode data, DevToolsInterface devToolsInterface) {
        if (command.equals("get-browser-callables")) {
            ObjectNode returnData = JacksonUtils.createObjectNode();
            returnData.put("reqId", data.get("reqId").asString());
            List<SpringBridge.ServiceMethodInfo> browserCallables = EndpointRequestUtil.isHillaAvailable()
                    ? SpringBridge.getEndpoints(getVaadinContext())
                    : new ArrayList<>();

            ArrayNode browserCallableJson = browserCallables.stream().sorted(UiServiceHandler::sortByClassAndMethodName)
                    .map(serviceMethod -> serviceMethodToJson(serviceMethod, true)).collect(JacksonUtils.asArray());
            returnData.set("browserCallables", browserCallableJson);

            ArrayNode flowUiServicesJson;
            if (SpringBridge.isSpringAvailable(getVaadinContext())) {
                List<SpringBridge.ServiceMethodInfo> flowUIServices = SpringBridge
                        .getFlowUIServices(getVaadinContext());
                flowUiServicesJson = flowUIServices.stream().sorted(UiServiceHandler::sortByClassAndMethodName)
                        .map(serviceMethod -> serviceMethodToJson(serviceMethod, false))
                        .collect(JacksonUtils.asArray());
            } else {
                flowUiServicesJson = JacksonUtils.createArray();
            }

            returnData.set("flowServices", flowUiServicesJson);

            devToolsInterface.send("resp" + command, returnData);
            return true;
        } else if (command.equals("create-service-entity")) {
            ObjectNode responseData = JacksonUtils.createObjectNode();
            responseData.put("reqId", data.get("reqId").asString());
            try {
                Path referenceFile = Util.findCurrentViewFile(getVaadinSession(), data.get("currentView")).orElseThrow()
                        .toPath();
                JavaSourcePathDetector.ModuleInfo moduleInfo = getProjectFileManager()
                        .findModule(referenceFile.toFile()).orElseThrow();
                String mainPackage = Util
                        .getPackageName(SpringBridge.getApplicationClass(getVaadinContext()).getName());

                String entityPackage = Util.decideEntityPackage(moduleInfo, referenceFile, mainPackage);
                String repositoryPackage = Util.decideRepositoryPackage(moduleInfo, referenceFile, mainPackage);
                String servicePackage = Util.decideServicePackage(moduleInfo, referenceFile, mainPackage);
                String beanName = data.get("beanName").asString();
                String qualifiedBeanName = entityPackage + "." + beanName;
                UIServiceCreator.FieldInfo[] properties = JacksonUtils.stream(data.withArray("properties"))
                        .map(obj -> UIServiceCreator.FieldInfo.fromJson(obj))
                        .toArray(UIServiceCreator.FieldInfo[]::new);
                UIServiceCreator.BeanInfo beanInfo = new UIServiceCreator.BeanInfo(qualifiedBeanName, properties);
                UIServiceCreator.DataStorage dataStorage = "jpa".equals(data.get("dataStorage").asString())
                        ? UIServiceCreator.DataStorage.JPA
                        : UIServiceCreator.DataStorage.IN_MEMORY;
                List<Map<String, Object>> exampleData = null;
                if (data.has("exampleData")) {
                    exampleData = JacksonUtils.stream(data.withArray("exampleData")).map(jsonObject -> {
                        Map<String, Object> exampleValues = new HashMap<>();
                        Iterator<Map.Entry<String, JsonNode>> it = jsonObject.properties().iterator();
                        while (it.hasNext()) {
                            Map.Entry<String, JsonNode> entry = it.next();
                            JsonNode value = jsonObject.get(entry.getKey());
                            if (value.isIntegralNumber()) {
                                exampleValues.put(entry.getKey(), value.asInt());
                            } else if (value.isFloatingPointNumber()) {
                                exampleValues.put(entry.getKey(), value.asDouble());
                            } else if (value.isBoolean()) {
                                exampleValues.put(entry.getKey(), value.asBoolean());
                            } else {
                                exampleValues.put(entry.getKey(), value.asString());
                            }
                        }
                        return exampleValues;
                    }).toList();

                }
                boolean browserCallable = data.get("browserCallable").asBoolean();
                UIServiceCreator.ServiceAndBeanInfo serviceAndBeanInfo = new UIServiceCreator.ServiceAndBeanInfo(
                        beanInfo, dataStorage, browserCallable, exampleData, servicePackage, repositoryPackage);
                boolean restartNeeded = new UIServiceCreator().createServiceAndBean(serviceAndBeanInfo, moduleInfo);
                responseData.put("restartNeeded", restartNeeded);
                ObjectNode serviceJson = JacksonUtils.createObjectNode();
                serviceJson.put("className", serviceAndBeanInfo.getServiceName());
                serviceJson.put("methodName", "list");
                JavaReflectionUtil.TypeInfo listReturnType = new JavaReflectionUtil.TypeInfo("java.util.List",
                        List.of(new JavaReflectionUtil.TypeInfo(qualifiedBeanName, List.of())));
                serviceJson.set("returnType", typeToJson(listReturnType));
                JavaReflectionUtil.ParameterTypeInfo listParameter = new JavaReflectionUtil.ParameterTypeInfo(
                        "pageable",
                        new JavaReflectionUtil.TypeInfo("org.springframework.data.domain.Pageable", List.of()));
                serviceJson.set("parameters", parametersToJson(List.of(listParameter)));
                responseData.set("service", serviceJson);
                devToolsInterface.send(command + "-resp", responseData);
            } catch (Exception e) {
                ErrorHandler.sendErrorResponse(devToolsInterface, command, responseData, "Unable to create service", e);
            }
            return true;
        }
        return false;
    }

    private JsonNode serviceMethodToJson(SpringBridge.ServiceMethodInfo serviceMethodInfo,
            boolean availableInTypescript) {
        ObjectNode json = JacksonUtils.createObjectNode();
        // canonical name can be null e.g. for anonymous classes

        json.put("className", JavaReflectionUtil.getClassName(serviceMethodInfo.serviceClass()));
        json.put("filename",
                getProjectFileManager().getFileForClass(serviceMethodInfo.serviceClass()).getAbsolutePath());
        json.put("lineNumber", 1);
        json.put("methodName", serviceMethodInfo.serviceMethod().getName());
        try {
            List<JavaReflectionUtil.ParameterTypeInfo> parameterTypes = JavaReflectionUtil
                    .getParameterTypes(serviceMethodInfo.serviceMethod(), serviceMethodInfo.serviceClass());

            json.set("parameters", parametersToJson(parameterTypes));
        } catch (Exception e) {
            getLogger().error("Unable to determine parameter types for " + serviceMethodInfo, e);
            json.set("parameters", JacksonUtils.createArray());
        }
        try {
            JavaReflectionUtil.TypeInfo returnTypeInfo = JavaReflectionUtil
                    .getReturnType(serviceMethodInfo.serviceMethod(), serviceMethodInfo.serviceClass());
            json.set("returnType", typeToJson(returnTypeInfo));
        } catch (Exception e) {
            getLogger().error("Unable to determine return type for " + serviceMethodInfo, e);
            json.set("returnType", JacksonUtils.createObjectNode());
        }
        json.put("availableInTypescript", availableInTypescript);
        json.put("canBindToFlowGrid", ConnectToService.canBindFlowGridToService(serviceMethodInfo).name());
        json.put("canBindToHillaGrid", ConnectToService.canBindHillaGridToService(serviceMethodInfo).name());
        json.put("canBindToFlowComboBox", ConnectToService.canBindFlowComboBoxToService(serviceMethodInfo).name());
        json.put("canBindToHillaComboBox", ConnectToService.canBindHillaComboBoxToService(serviceMethodInfo).name());
        try {
            AccessRequirement req = AccessRequirementUtil.getAccessRequirement(serviceMethodInfo.serviceMethod(),
                    serviceMethodInfo.serviceClass());
            json.set("accessRequirement", JacksonUtils.beanToJson(req));
        } catch (Exception e) {
            getLogger().error("Unable to determine access requirement", e);
        }
        return json;
    }

    private ArrayNode parametersToJson(List<JavaReflectionUtil.ParameterTypeInfo> parameterTypes) {
        return parameterTypes.stream().map(param -> {
            ObjectNode paramInfo = JacksonUtils.createObjectNode();
            paramInfo.put("name", param.name());
            paramInfo.set("type", typeToJson(param.type()));
            return paramInfo;
        }).collect(JacksonUtils.asArray());
    }

    private JsonNode typeToJson(JavaReflectionUtil.TypeInfo typeInfo) {
        ObjectNode json = JacksonUtils.createObjectNode();
        json.put("typeName", typeInfo.typeName());
        json.set("typeParameters",
                typeInfo.typeParameters().stream().map(this::typeToJson).collect(JacksonUtils.asArray()));
        return json;
    }

    public static int sortByClassAndMethodName(SpringBridge.ServiceMethodInfo e1, SpringBridge.ServiceMethodInfo e2) {
        return sortByClassAndMethodName(e1.serviceClass(), e1.serviceMethod(), e2.serviceClass(), e2.serviceMethod());
    }

    private static int sortByClassAndMethodName(Class<?> class1, Method method1, Class<?> class2, Method method2) {
        if (!class1.equals(class2)) {
            return JavaReflectionUtil.getClassName(class1).compareTo(JavaReflectionUtil.getClassName(class2));
        } else {
            return method1.getName().compareTo(method2.getName());
        }

    }

    private static Logger getLogger() {
        return LoggerFactory.getLogger(UiServiceHandler.class);
    }
}
