package com.vaadin.copilot;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.charset.MalformedInputException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.nio.file.StandardOpenOption;
import java.util.Base64;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

import com.vaadin.copilot.analytics.AnalyticsClient;
import com.vaadin.copilot.ide.CopilotIDEPlugin;
import com.vaadin.flow.component.internal.ComponentTracker;
import com.vaadin.flow.internal.UsageStatistics;
import com.vaadin.flow.router.RouteData;
import com.vaadin.flow.server.RouteRegistry;
import com.vaadin.flow.server.VaadinServletContext;
import com.vaadin.flow.server.VaadinSession;
import com.vaadin.flow.server.frontend.FrontendUtils;
import com.vaadin.flow.server.frontend.ThemeUtils;
import com.vaadin.flow.server.startup.ApplicationConfiguration;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ProjectFileManager {
    private static ProjectFileManager instance = null;

    private final ApplicationConfiguration _applicationConfiguration;
    private JavaSourcePathDetector.ProjectPaths projectPaths0 = null;

    /**
     * Public only for tests. Always access through {@link #get()} instead.
     *
     * @param applicationConfiguration
     *            the application configuration
     */
    public ProjectFileManager(ApplicationConfiguration applicationConfiguration) {
        this._applicationConfiguration = applicationConfiguration;
    }

    public static synchronized ProjectFileManager initialize(ApplicationConfiguration applicationConfiguration) {
        if (instance == null || instance.getApplicationConfiguration() != applicationConfiguration) {
            instance = new ProjectFileManager(applicationConfiguration);
        }
        return instance;
    }

    private static Logger getLogger() {
        return LoggerFactory.getLogger(ProjectFileManager.class);
    }

    public static ProjectFileManager get() {
        return instance;
    }

    public String readFile(String filename) throws IOException {
        return readFile(new File(filename));
    }

    public String readFile(Path filename) throws IOException {
        return readFile(filename.toFile());
    }

    public String readFile(File file) throws IOException {
        if (file == null) {
            throw new IllegalArgumentException("File cannot be null");
        }
        if (!file.exists()) {
            throw new FileNotFoundException(file.getAbsolutePath());
        }
        if (isFileInsideProject(file)) {
            try {
                return Files.readString(file.toPath(), StandardCharsets.UTF_8);
            } catch (MalformedInputException e) {
                return Files.readString(file.toPath(), StandardCharsets.ISO_8859_1);
            }
        } else {
            throw createFileIsNotInProjectException(file);
        }
    }

    private IllegalArgumentException createFileIsNotInProjectException(File file) {
        return new IllegalArgumentException(
                "File " + file.getPath() + " is not inside the project " + getProjectPaths());

    }

    public List<String> readLines(File file) throws IOException {
        if (!isFileInsideProject(file)) {
            throw createFileIsNotInProjectException(file);
        }
        return Files.readAllLines(file.toPath(), StandardCharsets.UTF_8);
    }

    boolean isFileInsideProject(File file) {
        return findModule(file).isPresent();
    }

    /**
     * Writes the given content to the given file inside the project.
     *
     * <p>
     * If the filename is absolute, it is used as is. Otherwise, it is resolved
     * relative to the project root.
     *
     * <p>
     * If the file is outside the project, an exception is thrown
     *
     * @param filename
     *            the filename to write to, absolute or relative to the project root
     * @param undoLabel
     *            the undo label for the change
     * @param content
     *            the content to write
     * @throws IOException
     *             if the file cannot be written
     */
    public void writeFile(String filename, String undoLabel, String content) throws IOException {
        writeFile(getAbsolutePath(filename), undoLabel, content);
    }

    /**
     * Writes the given content to the given file inside the project.
     *
     * <p>
     * If the filename is absolute, it is used as is. Otherwise, it is resolved
     * relative to the project root.
     *
     * <p>
     * If the file is outside the project, an exception is thrown
     *
     * @param file
     *            the filename to write to, absolute or relative to the project root
     * @param undoLabel
     *            the undo label for the change
     * @param content
     *            the content to write
     * @throws IOException
     *             if the file cannot be written
     */
    public void writeFile(File file, String undoLabel, String content) throws IOException {
        writeFile(file, undoLabel, content, false);
    }

    public void writeFile(Path file, String undoLabel, String content) throws IOException {
        writeFile(file.toFile(), undoLabel, content, false);
    }

    public File writeFileBase64(String filename, String undoLabel, String base64Content, boolean renameIfExists)
            throws IOException {
        var target = getAbsolutePath(filename);
        while (renameIfExists && target.exists()) {
            target = new File(target.getParentFile(), Util.increaseTrailingNumber(target.getName()));
        }
        writeFile(target, undoLabel, base64Content, true);
        return target;
    }

    private void writeFile(File file, String undoLabel, String content, boolean base64Encoded) throws IOException {
        if (!isFileInsideProject(file)) {
            throw createFileIsNotInProjectException(file);
        }

        File folder = file.getParentFile();
        if (!folder.exists() && !folder.mkdirs()) {
            throw new IOException("Unable to create folder " + folder.getAbsolutePath());
        }
        AnalyticsClient.getInstance().track("write-file",
                Map.of("binary", String.valueOf(base64Encoded), "type", Util.getExtension(file.getName())));
        UsageStatistics.markAsUsed("copilot/write-file", CopilotVersion.getVersion());
        CopilotIDEPlugin idePlugin = CopilotIDEPlugin.getInstance();
        if (idePlugin.isActive()) {
            if (base64Encoded && idePlugin.supports(CopilotIDEPlugin.Commands.WRITE_BASE64)) {
                idePlugin.writeBase64File(file, undoLabel, content);
                return;
            } else if (!base64Encoded && idePlugin.supports(CopilotIDEPlugin.Commands.WRITE)) {
                idePlugin.writeFile(file, undoLabel, content);
                return;
            }
        }
        writeFileIfNeeded(file, content, base64Encoded);
    }

    private void writeFileIfNeeded(File file, String content, boolean base64Encoded) throws IOException {
        if (file.exists()) {
            String currentContent;
            if (base64Encoded) {
                byte[] data = Files.readAllBytes(file.toPath());
                currentContent = base64Encode(data);
            } else {
                currentContent = Files.readString(file.toPath(), StandardCharsets.UTF_8);
            }
            if (content.equals(currentContent)) {
                return;
            }
        }

        byte[] data;
        if (base64Encoded) {
            data = base64Decode(content);
        } else {
            data = content.getBytes(StandardCharsets.UTF_8);
        }
        if (file.exists()) {
            try (FileChannel channel = FileChannel.open(file.toPath(), StandardOpenOption.WRITE)) {
                int written = channel.write(ByteBuffer.wrap(data));
                channel.truncate(written);
            }
        } else {
            Files.write(file.toPath(), data);
        }
        // If the file was a resource, we can directly copy it to target
        // to make it available
        copyResourceToTarget(file);
    }

    private void copyResourceToTarget(File resourceFile) throws IOException {
        Optional<Path> maybeResourceFolder = findResourceFolder(resourceFile);
        if (maybeResourceFolder.isEmpty()) {
            // Not a resource
            return;
        }
        File resourceFolder = maybeResourceFolder.get().toFile();
        Optional<JavaSourcePathDetector.ModuleInfo> module = findModule(resourceFolder);
        if (module.isEmpty()) {
            getLogger().error("Unable to determine module for resource folder {}", resourceFolder);
            return;
        }

        String relativeResourceName = Util.getRelativeName(resourceFile, resourceFolder);
        File target = new File(module.get().classesFolder().toFile(), relativeResourceName);
        Files.createDirectories(target.toPath().getParent());
        Files.copy(resourceFile.toPath(), target.toPath(), StandardCopyOption.REPLACE_EXISTING);
    }

    private byte[] base64Decode(String content) {
        return Base64.getDecoder().decode(content);
    }

    private String base64Encode(byte[] data) {
        return Base64.getEncoder().encodeToString(data);
    }

    /**
     * Changes the relative file name into an absolute file name and validates it is
     * inside the project.
     *
     * @deprecated This rarely ever works as expected because the file you want to
     *             write is in one of the Maven modules, not necessarily in the root
     *             project.
     */
    @Deprecated
    public String makeAbsolute(String projectRelativeFilename) throws IOException {
        Path projectRootPath = getProjectRoot().toPath().toRealPath();
        Path resolved = projectRootPath.resolve(projectRelativeFilename);
        if (!isFileInsideProject(resolved.toFile())) {
            throw new IllegalArgumentException(
                    "File " + projectRelativeFilename + " is not inside the project at " + projectRootPath);
        }
        return resolved.toAbsolutePath().toString();
    }

    public String makeRelative(String filename) throws IOException {
        Path projectRootPath = getProjectRoot().toPath().toRealPath();
        Path absolutePath = new File(filename).toPath().toRealPath();
        // If file is not inside project root, throw
        if (!absolutePath.startsWith(projectRootPath)) {
            throw new IllegalArgumentException("File " + filename + " is not inside the project at " + projectRootPath);
        }
        return projectRootPath.relativize(absolutePath).toString();
    }

    /**
     * Returns the name of the file, relative to the project root.
     *
     * @param projectFile
     *            the file
     * @return the relative name of the file
     */
    public String getProjectRelativeName(File projectFile) {
        return Util.getRelativeName(projectFile, getProjectRoot());
    }

    /**
     * Returns the Java file for the given class.
     *
     * @param cls
     *            the class
     * @return the file for the class
     */
    public File getFileForClass(Class<?> cls) {
        return getFileForClass(cls.getName());
    }

    /**
     * Returns the Java file for the given class.
     *
     * @param cls
     *            the class
     * @return the file for the class
     */
    public File getFileForClass(String cls) {
        if (cls.contains("$")) {
            cls = cls.substring(0, cls.indexOf("$"));
        }
        cls = cls.replace(".", File.separator);

        for (Path path : getSourceFolders()) {
            for (String extension : new String[] { "java", "kt" }) {
                String filename = cls + "." + extension;
                File file = new File(path.toFile(), filename);
                if (file.exists()) {
                    return file;
                }
            }
        }

        // This is to be compatible with existing code which assumes a non-null return
        // value even if the file does not exist
        return new File(getSourceFolders().get(0).toFile(), cls + ".java");
    }

    /**
     * Returns the source folders for the project.
     *
     * @return the source folders
     */
    public List<Path> getSourceFolders() {
        return getProjectPaths().allSourcePaths();
    }

    /**
     * Gets the path information for the project.
     * <p>
     * Synchronized as during startup there can be multiple threads trying to access
     * this information, e.g. the hotswap listener which will run many times in
     * parallel for many classes.
     *
     * @return the project paths
     */
    synchronized JavaSourcePathDetector.ProjectPaths getProjectPaths() {
        if (projectPaths0 == null) {
            projectPaths0 = JavaSourcePathDetector.detectProjectPaths(getApplicationConfiguration());
            getLogger().debug("Project folders detected: {}", projectPaths0);
        }
        return projectPaths0;
    }

    /**
     * Returns the resource folders for the project.
     *
     * @return the resource folders
     */
    public List<Path> getResourceFolders() {
        return getProjectPaths().allResourcePaths();
    }

    /**
     * Returns the Java file for the given component location.
     *
     * @param location
     *            the component location
     * @return the file for the class where the component is used
     */
    public File getSourceFile(ComponentTracker.Location location) {
        return getFileForClass(location.className());
    }

    /**
     * Returns the project base folder.
     * <p>
     * The project base folder is the common folder containing all module roots.
     *
     * @return the project root folder
     */
    public File getProjectRoot() {
        return getProjectPaths().basedir().toFile();
    }

    /**
     * Returns the frontend folder.
     *
     * @return the frontend folder
     */
    public File getFrontendFolder() {
        return FrontendUtils.getProjectFrontendDir(getApplicationConfiguration());
    }

    /**
     * Returns the styles folder (META-INF/resources).
     * 
     * @return the styles folder
     */
    public File getStylesFolder() {
        return new File(getJavaResourceFolder(), "META-INF/resources");
    }

    /**
     * Returns the styles.css file.
     *
     * @return the styles.css file
     */
    public File getStylesCss() {
        return new File(getThemeFolder(), "styles.css");
    }

    /**
     * Returns the java resource folder.
     *
     * @return the java resource folder.
     */
    public File getJavaResourceFolder() {
        return getApplicationConfiguration().getJavaResourceFolder();
    }

    /**
     * Returns the java source folder.
     *
     * @return the java source folder.
     */
    public File getJavaSourceFolder() {
        return getApplicationConfiguration().getJavaSourceFolder();
    }

    /**
     * Gets current theme name
     *
     * @return optional theme name
     */
    public Optional<String> getThemeName() {
        return ThemeUtils.getThemeName(getApplicationConfiguration().getContext());
    }

    /**
     * Gets current theme folder if present, fallbacks to frontend folder if none.
     *
     * @return theme folder if present, resources styles folder otherwise
     */
    public File getThemeFolder() {
        return getThemeName().map(t -> ThemeUtils.getThemeFolder(getFrontendFolder(), t))
                .orElseGet(() -> getStylesFolder());
    }

    public File getAbsolutePath(String filename) {
        File file = new File(filename);
        if (!file.isAbsolute()) {
            file = new File(getProjectRoot(), filename);
        }
        return file;
    }

    /**
     * Makes a string safe to use as a file name
     *
     * @param name
     *            the string to process
     * @return the sanitized string
     */
    public String sanitizeFilename(String name) {
        return name.replaceAll("[^a-zA-Z0-9-_@]", "_");
    }

    /**
     * Finds the folder where Hilla views should be created for the file system
     * router to pick them up.
     *
     * @return the folder where Hilla views should be created
     */
    public File getHillaViewsFolder() {
        return new File(getFrontendFolder(), "views");
    }

    /**
     * Finds the folder where a new Flow view should be created.
     * <p>
     * If all views are in the same package/folder, that folder is returned. If
     * views are in different packages, the folder for a common prefix is returned.
     * If no common prefix is found, the folder for the main package is returned.
     *
     * @return a suitable folder to place a new Flow view in
     */
    public File getFlowNewViewFolder(VaadinSession vaadinSession) {
        Map<String, Class<?>> viewPackages = new HashMap<>();
        vaadinSession.accessSynchronously(() -> {
            RouteRegistry routeRegistry = vaadinSession.getService().getRouter().getRegistry();
            for (RouteData route : routeRegistry.getRegisteredRoutes()) {
                viewPackages.put(route.getNavigationTarget().getPackageName(), route.getNavigationTarget());
            }
        });
        if (!viewPackages.isEmpty()) {
            // There is at least one view, so we can figure out a good package name from the
            // views
            File someViewJavaFile = getFileForClass(viewPackages.values().iterator().next());
            if (viewPackages.size() == 1) {
                return someViewJavaFile.getParentFile();
            }
            // Find common package prefix
            String commonPrefix = "";
            for (String viewPackage : viewPackages.keySet()) {
                if (commonPrefix.isEmpty()) {
                    commonPrefix = viewPackage;
                } else if (viewPackage.startsWith(commonPrefix)) {
                    continue;
                } else {
                    while (!commonPrefix.isEmpty() && !viewPackage.startsWith(commonPrefix)) {
                        commonPrefix = commonPrefix.substring(0, commonPrefix.lastIndexOf('.'));
                    }
                }
            }
            Optional<Path> sourceFolder = findSourceFolder(someViewJavaFile);
            if (!commonPrefix.isEmpty()) {
                for (String folder : commonPrefix.split("\\.")) {
                    sourceFolder = sourceFolder.map(p -> p.resolve(folder));
                }

                if (sourceFolder.isPresent()) {
                    return sourceFolder.get().toFile();
                }
            }
        }
        // Project might not have any Flow views yet
        var sourceFolder = getSourceFolders().get(0);
        VaadinServletContext context = Copilot.getContext(vaadinSession);
        if (SpringBridge.isSpringAvailable(context)) {
            Class<?> springApplicationClass = SpringBridge.getApplicationClass(context);
            if (springApplicationClass != null) {
                Optional<Path> folder = findSourceFolder(springApplicationClass);
                if (folder.isPresent()) {
                    sourceFolder = folder.get();
                }
            }
        }
        File folder = Util.getSinglePackage(sourceFolder.toFile());
        if (folder.getName().equals("views")) {
            return folder;
        }
        return new File(folder, "views");
    }

    /**
     * Finds the source folder where the given class is located.
     *
     * @param cls
     *            the class to find the source folder for
     * @return the source folder or an empty optional if the source folder could not
     *         be found
     * @throws IOException
     *             if something goes wrong
     */
    Optional<Path> findSourceFolder(Class<?> cls) {
        File sourceFile = getFileForClass(cls);
        if (!sourceFile.exists()) {
            return Optional.empty();
        }

        return findSourceFolder(sourceFile);
    }

    /**
     * Finds the given resource by name, in any resource folder.
     *
     * @param resource
     *            the name of the resource to look for
     */
    public Optional<Path> findResource(String resource) {
        for (Path resourceFolder : getResourceFolders()) {
            Path resourceFile = resourceFolder.resolve(resource);
            if (resourceFile.toFile().exists()) {
                return Optional.of(resourceFile);
            }
        }
        return Optional.empty();
    }

    /**
     * Finds the resource folder where the given resource is located.
     *
     * @param resourceFile
     *            the resource to find the resource folder for
     */
    public Optional<Path> findResourceFolder(File resourceFile) {
        for (Path resourceFolder : getResourceFolders()) {
            if (Util.isFileInside(resourceFile, resourceFolder.toFile())) {
                return Optional.of(resourceFolder);
            }
        }
        return Optional.empty();
    }

    /**
     * Finds the module where the given resource is located.
     *
     * @param file
     *            the file to look up
     */
    public Optional<JavaSourcePathDetector.ModuleInfo> findModule(File file) {
        if (file == null) {
            return Optional.empty();
        }
        for (JavaSourcePathDetector.ModuleInfo moduleInfo : getProjectPaths().modules()) {
            if (Util.isFileInside(file, moduleInfo.rootPath().toFile())) {
                return Optional.of(moduleInfo);
            }
        }
        return Optional.empty();
    }

    /**
     * Finds the source folder where the given file is located.
     *
     * @param sourceFile
     *            the source file to find the source folder for
     */
    public Optional<Path> findSourceFolder(File sourceFile) {
        for (Path sourceFolder : getSourceFolders()) {
            if (Util.isFileInside(sourceFile, sourceFolder.toFile())) {
                return Optional.of(sourceFolder);
            }
        }
        return Optional.empty();
    }

    /**
     * Gets the java package for the given source file
     *
     * @param sourceFile
     *            the source file
     * @return the java package for the given source file, or null if the source
     *         file is not inside the project
     * @throws IOException
     *             if something goes wrong
     */
    public String getJavaPackage(File sourceFile) throws IOException {
        Optional<Path> sourceFolder = findSourceFolder(sourceFile);
        if (sourceFolder.isEmpty()) {
            return null;
        }
        return Util.getRelativeName(sourceFile.getParentFile(), sourceFolder.get().toFile()).replace(File.separator,
                ".");
    }

    /**
     * Gets the modules in the project.
     *
     * @return a list of modules in the project
     */
    public List<JavaSourcePathDetector.ModuleInfo> getModules() {
        return getProjectPaths().modules();
    }

    public ApplicationConfiguration getApplicationConfiguration() {
        return _applicationConfiguration;
    }

    public void deleteFile(File file) throws IOException {
        Objects.requireNonNull(file);
        if (!isFileInsideProject(file)) {
            throw createFileIsNotInProjectException(file);
        }
        Files.delete(file.toPath());

        AnalyticsClient.getInstance().track("delete-file",
                Map.of("fileName", Util.getFileName(file.getAbsolutePath())));
    }

    public void moveFile(File sourceFile, File targetFile) throws IOException {
        Objects.requireNonNull(sourceFile);
        Objects.requireNonNull(targetFile);
        if (!isFileInsideProject(sourceFile)) {
            throw createFileIsNotInProjectException(sourceFile);
        }
        Files.move(sourceFile.toPath(), targetFile.toPath());
    }
}
