package com.vaadin.copilot;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.file.Files;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.vaadin.base.devserver.stats.ProjectHelpers;
import com.vaadin.flow.server.frontend.installer.DownloadException;
import com.vaadin.frontendtools.installer.ArchiveExtractionException;
import com.vaadin.frontendtools.installer.DefaultArchiveExtractor;
import com.vaadin.open.OSUtils;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/** Utility class for downloading and referencing a JetBrains Runtime. */
public final class JetbrainsRuntimeUtil {

    private static final String TAR_GZ = ".tar.gz";
    private static final String JETBRAINS_GITHUB_LATEST_RELEASE = "https://api.github.com/repos/JetBrains/JetBrainsRuntime/releases/latest";

    /** Describes a GitHub release with a body. */
    record GitHubReleaseWithBody(String body) {
    }

    /** Describes a GitHub release. */
    record GitHubRelease(int id, String name, String tag_name, boolean prerelease) {
    }

    /** Describes a version of JetBrains Runtime SDK that could be downloaded. */
    record JBRSdkInfo(String arch, String sdkType, String url) {
    }

    private static String getArchitecture() {
        return System.getProperty("os.arch");
    }

    /**
     * Unpacks the given JetBrains Runtime archive into a folder.
     *
     * @param jbrArchive
     *            the archive to unpack
     * @param statusUpdater
     *            the consumer to pass status information to
     * @return the folder where the archive was unpacked
     * @throws IOException
     *             if an I/O error occurs
     * @throws ArchiveExtractionException
     *             if the archive cannot be extracted
     */
    public File unpackJbr(File jbrArchive, Consumer<String> statusUpdater)
            throws IOException, ArchiveExtractionException {
        // Unpack the archive
        if (!jbrArchive.getName().endsWith(TAR_GZ)) {
            throw new IOException("Unexpected file format: " + jbrArchive.getName());
        }

        File folder = new File(jbrArchive.getParentFile(), jbrArchive.getName().replace(TAR_GZ, ""));
        if (folder.exists()) {
            File[] files = folder.listFiles();
            if (files == null || files.length == 0) {
                Files.delete(folder.toPath());
            } else {
                statusUpdater.accept("Using JetBrains already in " + folder.getAbsolutePath());
                return folder;
            }
        }
        statusUpdater.accept("Extracting " + jbrArchive.getAbsolutePath() + " into " + folder.getAbsolutePath());

        new DefaultArchiveExtractor().extract(jbrArchive, jbrArchive.getParentFile());
        statusUpdater.accept("Extraction complete");

        return folder;
    }

    private File getJavaHome(File jdkFolder) {
        if (OSUtils.isMac()) {
            return jdkFolder.toPath().resolve("Contents").resolve("Home").toFile();
        }
        return jdkFolder;
    }

    /**
     * Returns the location inside the JDK where HotswapAgent should be placed.
     *
     * @param jdkFolder
     *            the JDK folder
     * @return the location inside the JDK where HotswapAgent should be placed
     */
    public File getHotswapAgentLocation(File jdkFolder) {
        return new File(new File(new File(getJavaHome(jdkFolder), "lib"), "hotswap"), "hotswap-agent.jar");
    }

    /**
     * Returns the location of the Java executable inside the JDK.
     *
     * @param jdkFolder
     *            the JDK folder
     * @return the location of the Java executable inside the JDK
     */
    public File getJavaExecutable(File jdkFolder) {
        String bin = OSUtils.isWindows() ? "java.exe" : "java";
        return new File(new File(getJavaHome(jdkFolder), "bin"), bin);
    }

    /**
     * Downloads the latest JetBrains Runtime and returns the downloaded file.
     *
     * <p>
     * The downloaded file will be placed in the Vaadin home directory under the
     * "jdk" folder. If the file already exists, it will not be downloaded again.
     *
     * <p>
     * If no suitable download is found, an empty optional is returned.
     *
     * @param statusUpdater
     *            the consumer to pass status information to
     * @return the downloaded file, if any
     * @throws IOException
     *             if an I/O error occurs
     * @throws URISyntaxException
     *             if there is an internal error with url handling
     * @throws DownloadException
     *             if the download fails
     */
    public Optional<File> downloadLatestJBR(Consumer<String> statusUpdater)
            throws IOException, URISyntaxException, DownloadException {
        statusUpdater.accept("Finding JetBrains Runtime download location");

        // Fetch the latest release from the given URL
        GitHubRelease latest = findLatestJBRRelease();
        Optional<URL> downloadUrl = findJBRDownloadUrl(latest);
        if (downloadUrl.isPresent()) {
            URL url = downloadUrl.get();
            String filename = getFilename(url);
            File target = new File(new File(ProjectHelpers.resolveVaadinHomeDirectory(), "jdk"), filename);
            File folder = target.getParentFile();
            if (!folder.exists() && !folder.mkdirs()) {
                throw new IOException("Unable to create " + folder.getAbsolutePath());
            }

            downloadIfNotPresent(url, target, statusUpdater);
            return Optional.of(target);
        }
        return Optional.empty();
    }

    private void downloadIfNotPresent(URL url, File target, Consumer<String> statusUpdater)
            throws URISyntaxException, DownloadException {
        if (target.exists() && target.length() > 0) {
            statusUpdater.accept("JetBrains Runtime already downloaded into " + target.getAbsolutePath());
            return;
        }
        statusUpdater.accept("Downloading JetBrains Runtime from " + url);
        Downloader.downloadFile(url, target, (bytesTransferred, totalBytes, progress) -> statusUpdater
                .accept(HotswapDownloadHandler.PROGRESS_PREFIX + progress));
        statusUpdater.accept("Downloaded JetBrains Runtime into " + target.getAbsolutePath());
    }

    private String getFilename(URL url) {
        return url.getFile().replaceAll(".*/", "");
    }

    private Optional<URL> findJBRDownloadUrl(GitHubRelease latest) {
        try {
            URL downloadUrl = new URI("https://api.github.com/repos/JetBrains/JetBrainsRuntime/releases/" + latest.id)
                    .toURL();
            InputStream data = Downloader.download(downloadUrl, null);
            GitHubReleaseWithBody release = CopilotJacksonUtils.readValue(data, GitHubReleaseWithBody.class);
            Optional<JBRSdkInfo> sdk = findCorrectReleaseForArchitecture(release.body);
            if (sdk.isPresent()) {
                return Optional.of(new URL(sdk.get().url()));
            }
        } catch (IOException | URISyntaxException | DownloadException e) {
            getLogger().error("Unable to fetch JetBrains Runtime download URL", e);
        }
        return Optional.empty();
    }

    private static Logger getLogger() {
        return LoggerFactory.getLogger(JetbrainsRuntimeUtil.class);
    }

    private GitHubRelease findLatestJBRRelease() throws IOException {
        try {
            URL latestReleaseUrl = new URI(JETBRAINS_GITHUB_LATEST_RELEASE).toURL();
            InputStream stream = Downloader.download(latestReleaseUrl, null);
            return CopilotJacksonUtils.readValue(stream, GitHubRelease.class);
        } catch (IOException | URISyntaxException | DownloadException e) {
            throw new IOException("Unable to fetch JetBrains Runtime releases info", e);
        }
    }

    private Optional<JBRSdkInfo> findCorrectReleaseForArchitecture(String body) {
        Map<String, JBRSdkInfo> jbrSdks = findAllJbrSdks(body);
        String key = getDownloadKey();
        return Optional.ofNullable(jbrSdks.get(key));
    }

    /**
     * Returns the key to use for downloading the correct JBR SDK for the current
     * architecture.
     *
     * @return the key to use for downloading
     */
    public String getDownloadKey() {
        String jvmArch = getArchitecture();
        String prefix;
        if (OSUtils.isMac()) {
            prefix = "osx";
        } else if (OSUtils.isWindows()) {
            prefix = "windows";
        } else {
            prefix = "linux";
        }
        String suffix;
        if ("aarch64".equals(jvmArch)) {
            suffix = "aarch64";
        } else if ("x86".equals(jvmArch)) {
            suffix = "x86";
        } else {
            suffix = "x64";
        }
        return prefix + "-" + suffix;
    }

    private static Map<String, JBRSdkInfo> findAllJbrSdks(String body) {
        String[] lines = body.replace("\r", "").split("\n");
        List<JBRSdkInfo> sdks = Arrays.stream(lines).map(line -> {
            String[] parts = line.split("\\|");
            if (parts.length < 4) {
                return null;
            }
            String arch = parts[1].trim();
            String sdkType = parts[2].replace("*", "").trim();
            String url = parts[3].replaceAll("\\[.*]", "").replaceAll("[()]", "").trim();
            return new JBRSdkInfo(arch, sdkType, url);
        }).filter(Objects::nonNull).toList();
        Stream<JBRSdkInfo> filtered = sdks.stream().filter(jbrSdkInfo -> jbrSdkInfo.sdkType.equals("JBRSDK"))
                .filter(jbrSdkInfo -> jbrSdkInfo.url.endsWith(TAR_GZ))
                .filter(jbrSdkInfo -> !jbrSdkInfo.url.contains("_diz"));
        return filtered.collect(Collectors.toMap(sdkInfo -> sdkInfo.arch, sdkInfo -> sdkInfo, (sdkInfo1, sdkInfo2) -> {
            // We don't really know how to handle collisions as there shouldn't be any. Just
            // use the first
            return sdkInfo1;
        }));
    }
}
