package com.vaadin.copilot.javarewriter;

import java.io.File;
import java.io.IOException;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

import com.vaadin.copilot.ProjectManager;
import com.vaadin.flow.component.Component;
import com.vaadin.flow.component.HasEnabled;
import com.vaadin.flow.component.HasSize;
import com.vaadin.flow.component.HasText;
import com.vaadin.flow.component.HasTheme;
import com.vaadin.flow.component.HasValueAndElement;
import com.vaadin.flow.dom.ElementStateProvider;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Handles the copy &amp; paste functionality for flow components.
 */
public class JavaRewriterCopyPasteHandler {

    private static final String VALUE_PROPERTY_KEY = "value";
    private static final String INVALID_PROPERTY_KEY = "invalid";
    private static final String ERROR_MESSAGE_PROPERTY_KEY = "errorMessage";
    private static final String ENABLED_PROPERTY_KEY = "enabled";

    private static final String INDETERMINATE_PROPERTY_KEY = "indeterminate";
    private static final String CHECKED_PROPERTY_KEY = "checked";
    private static final String AUTO_FOCUS_PROPERTY_KEY = "autofocus";
    private static final String REQUIRED_PROPERTY_KEY = "required";

    private static final String ACCESSIBLE_NAME_REF = "accessibleNameRef";
    private static final String ACCESSIBLE_NAME = "accessibleName";
    private static final String REQUIRED_INDICATOR_VISIBLE = "requiredIndicatorVisible";

    private static final String READONLY = "readonly";
    private static final String READ_ONLY_CAMEL_CASE = "readOnly";
    private final ProjectManager projectManager;

    public JavaRewriterCopyPasteHandler(ProjectManager projectManager) {
        this.projectManager = projectManager;
    }

    /**
     * Collects required data for copying and pasting a component
     *
     * @param componentTypeAndSourceLocation
     *            Component type and source location with children info included
     * @return JavaComponent that will be used for pasting.
     * @throws IOException
     *             exception happens when file read fails
     */
    public JavaComponent getCopiedJavaComponent(ComponentTypeAndSourceLocation componentTypeAndSourceLocation)
            throws IOException {
        Component component = componentTypeAndSourceLocation.component();
        ElementStateProvider stateProvider = component.getElement().getStateProvider();
        Map<String, Object> properties = stateProvider.getPropertyNames(component.getElement().getNode())
                .collect(Collectors.toMap(k -> k, k -> stateProvider.getProperty(component.getElement().getNode(), k)));

        File javaFile = projectManager.getSourceFile(componentTypeAndSourceLocation.getCreateLocationOrThrow());
        JavaRewriter rewriter = new JavaRewriter(projectManager.readFile(javaFile));
        ComponentInfo componentInfo = rewriter.findComponentInfo(componentTypeAndSourceLocation);

        addSuffixOrPrefixAttributes(component, properties);
        addExtraProperties(component, properties);
        filterOutProperties(component, componentInfo, componentTypeAndSourceLocation.inheritanceChain(), properties);

        JavaComponent.Metadata copyMetadata = new JavaComponent.Metadata();
        try {
            copyMetadata = getCopyMetadata(componentTypeAndSourceLocation);
        } catch (Exception ex) {
            getLogger().error(ex.getMessage(), ex);
        }

        List<JavaComponent> childrenList = childrenFilter(componentTypeAndSourceLocation);

        return new JavaComponent(getTag(componentTypeAndSourceLocation.inheritanceChain()),
                getFlowComponentClassName(componentTypeAndSourceLocation.inheritanceChain()), properties, childrenList,
                copyMetadata);
    }

    private List<JavaComponent> childrenFilter(ComponentTypeAndSourceLocation componentTypeAndSourceLocation)
            throws IOException {
        Component component = componentTypeAndSourceLocation.component();
        boolean isDateTimePicker = "com.vaadin.flow.component.datetimepicker.DateTimePicker"
                .equals(component.getClass().getName());
        if (isDateTimePicker) {
            return List.of();
        }
        List<JavaComponent> childrenList = new ArrayList<>();
        for (ComponentTypeAndSourceLocation typeAndSourceLocation : componentTypeAndSourceLocation.children()) {
            JavaComponent copiedJavaComponent = getCopiedJavaComponent(typeAndSourceLocation);
            childrenList.add(copiedJavaComponent);
        }
        return childrenList;
    }

    private String getFlowComponentClassName(List<Class<?>> inheritanceChain) {
        for (Class<?> aClass : inheritanceChain) {
            if (aClass.getName().startsWith("com.vaadin.flow")) {
                return aClass.getName();
            }
        }
        return inheritanceChain.get(0).getName();
    }

    /**
     * When a view is copied from the project, it should return the extended class
     * instead of view itself. //TODO in future this might change
     *
     * @param inheritanceChain
     *            inheritance chain of class
     * @return Tag name of the component e.g. class simple name
     */
    private String getTag(List<Class<?>> inheritanceChain) {
        for (Class<?> aClass : inheritanceChain) {
            if (aClass.getName().startsWith("com.vaadin.flow")) {
                return getTemplateTag(aClass.getSimpleName());
            }
        }
        return getTemplateTag(inheritanceChain.get(0).getSimpleName());
    }

    private String getTemplateTag(String originalTag) {
        if ("RadioButtonGroup".equals(originalTag)) {
            return "RadioGroup";
        }
        if ("CheckBoxItem".equals(originalTag)) {
            return "Checkbox";
        }
        return originalTag;
    }

    private JavaComponent.Metadata getCopyMetadata(ComponentTypeAndSourceLocation componentTypeAndSourceLocation)
            throws IOException {
        JavaComponent.Metadata metadata = new JavaComponent.Metadata();
        if (componentTypeAndSourceLocation.getCreateLocationOrThrow().className().startsWith("com.vaadin.flow.data")) {
            return metadata;
        }
        JavaRewriter rewriter = new JavaRewriter(projectManager.readFile(componentTypeAndSourceLocation.javaFile()));
        ComponentInfo componentInfo = rewriter.findComponentInfo(componentTypeAndSourceLocation);

        Optional.ofNullable(componentInfo.localVariableName()).ifPresent(metadata::setLocalVariableName);
        Optional.ofNullable(componentInfo.fieldName()).ifPresent(metadata::setFieldVariableName);
        metadata.setOriginalClassName(componentTypeAndSourceLocation.component().getClass().getName());
        return metadata;
    }

    private void addExtraProperties(Component component, Map<String, Object> properties) {

        if (component.getClassName() != null && !component.getClassName().isEmpty()) {
            properties.put("className", component.getClassName());
        }

        getFieldValue(component, "ariaLabel").ifPresent(value -> properties.put("ariaLabel", value));
        getFieldValue(component, "ariaLabelledBy").ifPresent(value -> properties.put("ariaLabelledBy", value));
        if (component.getClassName() != null && !component.getClassName().isEmpty()) {
            properties.put("className", component.getClassName());
        }
        if (component.getElement().hasAttribute("disabled")) {
            properties.put(ENABLED_PROPERTY_KEY, false);
        }

        if (component instanceof HasText hasText && !"".equals(hasText.getText())) {
            properties.put("text", hasText.getText());
        }

        if (component instanceof HasTheme hasTheme && null != hasTheme.getThemeName()) {
            properties.put("themeName", hasTheme.getThemeName());
        }
        if (component instanceof HasEnabled hasEnabled) {
            properties.put(ENABLED_PROPERTY_KEY, hasEnabled.isEnabled());
        }

        // adding src and alt attributes as properties.
        if ("com.vaadin.flow.component.html.Image".equals(component.getClass().getName())) {
            if (!properties.containsKey("src") && component.getElement().hasAttribute("src")) {
                properties.put("src", component.getElement().getAttribute("src"));
            }
            if (!properties.containsKey("alt") && component.getElement().hasAttribute("alt")) {
                properties.put("alt", component.getElement().getAttribute("alt"));
            }
        }
        if ("com.vaadin.flow.component.radiobutton.RadioButton".equals(component.getClass().getName())
                && !properties.containsKey("label")) {
            component.getChildren()
                    .filter(child -> child.getClass().getName().equals("com.vaadin.flow.component.html.Label"))
                    .findFirst().ifPresent(label -> properties.put("label", label.getElement().getTextRecursively()));
        }
        if ("com.vaadin.flow.component.richtexteditor.RichTextEditor".equals(component.getClass().getName())) {
            getFieldValue(component, "currentMode")
                    .ifPresent(currentModeValue -> properties.put("valueChangeMode", currentModeValue));
        }
        if ("com.vaadin.flow.component.avatar.Avatar".equals(component.getClass().getName())
                && component.getElement().hasAttribute("img")) {
            properties.put("img", component.getElement().getAttribute("img"));
        }
        if ("com.vaadin.flow.component.icon.Icon".equals(component.getClass().getName())
                && component.getElement().hasAttribute("icon")) {
            properties.put("icon", component.getElement().getAttribute("icon"));
        }
        if ("com.vaadin.flow.component.html.Anchor".equals(component.getClass().getName())
                && component.getElement().hasAttribute("href")) {
            properties.put("href", component.getElement().getAttribute("href"));
        }
        addHasSizeProperties(component, properties);
    }

    private void addHasSizeProperties(Component component, Map<String, Object> properties) {
        if (component instanceof HasSize hasSize) {
            if (hasSize.getMaxHeight() != null) {
                properties.put("maxHeight", hasSize.getMaxHeight());
            }
            if (hasSize.getMinHeight() != null) {
                properties.put("minHeight", hasSize.getMinHeight());
            }
            if (hasSize.getMinWidth() != null) {
                properties.put("minWidth", hasSize.getMinWidth());
            }
            if (hasSize.getMaxWidth() != null) {
                properties.put("maxWidth", hasSize.getMaxWidth());
            }
            if (hasSize.getWidth() != null) {
                properties.put("width", hasSize.getWidth());
            }
            if (hasSize.getHeight() != null) {
                properties.put("height", hasSize.getHeight());
            }
        }
    }

    private void filterOutProperties(Component component, ComponentInfo componentInfo, List<Class<?>> inheritanceChain,
            Map<String, Object> properties) {
        boolean textFieldBase = inheritanceChain.stream()
                .anyMatch(clazz -> clazz.getName().equals("com.vaadin.flow.component.textfield.TextFieldBase"));
        boolean select = "com.vaadin.flow.component.select.Select".equals(component.getClass().getName());
        boolean checkboxGroup = "com.vaadin.flow.component.checkbox.CheckboxGroup"
                .equals(component.getClass().getName());
        boolean radioButtonGroup = "com.vaadin.flow.component.radiobutton.RadioButtonGroup"
                .equals(component.getClass().getName());
        boolean richTextEditor = "com.vaadin.flow.component.richtexteditor.RichTextEditor"
                .equals(component.getClass().getName());
        boolean timePicker = "com.vaadin.flow.component.timepicker.TimePicker".equals(component.getClass().getName());
        boolean dateTimePicker = "com.vaadin.flow.component.datetimepicker.DateTimePicker"
                .equals(component.getClass().getName());
        boolean checkbox = "com.vaadin.flow.component.checkbox.Checkbox".equals(component.getClass().getName());

        if (textFieldBase || select || checkboxGroup || radioButtonGroup || timePicker || dateTimePicker) {
            // filtering out invalid properties that are generated in runtime for some
            // components
            properties.remove(VALUE_PROPERTY_KEY);
            properties.remove(INVALID_PROPERTY_KEY);
            properties.remove(ERROR_MESSAGE_PROPERTY_KEY);
            if (select) {
                properties.remove("opened");
            }
        }
        if (richTextEditor) {
            properties.remove("htmlValue");
            properties.remove(VALUE_PROPERTY_KEY);
        }
        if ("com.vaadin.flow.component.checkbox.CheckboxGroup$CheckBoxItem".equals(component.getClass().getName())) {
            properties.remove(CHECKED_PROPERTY_KEY);
            properties.remove("disabled");
        }
        if ("com.vaadin.flow.component.progressbar.ProgressBar".equals(component.getClass().getName())) {
            removeSetterIfNotPresentInSourceCode(componentInfo, properties, "max");
            removeSetterIfNotPresentInSourceCode(componentInfo, properties, "min");
            removeSetterIfNotPresentInSourceCode(componentInfo, properties, VALUE_PROPERTY_KEY);
        }
        if (properties.containsKey(ACCESSIBLE_NAME_REF)) {
            properties.put("ariaLabelledBy", properties.get(ACCESSIBLE_NAME_REF));
            properties.remove(ACCESSIBLE_NAME_REF);
        }
        if (properties.containsKey(ACCESSIBLE_NAME)) {
            properties.put("ariaLabel", properties.get(ACCESSIBLE_NAME));
            properties.remove(ACCESSIBLE_NAME);
        }
        if (checkbox) {
            if (properties.containsKey(INDETERMINATE_PROPERTY_KEY)
                    && properties.get(INDETERMINATE_PROPERTY_KEY).equals(Boolean.FALSE)) {
                removeSetterIfNotPresentInSourceCode(componentInfo, properties, INDETERMINATE_PROPERTY_KEY);
            }
            if (properties.containsKey(CHECKED_PROPERTY_KEY)
                    && properties.get(CHECKED_PROPERTY_KEY).equals(Boolean.FALSE)) {
                removeSetterIfNotPresentInSourceCode(componentInfo, properties, CHECKED_PROPERTY_KEY);
            }
            if (properties.containsKey(AUTO_FOCUS_PROPERTY_KEY)
                    && properties.get(AUTO_FOCUS_PROPERTY_KEY).equals(Boolean.FALSE)) {
                removeSetterIfNotPresentInSourceCode(componentInfo, properties, AUTO_FOCUS_PROPERTY_KEY);
            }
            removeSetterIfNotPresentInSourceCode(componentInfo, properties, VALUE_PROPERTY_KEY);
        }
        if (component instanceof HasEnabled hasEnabled && hasEnabled.isEnabled()) {
            // check if source code is enabled
            removeSetterIfNotPresentInSourceCode(componentInfo, properties, ENABLED_PROPERTY_KEY);
        }
        if (component instanceof HasValueAndElement<?, ?>) {
            if (properties.containsKey(READONLY)) {
                properties.put(READ_ONLY_CAMEL_CASE, properties.get(READONLY));
                properties.remove(READONLY);
                if (properties.get(READ_ONLY_CAMEL_CASE).equals(Boolean.FALSE)) {
                    removeSetterIfNotPresentInSourceCode(componentInfo, properties, READ_ONLY_CAMEL_CASE);
                }
            }
            if (properties.containsKey(REQUIRED_PROPERTY_KEY)) {
                properties.put(REQUIRED_INDICATOR_VISIBLE, properties.get(REQUIRED_PROPERTY_KEY));
                properties.remove(REQUIRED_PROPERTY_KEY);
                if (properties.get(REQUIRED_INDICATOR_VISIBLE).equals(Boolean.FALSE)) {
                    removeSetterIfNotPresentInSourceCode(componentInfo, properties, REQUIRED_INDICATOR_VISIBLE);
                }
            }
        }
    }

    private void removeSetterIfNotPresentInSourceCode(ComponentInfo componentInfo, Map<String, Object> properties,
            String propertyKey) {
        String setterName = JavaRewriterUtil.getSetterName(propertyKey, componentInfo.type(), false);
        boolean setterFound = JavaRewriterUtil.findMethodCalls(componentInfo).stream()
                .anyMatch(f -> f.getNameAsString().equals(setterName));
        if (!setterFound) {
            properties.remove(propertyKey);
        }
    }

    private Optional<Object> getFieldValue(Object target, String fieldName) {
        try {
            Field field = target.getClass().getDeclaredField(fieldName);
            field.setAccessible(true);
            Object o = field.get(target);
            return Optional.ofNullable(o);
        } catch (NoSuchFieldException ex) {
            getLogger().debug("Could not find field {} in class {}", fieldName, target.getClass().getName());
        } catch (IllegalAccessException e) {
            getLogger().debug("Could not access the field {}", fieldName, e);
        }
        return Optional.empty();
    }

    /**
     * Sets slot property of the given component when it's defined as prefix or
     * suffix in its parent.
     *
     * @param possiblyPrefixOrSuffixComponent
     *            Component that may be a prefix or suffix component. e.g. Icon
     *            component in Button
     * @param properties
     *            properties of the component
     */
    private void addSuffixOrPrefixAttributes(Component possiblyPrefixOrSuffixComponent,
            Map<String, Object> properties) {
        Optional<Component> parentOptional = possiblyPrefixOrSuffixComponent.getParent();
        if (parentOptional.isEmpty()) {
            return;
        }
        Component parent = parentOptional.get();
        // check if element is a prefix element of its parent
        getPrefixComponent(parent).filter(prefix -> prefix.getElement().getNode()
                .getId() == possiblyPrefixOrSuffixComponent.getElement().getNode().getId())
                .ifPresent(prefix -> properties.put("slot", "prefix"));
        getSuffixComponent(parent)
                .filter(suffixComponent -> suffixComponent.getElement().getNode()
                        .getId() == possiblyPrefixOrSuffixComponent.getElement().getNode().getId())
                .ifPresent(suffixComponent -> properties.put("slot", "suffix"));
    }

    private Optional<Component> getPrefixComponent(Component component) {
        boolean hasPrefixComponent = Arrays.stream(component.getClass().getInterfaces())
                .anyMatch(interface0 -> interface0.getName().equals("com.vaadin.flow.component.shared.HasPrefix"));
        if (!hasPrefixComponent) {
            return Optional.empty();
        }
        return getMethodValue(component, "getPrefixComponent");
    }

    private Optional<Component> getSuffixComponent(Component component) {
        boolean hasPrefixComponent = Arrays.stream(component.getClass().getInterfaces())
                .anyMatch(interface0 -> interface0.getName().equals("com.vaadin.flow.component.shared.HasSuffix"));
        if (!hasPrefixComponent) {
            return Optional.empty();
        }
        return getMethodValue(component, "getSuffixComponent");
    }

    private <T> Optional<T> getMethodValue(Object target, String methodName) {
        try {
            Method declaredMethod = target.getClass().getMethod(methodName);
            Object result = declaredMethod.invoke(target);
            return Optional.ofNullable(result).map(o -> (T) o);
        } catch (NoSuchMethodException e) {
            getLogger().debug("Could not find method {} in {}", methodName, target.getClass());
        } catch (InvocationTargetException | IllegalAccessException e) {
            getLogger().debug("Could not invoke method {} in {}", methodName, target.getClass());
        }
        return Optional.empty();
    }

    private Logger getLogger() {
        return LoggerFactory.getLogger(getClass());
    }
}
