package com.vaadin.copilot;

import java.io.IOException;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Enumeration;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Predicate;

import com.vaadin.copilot.ide.CopilotIDEPlugin;
import com.vaadin.flow.server.startup.ApplicationConfiguration;

import elemental.json.JsonArray;
import elemental.json.JsonObject;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Detects the source folders in use by the project.
 */
public class JavaSourcePathDetector {

    public record ModuleInfo(String name, Path rootPath, List<Path> javaSourcePaths, List<Path> javaTestSourcePaths,
            List<Path> resourcePaths, List<Path> testResourcePaths, Path classesFolder) {

        public Optional<Path> getOrGuessResourceFolder() {
            if (!resourcePaths.isEmpty()) {
                return Optional.of(resourcePaths.get(0));
            }
            if (!javaSourcePaths.isEmpty()) {
                return Optional.of(javaSourcePaths.get(0).getParent().resolve("resources"));
            }
            return Optional.empty();
        }

        public Optional<Path> getOrGuessTestFolder() {
            if (!javaTestSourcePaths.isEmpty()) {
                return Optional.of(javaTestSourcePaths.get(0));
            }
            if (!javaSourcePaths.isEmpty()) {
                return Optional.of(javaSourcePaths.get(0).getParent().getParent().resolve("test").resolve("java"));
            }
            return Optional.empty();
        }
    }

    public record ProjectPaths(Path basedir, List<ModuleInfo> modules, List<Path> allSourcePaths,
            List<Path> allResourcePaths) {
    }

    private JavaSourcePathDetector() {
        // Utils only
    }

    /**
     * Detects the source folders in use by the project.
     *
     * @param applicationConfiguration
     *            the application configuration
     * @return the source folders in use by the project
     */
    public static ProjectPaths detectProjectPaths(ApplicationConfiguration applicationConfiguration) {
        return detectSourceFoldersUsingIDEPlugin()
                .orElseGet(() -> detectSingleModuleSourceFolders(applicationConfiguration));
    }

    private static Optional<ProjectPaths> detectSourceFoldersUsingIDEPlugin() {
        CopilotIDEPlugin plugin = CopilotIDEPlugin.getInstance();
        if (!plugin.isActive()) {
            return Optional.empty();
        }

        try {
            JsonObject sourcePathResponse = plugin.getModulePaths();
            JsonObject project = sourcePathResponse.getObject("project");
            String projectBasePath = project.getString("basePath");
            JsonArray modules = project.getArray("modules");
            List<ModuleInfo> moduleInfos = new ArrayList<>();
            for (int i = 0; i < modules.length(); i++) {
                JsonObject module = modules.getObject(i);
                String name = module.getString("name");
                List<Path> contentRoots = stringArrayToPathList(module, "contentRoots");
                List<Path> javaSourcePaths = stringArrayToPathList(module, "javaSourcePaths").stream()
                        .filter(filterSources()).toList();
                List<Path> javaTestSourcePaths = stringArrayToPathList(module, "javaTestSourcePaths").stream()
                        .filter(filterSources()).toList();
                List<Path> resourcePaths = stringArrayToPathList(module, "resourcePaths");
                List<Path> testResourcePaths = stringArrayToPathList(module, "testResourcePaths");
                // A parent POM is a module but has no output path
                Path outputPath = module.hasKey("outputPath") ? Path.of(module.getString("outputPath")) : null;

                if (isInvalidModule(contentRoots, javaSourcePaths, javaTestSourcePaths, resourcePaths,
                        testResourcePaths)) {
                    getLogger().debug("Ignoring invalid module " + name + " without any paths");
                    continue;
                }
                Path contentRoot;
                if (contentRoots.size() == 1) {
                    contentRoot = contentRoots.get(0);
                } else if (contentRoots.isEmpty()) {
                    contentRoot = null;
                    getLogger().warn("No content root found for module {} with Java source paths {}", name,
                            javaSourcePaths);
                } else {
                    getLogger().warn("Multiple content roots ({}) found for module {} with Java source paths {}",
                            contentRoots, name, javaSourcePaths);
                    contentRoot = contentRoots.get(0);
                }
                moduleInfos.add(new ModuleInfo(name, contentRoot, javaSourcePaths, javaTestSourcePaths, resourcePaths,
                        testResourcePaths, outputPath));
            }

            return Optional.of(filterAndFindRoot(projectBasePath, moduleInfos));
        } catch (CopilotIDEPlugin.UnsupportedOperationByPluginException e) {
            return Optional.empty();
        }
    }

    private static boolean isInvalidModule(List<Path> contentRoots, List<Path> javaSourcePaths,
            List<Path> javaTestSourcePaths, List<Path> resourcePaths, List<Path> testResourcePaths) {
        // IntelliJ sometimes reports modules that used to be in the project but are not
        // anymore
        return contentRoots.isEmpty() && javaSourcePaths.isEmpty() && javaTestSourcePaths.isEmpty()
                && resourcePaths.isEmpty() && testResourcePaths.isEmpty();
    }

    private static Predicate<Path> filterSources() {
        return path -> !path.endsWith("generated-sources/annotations");
    }

    private static ProjectPaths filterAndFindRoot(String projectBasePath, List<ModuleInfo> moduleInfos) {
        try {
            // We should be able to find the build folder from the classpath, e.g.
            // some/app/target/classes
            // If we do, we filter out irrelevant modules from the IDE project, i.e. folders
            // that are not actually deployed but still exist in the same IDE project (such
            // as the Hilla test project in Copilot when running the Flow test project)
            List<Path> classpathRootFolders = getClasspathModuleRootFolders();
            if (!classpathRootFolders.isEmpty()) {
                moduleInfos = moduleInfos.stream()
                        .filter(moduleInfo -> isInAnyFolder(classpathRootFolders, moduleInfo.javaSourcePaths()))
                        .toList();
            }

            // Figure out a common base dir for the whole project
        } catch (IOException | URISyntaxException e) {
            getLogger().error("Error filtering source folders using classpath", e);
        }

        Path projectBaseDir = getCommonAncestor(
                moduleInfos.stream()
                        .map(module -> getCommonAncestor(module.javaSourcePaths(), module.javaTestSourcePaths(),
                                module.resourcePaths(), module.testResourcePaths(), List.of(module.rootPath())))
                        .toList());
        Path projectBaseDirFromIde = Path.of(projectBasePath);
        if (!projectBaseDir.equals(projectBaseDirFromIde)) {
            // This is here mostly so users can tell us in which case this fails and which
            // is the correct one and why
            getLogger().warn(
                    "Calculated project directory ({}) and directory reported by IDE ({}) do not match. Using {} and hoping for the best",
                    projectBaseDir, projectBaseDirFromIde, projectBaseDirFromIde);
            projectBaseDir = projectBaseDirFromIde;
        }
        return createProjectPaths(projectBaseDir, moduleInfos);
    }

    private static ProjectPaths createProjectPaths(Path projectBaseDir, List<ModuleInfo> modules) {
        List<Path> allSourcePaths = new ArrayList<>();
        allSourcePaths.addAll(
                modules.stream().flatMap((ModuleInfo moduleInfo) -> moduleInfo.javaSourcePaths().stream()).toList());
        allSourcePaths.addAll(modules.stream()
                .flatMap((ModuleInfo moduleInfo) -> moduleInfo.javaTestSourcePaths().stream()).toList());

        List<Path> allResourcePaths = new ArrayList<>();
        allResourcePaths.addAll(
                modules.stream().flatMap((ModuleInfo moduleInfo) -> moduleInfo.resourcePaths().stream()).toList());
        allResourcePaths.addAll(
                modules.stream().flatMap((ModuleInfo moduleInfo) -> moduleInfo.testResourcePaths().stream()).toList());

        return new ProjectPaths(projectBaseDir, modules, allSourcePaths, allResourcePaths);
    }

    @SafeVarargs
    private static Path getCommonAncestor(List<Path>... folders) {
        List<Path> allPaths = Arrays.stream(folders).flatMap(Collection::stream).toList();
        Optional<Path> projectRoot = Util.findCommonAncestor(allPaths);
        if (projectRoot.isEmpty()) {
            throw new IllegalStateException("Unable to deduce project folder using source paths: " + allPaths);
        }
        return projectRoot.get();
    }

    private static List<Path> getClasspathModuleRootFolders() throws IOException, URISyntaxException {
        return getClasspathClassesFolders().stream().map(JavaSourcePathDetector::getProjectFolderFromClasspath)
                .filter(Objects::nonNull).toList();
    }

    private static Path getProjectFolderFromClasspath(Path classesFolder) {
        return classesFolder.endsWith(Path.of("target", "classes")) ? classesFolder.getParent().getParent() : null;
    }

    private static boolean isInAnyFolder(List<Path> folders, List<Path> targets) {
        return folders.stream().anyMatch(folder -> targets.stream()
                .anyMatch(target -> ProjectManager.isFileInside(target.toFile(), folder.toFile())));
    }

    private static Logger getLogger() {
        return LoggerFactory.getLogger(JavaSourcePathDetector.class);
    }

    private static List<Path> getClasspathClassesFolders() throws IOException, URISyntaxException {
        Enumeration<URL> resources = JavaSourcePathDetector.class.getClassLoader().getResources(".");

        List<Path> paths = new ArrayList<>();
        while (resources.hasMoreElements()) {
            URL url = resources.nextElement();
            if (url.getProtocol().equals("file")) {
                Path path = Path.of(url.toURI());
                paths.add(path);
            }
        }
        return paths;
    }

    private static List<Path> stringArrayToPathList(JsonObject jsonObject, String key) {
        JsonArray pathsJson = jsonObject.getArray(key);
        List<Path> paths = new ArrayList<>();
        for (int i = 0; i < pathsJson.length(); i++) {
            paths.add(Path.of(pathsJson.getString(i)));
        }
        return paths;
    }

    static ProjectPaths detectSingleModuleSourceFolders(ApplicationConfiguration applicationConfiguration) {
        Path javaSourcePath = applicationConfiguration.getJavaSourceFolder().toPath();
        Path javaTestSourcePath = Util.replaceFolderInPath(javaSourcePath, "main", "test", "src");

        Path javaResourcePath = applicationConfiguration.getJavaResourceFolder().toPath();
        Path javaTestResourcePath = Util.replaceFolderInPath(javaResourcePath, "main", "test", "src");

        Path kotlinSourcePath = Util.replaceFolderInPath(javaSourcePath, "java", "kotlin", "main");
        Path kotlinTestSourcePath = Util.replaceFolderInPath(kotlinSourcePath, "main", "test", "src");

        Path rootPath = applicationConfiguration.getProjectFolder().toPath();

        // IntelliJ reports Kotlin source folders as Java source folders, so we don't
        // separate them out here either
        List<Path> sourcePaths = new ArrayList<>();
        sourcePaths.addAll(existingList(javaSourcePath));
        sourcePaths.addAll(existingList(kotlinSourcePath));
        List<Path> testSourcePaths = new ArrayList<>();
        testSourcePaths.addAll(existingList(javaTestSourcePath));
        testSourcePaths.addAll(existingList(kotlinTestSourcePath));

        ModuleInfo module = new ModuleInfo(rootPath.getFileName().toString(), rootPath, sourcePaths, testSourcePaths,
                existingList(javaResourcePath), existingList(javaTestResourcePath),
                getSingleModuleClassesFolder(rootPath, applicationConfiguration.getBuildFolder()));
        return createProjectPaths(module.rootPath(), List.of(module));
    }

    private static Path getSingleModuleClassesFolder(Path moduleRoot, String buildFolder) {
        Path buildFolderPath = Path.of(buildFolder);
        if (!buildFolderPath.isAbsolute()) {
            buildFolderPath = moduleRoot.resolve(buildFolderPath);
        }
        return buildFolderPath.resolve("classes");
    }

    private static List<Path> existingList(Path path) {
        if (path == null || !Files.exists(path)) {
            return Collections.emptyList();
        }
        return Collections.singletonList(path);
    }

}
