package com.vaadin.copilot.plugins.themeeditor;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.vaadin.base.devserver.DevToolsInterface;
import com.vaadin.base.devserver.DevToolsMessageHandler;
import com.vaadin.copilot.plugins.themeeditor.handlers.ComponentMetadataHandler;
import com.vaadin.copilot.plugins.themeeditor.handlers.HistoryHandler;
import com.vaadin.copilot.plugins.themeeditor.handlers.LoadPreviewHandler;
import com.vaadin.copilot.plugins.themeeditor.handlers.LoadRulesHandler;
import com.vaadin.copilot.plugins.themeeditor.handlers.LocalClassNameHandler;
import com.vaadin.copilot.plugins.themeeditor.handlers.OpenCssHandler;
import com.vaadin.copilot.plugins.themeeditor.handlers.RulesHandler;
import com.vaadin.copilot.plugins.themeeditor.messages.BaseResponse;
import com.vaadin.copilot.plugins.themeeditor.messages.ErrorResponse;
import com.vaadin.copilot.plugins.themeeditor.messages.StateMessage;
import com.vaadin.copilot.plugins.themeeditor.utils.HasSourceModifier;
import com.vaadin.copilot.plugins.themeeditor.utils.HasThemeModifier;
import com.vaadin.copilot.plugins.themeeditor.utils.MessageHandler;
import com.vaadin.copilot.plugins.themeeditor.utils.ThemeEditorException;
import com.vaadin.copilot.plugins.themeeditor.utils.ThemeEditorHistory;
import com.vaadin.flow.server.VaadinContext;
import com.vaadin.flow.server.VaadinService;
import elemental.json.Json;
import elemental.json.JsonObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.HashSet;
import java.util.Optional;
import java.util.Set;

/**
 * Handler for ThemeEditor debug window communication messages. Responsible for
 * preparing data for
 * {@link com.vaadin.copilot.plugins.themeeditor.ThemeModifier} and
 * {@link JavaSourceModifier}.
 */
public class ThemeEditorMessageHandler
        implements HasSourceModifier, HasThemeModifier, DevToolsMessageHandler {

    private final ObjectMapper objectMapper = new ObjectMapper();

    private JavaSourceModifier sourceModifier;

    private ThemeModifier themeModifier;

    private final Set<MessageHandler> handlers = new HashSet<>();

    public ThemeEditorMessageHandler() {

    }

    public boolean isEnabled() {
        return getThemeModifier().isEnabled();
    }

    public StateMessage getStateMessage() {
        return new StateMessage(
                getThemeModifier().getState().name().toLowerCase());
    }

    @Override
    public JavaSourceModifier getSourceModifier() {
        return sourceModifier;
    }

    @Override
    public ThemeModifier getThemeModifier() {
        return themeModifier;
    }

    /**
     * Checks if given command can be handled by ThemeEditor.
     *
     * @param command
     *            command to be verified if supported
     * @param data
     *            data object to be verified if is of proper structure
     * @return true if it can be handled, false otherwise
     */
    public boolean canHandle(String command, JsonObject data) {
        return command != null && data != null && data.hasKey("requestId")
                && getHandler(command).isPresent();
    }

    /**
     * Handles debug message command and performs given action.
     *
     * @param command
     *            Command name
     * @param data
     *            Command data
     * @return response in form of JsonObject
     */
    public BaseResponse handleDebugMessageData(String command,
            JsonObject data) {
        assert canHandle(command, data);
        String requestId = data.getString("requestId");
        Integer uiId = (int) data.getNumber("uiId");
        ThemeEditorHistory history = ThemeEditorHistory.forUi(uiId);
        try {
            MessageHandler.ExecuteAndUndo executeAndUndo = getHandler(command)
                    .get().handle(data);
            executeAndUndo.undoCommand()
                    .ifPresent(undo -> history.put(requestId, executeAndUndo));
            BaseResponse response = executeAndUndo.executeCommand().execute();
            response.setRequestId(requestId);
            return response;
        } catch (ThemeEditorException ex) {
            getLogger().error(ex.getMessage(), ex);
            return new ErrorResponse(requestId, ex.getMessage());
        }
    }

    private Optional<MessageHandler> getHandler(String command) {
        return handlers.stream().filter(h -> h.getCommandName().equals(command))
                .findFirst();
    }

    private static Logger getLogger() {
        return LoggerFactory
                .getLogger(ThemeEditorMessageHandler.class.getName());
    }

    protected void registerHandlers() {
        this.handlers.add(new ComponentMetadataHandler(this));
        this.handlers.add(new RulesHandler(this));
        this.handlers.add(new LocalClassNameHandler(this, this));
        this.handlers.add(new HistoryHandler());
        this.handlers.add(new LoadRulesHandler(this));
        this.handlers.add(new LoadPreviewHandler(this));
        this.handlers.add(new OpenCssHandler(this));
    }

    @Override
    public void handleConnect(DevToolsInterface devToolsInterface) {
        VaadinContext context = VaadinService.getCurrent().getContext();
        this.sourceModifier = new JavaSourceModifier(context);
        this.themeModifier = new ThemeModifier(context);
        if (isEnabled()) {
            devToolsInterface.send(ThemeEditorCommand.STATE.getValue(),
                    toJsonObject(getStateMessage()));
            registerHandlers();
        }
    }

    @Override
    public boolean handleMessage(String command, JsonObject data,
            DevToolsInterface devToolsInterface) {
        if (isEnabled() && !canHandle(command, data)) {
            return false;
        }

        BaseResponse resultData = handleDebugMessageData(command, data);
        devToolsInterface.send(ThemeEditorCommand.RESPONSE.getValue(),
                toJsonObject(resultData));

        return true;
    }

    private JsonObject toJsonObject(Object object) {
        try {
            String resultAsString = objectMapper.writeValueAsString(object);
            return Json.parse(resultAsString);
        } catch (JsonProcessingException e) {
            throw new RuntimeException(e);
        }
    }
}
