package com.vaadin.copilot;

import com.vaadin.base.devserver.DevToolsInterface;
import com.vaadin.base.devserver.DevToolsMessageHandler;
import com.vaadin.base.devserver.stats.DevModeUsageStatistics;
import com.vaadin.flow.server.VaadinService;
import com.vaadin.flow.server.startup.ApplicationConfiguration;
import elemental.json.JsonObject;

import java.io.IOException;
import java.nio.file.Path;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.List;

public class Copilot implements DevToolsMessageHandler {

    static final String PREFIX = "copilot-";

    ProjectManager projectManager;

    private List<CopilotCommand> commands;

    @Override
    public void handleConnect(DevToolsInterface devToolsInterface) {
        try {
            ApplicationConfiguration applicationConfiguration = ApplicationConfiguration
                    .get(VaadinService.getCurrent().getContext());
            Path projectRoot = applicationConfiguration.getProjectFolder()
                    .toPath();
            projectManager = new ProjectManager(projectRoot);
            setupCommands(devToolsInterface, applicationConfiguration);
            devToolsInterface.send(PREFIX + "init", null);
        } catch (IOException e) {
            throw new RuntimeException("Unable to create project manager", e);
        }
    }

    private void setupCommands(DevToolsInterface devToolsInterface,
            ApplicationConfiguration applicationConfiguration) {
        commands = List.of(new OpenComponentInIDE(applicationConfiguration),
                new ProjectFileHandler(projectManager),
                new AICommandHandler(projectManager), new UserInfoHandler());
    }

    @Override
    public boolean handleMessage(String command, JsonObject data,
            DevToolsInterface devToolsInterface) {
        if (!command.startsWith(PREFIX)) {
            return false;
        }
        if (command.equals("copilot-track-event")) {
            String event = PREFIX + data.getString("event");
            if (data.hasKey("value")) {
                DevModeUsageStatistics.collectEvent(event,
                        data.getNumber("value"));
            } else {
                DevModeUsageStatistics.collectEvent(event);
            }
            return true;
        }
        Instant start = Instant.now();
        try {

            command = command.substring(PREFIX.length());

            for (CopilotCommand copilotCommand : commands) {
                if (copilotCommand.handleMessage(command, data,
                        devToolsInterface)) {
                    return true;
                }
            }
        } finally {
            long time = start.until(Instant.now(), ChronoUnit.MILLIS);
            DevModeUsageStatistics.collectEvent("copilot-command-" + command,
                    time);
        }
        return false;

    }
}
