package com.utsmannn.pocketdb

import android.content.SharedPreferences
import com.google.gson.Gson
import com.google.gson.JsonSyntaxException
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.channels.awaitClose
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.channelFlow
import kotlinx.coroutines.flow.flowOn
import kotlinx.coroutines.flow.map
import java.lang.reflect.Type
import kotlin.coroutines.CoroutineContext

class PocketValue(
    private val key: String,
    private val pref: SharedPreferences,
    private val gson: Gson,
    private val secret: String
) {
    private fun SharedPreferences.observeKey(
        key: String,
        default: String,
        dispatcher: CoroutineContext = Dispatchers.Default
    ): Flow<String?> {
        val flow: Flow<String?> = channelFlow {
            logi("offer --> ${getStringItem(key, default)}")
            offer(getStringItem(key, default.encrypt(secret))?.decrypt(secret))

            val listener = SharedPreferences.OnSharedPreferenceChangeListener { _, k ->
                if (key == k) {
                    logi("changed --> ${getStringItem(key, default)}")
                    offer(getStringItem(key, default.encrypt(secret))?.decrypt(secret))
                }
            }

            registerOnSharedPreferenceChangeListener(listener)
            awaitClose { unregisterOnSharedPreferenceChangeListener(listener) }
        }
        return flow.flowOn(dispatcher)
    }

    // ------ ROW ------ //

    internal fun <T> insert(data: T) =
        pref.edit().putString(key, data.convertToString(gson).encrypt(secret)).apply()

    internal fun <T> select(default: T, type: Type): Flow<T?> {
        return pref.observeKey(key, default.convertToString(gson)).map {
            return@map try {
                logi("mmmm -> $it")
                gson.fromJson<T>(it, type)
            } catch (e: JsonSyntaxException) {
                throw IllegalArgumentException("Wrong key of existing data, key of data: $key")
            }
        }
    }

    internal fun <T> selectSingle(default: T, type: Type): T? {
        val defaultString = default.convertToString(gson)
        val stringRaw = pref.getString(key, defaultString)
        return try {
            logi("raw => $stringRaw")
            gson.fromJson<T>(stringRaw, type)
        } catch (e: JsonSyntaxException) {
            throw IllegalArgumentException("Wrong key of existing data, key of data: $key")
        }
    }

    // ------ END OF ROW ------ //


    // ------ COLLECTION ------ //

    internal fun <T> selectSingleCollection(type: Type): Collection<T> {
        val default = emptyList<T>()
        val defaultString = gson.toJson(default, type).encrypt(secret)
        val stringRaw = pref.getString(key, defaultString)?.decrypt(secret)
        return if (stringRaw == null) {
            return emptyList()
        } else {
            gson.fromJson(stringRaw, type)
        }
    }

    internal fun <T> selectCollection(default: Collection<T>?, type: Type): Flow<Collection<T>> {
        val defaultString = gson.toJson(default, type)
        return pref.observeKey(key, defaultString).map {
            logi("key is -> $key")
            logi("observing list ----> $it")
            return@map try {
                gson.fromJson<Collection<T>>(it, type)
            } catch (e: JsonSyntaxException) {
                logi("taee")
                e.printStackTrace()
                throw IllegalArgumentException("Wrong key of existing data, key of data: $key")
            }
        }
    }

    internal fun <T> insertCollectionItem(data: T, type: Type) {
        val currentCollection = selectSingleCollection<T>(type)
        val newCollection = currentCollection.toMutableList().apply {
            add(data)
        }
        val stringCollection = gson.toJson(newCollection, type).encrypt(secret)
        pref.edit().putString(key, stringCollection).apply()
        logi("inserting -> $currentCollection")
    }

    internal fun <T> insertCollections(data: Collection<T>, type: Type) {
        val currentCollection = selectSingleCollection<T>(type)
        val newCollection = currentCollection.toMutableList().apply {
            addAll(data)
        }
        val stringCollection = gson.toJson(newCollection, type).encrypt(secret)
        pref.edit().putString(key, stringCollection).apply()
        logi("inserting -> $currentCollection")
    }

    // ------ END OF COLLECTION ------ //

}