package com.unity3d.services.core.network.core

import com.unity3d.ads.core.data.model.UnityAdsNetworkException
import com.unity3d.services.core.domain.ISDKDispatchers
import com.unity3d.services.core.network.model.HttpRequest
import com.unity3d.services.core.network.model.HttpResponse
import com.unity3d.services.core.network.model.RequestType
import kotlinx.coroutines.asExecutor
import kotlinx.coroutines.runBlocking
import kotlinx.coroutines.suspendCancellableCoroutine
import org.chromium.net.CronetEngine
import org.chromium.net.CronetException
import org.chromium.net.NetworkException
import org.chromium.net.UploadDataProviders
import org.chromium.net.UrlRequest
import org.chromium.net.UrlResponseInfo
import kotlin.coroutines.resume
import kotlin.coroutines.resumeWithException

class CronetClient(
    private val engine: CronetEngine,
    private val dispatchers: ISDKDispatchers,
): HttpClient {

    override fun executeBlocking(request: HttpRequest): HttpResponse = runBlocking(dispatchers.io) {
        execute(request)
    }

    override suspend fun execute(request: HttpRequest): HttpResponse {
        return suspendCancellableCoroutine {
                cont ->
            val cronetRequest = engine.newUrlRequestBuilder(buildUrl(request) , object: UnityAdsUrlRequestCallback() {
                override fun onSucceeded(
                    request: UrlRequest,
                    info: UrlResponseInfo,
                    bodyBytes: ByteArray
                ) {
                    cont.resume(
                        HttpResponse(
                            statusCode = info.httpStatusCode,
                            headers = info.allHeaders,
                            urlString = info.url,
                            body = bodyBytes,
                            protocol = info.negotiatedProtocol,
                            client = NETWORK_CLIENT_CRONET
                        )
                    )
                }

                override fun onFailed(
                    request: UrlRequest?,
                    info: UrlResponseInfo?,
                    error: CronetException?
                ) {
                    val cronetCode = (error as? NetworkException)?.cronetInternalErrorCode
                    val exception = UnityAdsNetworkException(
                        message = MSG_CONNECTION_FAILED,
                        code = info?.httpStatusCode,
                        url = info?.url,
                        protocol = info?.negotiatedProtocol,
                        cronetCode = cronetCode,
                        client = NETWORK_CLIENT_CRONET
                    )
                    cont.resumeWithException(exception)
                }
            }, dispatchers.io.asExecutor())


            request.headers.forEach { (key, value) ->
                value.forEach {
                    cronetRequest.addHeader(key, it)
                }
            }
            if (request.method == RequestType.POST) {
                val body = when (request.body) {
                    is ByteArray -> request.body
                    is String -> request.body.toByteArray()
                    else -> ByteArray(0)
                }
                cronetRequest.setUploadDataProvider(UploadDataProviders.create(body), dispatchers.io.asExecutor())
            }
            cronetRequest
                .setHttpMethod(request.method.toString())
                .setPriority(UrlRequest.Builder.REQUEST_PRIORITY_HIGHEST)
                .build().start()
        }
    }

    private fun buildUrl(request: HttpRequest): String {
        return "${request.baseURL.trim('/')}/${request.path.trim('/')}".removeSuffix("/")
    }

    companion object {
        private const val MSG_CONNECTION_FAILED = "Network request failed"
        private const val NETWORK_CLIENT_CRONET = "cronet"
    }
}