package com.unity3d.services

import com.unity3d.ads.core.configuration.AlternativeFlowReader
import com.unity3d.ads.core.domain.SendDiagnosticEvent
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON
import com.unity3d.services.core.log.DeviceLog
import com.unity3d.services.core.request.metrics.Metric
import com.unity3d.services.core.request.metrics.SDKMetricsSender
import kotlinx.coroutines.CoroutineDispatcher
import kotlinx.coroutines.CoroutineExceptionHandler
import kotlinx.coroutines.CoroutineName
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.launch
import kotlinx.coroutines.plus
import java.lang.IllegalStateException
import java.lang.RuntimeException
import kotlin.coroutines.CoroutineContext

class SDKErrorHandler(
    private val ioDispatcher: CoroutineDispatcher,
    private val alternativeFlowReader: AlternativeFlowReader,
    private val sendDiagnosticEvent: SendDiagnosticEvent,
    private val sdkMetricsSender: SDKMetricsSender
) : CoroutineExceptionHandler {
    private val scope = CoroutineScope(ioDispatcher) + CoroutineName("SDKErrorHandler")

    override val key = CoroutineExceptionHandler.Key

    override fun handleException(context: CoroutineContext, exception: Throwable) {
        val className: String = exception.stackTrace[0]?.fileName ?: "unknown"
        val line: Int = exception.stackTrace[0]?.lineNumber ?: 0

        val name: String = when (exception) {
            is NullPointerException -> "native_exception_npe"
            is OutOfMemoryError -> "native_exception_oom"
            is IllegalStateException -> "native_exception_ise"
            is SecurityException -> "native_exception_se"
            is RuntimeException -> "native_exception_re"
            else -> "native_exception"
        }

        val isAlternativeFlowEnabled = alternativeFlowReader()
        val crashValue = "${className}_$line"
        DeviceLog.error("Unity Ads SDK encountered an exception: $crashValue")
        if (isAlternativeFlowEnabled) {
            sendDiagnostic(name, crashValue)
        } else {
            sendMetric(Metric(name, crashValue))
        }

    }

    private fun sendDiagnostic(name: String, value: String) {
        scope.launch {
            sendDiagnosticEvent(event = name, tags = mapOf(REASON to value))
        }
    }


    private fun sendMetric(metric: Metric) = sdkMetricsSender.sendMetric(metric);
}