package com.unity3d.ads.core.domain

import android.content.Context
import com.unity3d.ads.IUnityAdsShowListener
import com.unity3d.ads.UnityAds
import com.unity3d.ads.UnityAdsShowOptions
import com.unity3d.ads.core.data.model.AdObject
import com.unity3d.ads.core.data.model.OperationType
import com.unity3d.ads.core.data.model.ShowEvent
import com.unity3d.ads.core.data.model.ShowStatus
import com.unity3d.ads.core.data.repository.AdRepository
import com.unity3d.ads.core.data.repository.SessionRepository
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.OPERATION
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.SHOW_FAILURE
import com.unity3d.ads.core.domain.events.GetOperativeEventApi
import com.unity3d.ads.core.extensions.duration
import com.unity3d.ads.core.extensions.timeoutAfter
import com.unity3d.ads.core.extensions.toByteString
import com.unity3d.ads.core.extensions.toDiagnosticReason
import com.unity3d.services.core.log.DeviceLog
import gateway.v1.OperativeEventRequestOuterClass.OperativeEventErrorType
import gateway.v1.OperativeEventRequestOuterClass.OperativeEventType
import gateway.v1.operativeEventErrorData
import kotlinx.coroutines.CoroutineDispatcher
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.catch
import kotlinx.coroutines.withContext
import java.util.UUID
import java.util.concurrent.atomic.AtomicBoolean

internal class LegacyShowUseCase(
    private val dispatcher: CoroutineDispatcher,
    private val show: Show,
    private val adRepository: AdRepository,
    private val sendDiagnosticEvent: SendDiagnosticEvent,
    private val getOperativeEventApi: GetOperativeEventApi,
    private val sessionRepository: SessionRepository
) {
    private val isShowing = AtomicBoolean(false)
    private val hasStarted = MutableStateFlow(false)
    private val timeoutCancellationRequested = MutableStateFlow(false)

    suspend operator fun invoke(
        context: Context,
        placement: String,
        unityAdsShowOptions: UnityAdsShowOptions?,
        unityShowListener: IUnityAdsShowListener?
    ) {
        val startTime = System.nanoTime()
        DeviceLog.debug("Unity Ads Show Start for placement $placement")
        showStart()

        if (isShowing.getAndSet(true)) {
            showError(startTime, UnityAds.UnityAdsShowError.ALREADY_SHOWING, MESSAGE_ALREADY_SHOWING, placement, unityShowListener)
            return
        }

        val opportunityId = unityAdsShowOptions?.let { getOpportunityId(it) }
        if (opportunityId == null) {
            showError(startTime, UnityAds.UnityAdsShowError.INVALID_ARGUMENT, MESSAGE_OPPORTUNITY_ID, placement, unityShowListener)
            return
        }

        val opportunityIdByteString =  UUID.fromString(opportunityId).toByteString()
        val adObject = adRepository.getAd(opportunityIdByteString)
        if (adObject == null) {
            showError(startTime, UnityAds.UnityAdsShowError.INTERNAL_ERROR, MESSAGE_NO_AD_OBJECT + opportunityId, placement, unityShowListener)
            return
        }

        if (sessionRepository.nativeConfiguration.featureFlags.opportunityIdPlacementValidation && adObject.placementId != placement) {
            showError(startTime, UnityAds.UnityAdsShowError.INVALID_ARGUMENT, MSG_OPPORTUNITY_AND_PLACEMENT_NOT_MATCHING, placement, unityShowListener)
            return
        }

        val useTimeout = true // sessionRepository.nativeConfiguration.featureFlags.showTimeoutEnabled
        val timeoutMillis = sessionRepository.nativeConfiguration.adOperations.showTimeoutMs.toLong()

        show(context, adObject)
            .timeoutAfter(timeoutMillis) {
                if (hasStarted.value || timeoutCancellationRequested.value) return@timeoutAfter
                sendOperativeError(OperativeEventErrorType.OPERATIVE_EVENT_ERROR_TYPE_TIMEOUT, MESSAGE_OPT_TIMEOUT, adObject)
                showTimeout(adObject, useTimeout, startTime, placement, unityShowListener)
            }
            .catch {
                showError(startTime, UnityAds.UnityAdsShowError.INTERNAL_ERROR, it.message ?: "", placement, unityShowListener)
            }
            .collect {
                when (it) {
                    is ShowEvent.Started -> showStarted(startTime, placement, unityShowListener)
                    is ShowEvent.Clicked -> showClicked(startTime, placement, unityShowListener)
                    is ShowEvent.Completed -> showCompleted(startTime, placement, it.status, unityShowListener)
                    is ShowEvent.Error ->  {
                        sendOperativeError(OperativeEventErrorType.OPERATIVE_EVENT_ERROR_TYPE_UNSPECIFIED, it.message, adObject)
                        showError(startTime, UnityAds.UnityAdsShowError.INTERNAL_ERROR, it.message, placement, unityShowListener)
                    }
                    is ShowEvent.CancelTimeout -> cancelTimeout(startTime)
                }
            }

        isShowing.set(false)
    }

    private suspend fun sendOperativeError(operativeEvent: OperativeEventErrorType, operativeMessage: String, adObject: AdObject) {
        val errorData = operativeEventErrorData {
            errorType = operativeEvent
            message = operativeMessage
        }
        getOperativeEventApi(
            operativeEventType = OperativeEventType.OPERATIVE_EVENT_TYPE_SHOW_ERROR,
            adObject = adObject,
            additionalEventData = errorData.toByteString()
        )
    }

    private suspend fun showTimeout(
        adObject: AdObject,
        useTimeout: Boolean,
        startTime: Long,
        placement: String,
        unityShowListener: IUnityAdsShowListener?
    ) {
        if (useTimeout) {
            show.terminate(adObject)
            showError(startTime, UnityAds.UnityAdsShowError.TIMEOUT,MESSAGE_TIMEOUT + placement, placement, unityShowListener)
        }
    }

    private suspend fun cancelTimeout(startTime: Long) {
        timeoutCancellationRequested.value = true
        sendDiagnosticEvent(event = SendDiagnosticEvent.SHOW_CANCEL_TIMEOUT, value = startTime.duration())
    }

    private suspend fun showStarted(startTime: Long,placement: String, unityShowListener: IUnityAdsShowListener?) {
        DeviceLog.debug("Unity Ads Show WV Start for placement $placement")
        hasStarted.value = true
        sendDiagnosticEvent(event = SendDiagnosticEvent.SHOW_WV_STARTED, value = startTime.duration())
        withContext(dispatcher) {
            unityShowListener?.onUnityAdsShowStart(placement)
        }
    }

    private suspend fun showClicked(startTime: Long,placement: String, unityShowListener: IUnityAdsShowListener?) {
        DeviceLog.debug("Unity Ads Show Clicked for placement $placement")
        sendDiagnosticEvent(event = SendDiagnosticEvent.SHOW_CLICKED, value = startTime.duration())
        withContext(dispatcher) {
            unityShowListener?.onUnityAdsShowClick(placement)
        }
    }

    private fun getOpportunityId(unityAdsShowOptions: UnityAdsShowOptions): String? {
        val objectId = unityAdsShowOptions.data?.opt(KEY_OBJECT_ID)?.toString()
        return try {
            UUID.fromString(objectId).toString()
        } catch (t: Throwable) {
            null
        }
    }

    private suspend fun showStart() = sendDiagnosticEvent(event = SendDiagnosticEvent.SHOW_STARTED)

    private suspend fun showCompleted(
        startTime: Long,
        placement: String,
        status: ShowStatus,
        unityShowListener: IUnityAdsShowListener?,
    ) {
        DeviceLog.debug("Unity Ads Show Completed for placement $placement")
        sendDiagnosticEvent(event = SendDiagnosticEvent.SHOW_SUCCESS, value = startTime.duration())
        withContext(dispatcher) {
            unityShowListener?.onUnityAdsShowComplete(placement, status.toUnityAdsShowCompletionState())
        }
    }

    private suspend fun showError(
        startTime: Long,
        reason: UnityAds.UnityAdsShowError,
        message: String = "",
        placement: String,
        unityShowListener: IUnityAdsShowListener?
    ) {
        DeviceLog.debug("Unity Ads Show Failed for placement $placement")
        sendDiagnosticEvent(event = SHOW_FAILURE, value = startTime.duration(), tags = getTags(reason))
        withContext(dispatcher) {
            unityShowListener?.onUnityAdsShowFailure(placement, reason, message)
        }
        isShowing.set(false)
    }

    private fun getTags(reason: UnityAds.UnityAdsShowError): Map<String, String> {
        return mapOf(
            OPERATION to OperationType.SHOW.toString(),
            REASON to reason.toDiagnosticReason(),
        )
    }

    companion object {
        const val KEY_OBJECT_ID = "objectId"
        const val MESSAGE_OPPORTUNITY_ID = "No valid opportunity id provided"
        const val MESSAGE_NO_AD_OBJECT = "No ad object found for opportunity id: "
        const val MESSAGE_ALREADY_SHOWING = "Can't show a new ad unit when ad unit is already open"
        const val MESSAGE_TIMEOUT = "[UnityAds] Timeout while trying to show "
        const val MSG_OPPORTUNITY_AND_PLACEMENT_NOT_MATCHING = "[UnityAds] Object ID and Placement ID provided does not match previously loaded ad"
        const val MESSAGE_OPT_TIMEOUT = "timeout"
    }
}