package com.unity3d.ads.core.domain

import com.unity3d.ads.core.data.model.InitializationState
import com.unity3d.ads.core.data.repository.SessionRepository
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.INIT_FAILURE
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.INIT_STARTED
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.INIT_SUCCESS
import com.unity3d.ads.UnityAds.UnityAdsInitializationError.INTERNAL_ERROR
import com.unity3d.ads.core.data.manager.SDKPropertiesManager
import com.unity3d.ads.core.data.repository.DiagnosticEventRepository
import com.unity3d.ads.core.domain.events.EventObservers
import com.unity3d.ads.core.extensions.duration
import com.unity3d.ads.gatewayclient.GatewayClient
import com.unity3d.ads.core.data.manager.StorageManager
import com.unity3d.ads.core.data.model.OperationType
import com.unity3d.ads.core.data.model.UnityAdsNetworkException
import com.unity3d.ads.core.data.repository.DeviceInfoRepository
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.OPERATION
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON_GATEWAY
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON_NETWORK
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON_TIMEOUT
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON_UNKNOWN
import com.unity3d.services.UnityAdsConstants
import com.unity3d.services.core.extensions.toUnityMessage
import com.unity3d.services.core.log.DeviceLog
import kotlinx.coroutines.CoroutineDispatcher
import kotlinx.coroutines.TimeoutCancellationException
import kotlinx.coroutines.withContext
import kotlinx.coroutines.withTimeout

internal class InitializeAndroidBoldSDK(
    private val defaultDispatcher: CoroutineDispatcher,
    private val getInitializeRequest: GetInitializationRequest,
    private val getRequestPolicy: GetRequestPolicy,
    private val handleGatewayInitializationResponse: HandleGatewayInitializationResponse,
    private val gatewayClient: GatewayClient,
    private val sessionRepository: SessionRepository,
    private val eventObservers: EventObservers,
    private val triggerInitializeListener: TriggerInitializeListener,
    private val sendDiagnosticEvent: SendDiagnosticEvent,
    private val diagnosticEventRepository: DiagnosticEventRepository,
    private val deviceInfoRepository: DeviceInfoRepository,
    private val storageManager: StorageManager,
    private val sdkPropertiesManager: SDKPropertiesManager,
) : InitializeBoldSDK {
    override suspend fun invoke() = withContext(defaultDispatcher) {
        val startTime = System.nanoTime()

        try {
            withTimeout(UnityAdsConstants.Timeout.INIT_TIMEOUT_MS) {
                initializationStart()
                checkCanInitialize()
                val initializationRequest = getInitializeRequest()
                val requestPolicy = getRequestPolicy()
                val response = gatewayClient.request(request = initializationRequest, requestPolicy = requestPolicy, operationType = OperationType.INITIALIZATION)
                handleGatewayInitializationResponse(response.payload.initializationResponse)
            }
        } catch (e: Exception) {
            initializationFailure(startTime, e)
            return@withContext
        }

        initializationSuccess(startTime)
    }

    private fun checkCanInitialize() {
        check(sessionRepository.shouldInitialize) { MSG_GATEWAY_DENIED }
        check(deviceInfoRepository.hasInternet) { MSG_NO_INTERNET }
    }

    private suspend fun initializationStart() {
        DeviceLog.debug("Unity Ads Initialization Start")
        sendDiagnosticEvent(event = INIT_STARTED)
        sessionRepository.initializationState = InitializationState.INITIALIZING
        eventObservers()
    }

    private suspend fun initializationSuccess(startTime: Long) {
        DeviceLog.debug("Unity Ads Initialization Success")
        sendDiagnosticEvent(event = INIT_SUCCESS, value = startTime.duration())
        storageManager.hasInitialized()
        triggerInitializeListener.success()
        sessionRepository.initializationState = InitializationState.INITIALIZED_SUCCESSFULLY
        sdkPropertiesManager.setInitialized(true)
        setupDiagnosticEvents()
    }

    private suspend fun initializationFailure(startTime: Long, e: Exception) {
        DeviceLog.debug("Unity Ads Initialization Failure: ${e.message}")
        sendDiagnosticEvent(event = INIT_FAILURE, value = startTime.duration(), tags = getTags(e))
        triggerInitializeListener.error(INTERNAL_ERROR, e.message.toUnityMessage())
        sessionRepository.initializationState = InitializationState.INITIALIZED_FAILED
        sdkPropertiesManager.setInitialized(false)
    }

    private fun getTags(e: Exception): Map<String, String> {
        val tags = mutableMapOf(
            OPERATION to OperationType.INITIALIZATION.toString(),
            REASON to REASON_UNKNOWN
        )
        when (e) {
            is TimeoutCancellationException -> tags[REASON] = REASON_TIMEOUT
            is UnityAdsNetworkException -> tags[REASON] = REASON_NETWORK
        }
        when (e.message) {
            MSG_NO_INTERNET -> tags[REASON] = REASON_NETWORK
            MSG_GATEWAY_DENIED ->tags[REASON] = REASON_GATEWAY
        }
        return tags
    }

    private fun setupDiagnosticEvents() {
        val config = sessionRepository.nativeConfiguration.diagnosticEvents
        diagnosticEventRepository.configure(config)
    }

    companion object {
        const val MSG_NO_INTERNET = "No internet connection"
        const val MSG_GATEWAY_DENIED = "Gateway communication failure"
    }
}
