package com.unity3d.ads.core.domain

import android.content.Context
import com.google.protobuf.ByteString
import com.unity3d.ads.core.data.model.LoadResult
import com.unity3d.ads.core.data.repository.AdRepository
import gateway.v1.AdResponseOuterClass.AdResponse
import com.unity3d.ads.core.data.repository.SessionRepository
import com.unity3d.ads.gatewayclient.GatewayClient
import com.unity3d.ads.UnityAds.UnityAdsLoadError
import com.unity3d.ads.core.data.model.LoadResult.Companion.MSG_AD_OBJECT
import com.unity3d.ads.core.data.model.OperationType
import gateway.v1.copy
import kotlinx.coroutines.CoroutineDispatcher
import kotlinx.coroutines.withContext

internal class AndroidLoad(
    private val defaultDispatcher: CoroutineDispatcher,
    private val getAdRequest: GetAdRequest,
    private val getRequestPolicy: GetRequestPolicy,
    private val handleGatewayAdResponse: HandleGatewayAdResponse,
    private val sessionRepository: SessionRepository,
    private val gatewayClient: GatewayClient,
    private val adRepository: AdRepository,
) : Load {
    override suspend fun invoke(
        context: Context,
        placement: String,
        opportunityId: ByteString,
        loadAdResponse: AdResponse
    ): LoadResult = withContext(defaultDispatcher) {
        if (!sessionRepository.isSdkInitialized) {
            return@withContext LoadResult.Failure(error = UnityAdsLoadError.INITIALIZE_FAILED)
        }

        val response = if (loadAdResponse.adData.isEmpty) {
            sessionRepository.incrementLoadRequestCount()
            val loadAdRequest = getAdRequest(placement, opportunityId)
            val requestPolicy = getRequestPolicy()
            val gatewayAdResponse = gatewayClient.request(request = loadAdRequest, requestPolicy = requestPolicy, operationType = OperationType.LOAD)
            gatewayAdResponse.payload.adResponse
        } else {
            sessionRepository.incrementLoadRequestAdmCount()
            loadAdResponse
        }

        val adLoadResponse = handleGatewayAdResponse(opportunityId, response, context, placement)
        when (adLoadResponse) {
            is LoadResult.Success -> {
                val adObject = adRepository.getAd(opportunityId)
                if (adObject == null) {
                    LoadResult.Failure(error = UnityAdsLoadError.INTERNAL_ERROR, message = MSG_AD_OBJECT)
                } else {
                    LoadResult.Success(adObject = adObject)
                }
            }
            is LoadResult.Failure -> adLoadResponse
        }
    }
}