package com.unity3d.ads.core.domain

import android.content.Context
import android.os.Environment
import com.unity3d.ads.core.data.model.CacheDirectory
import com.unity3d.ads.core.data.model.CacheDirectoryType
import com.unity3d.services.core.log.DeviceLog
import kotlinx.coroutines.*
import kotlinx.coroutines.flow.*
import java.io.File
import kotlin.contracts.ExperimentalContracts
import kotlin.contracts.contract

@OptIn(ExperimentalContracts::class)
class AndroidGetCacheDirectoryUseCase(
    private val context: Context,
    private val cacheDirName: String,
    dispatcher: CoroutineDispatcher
) : GetCacheDirectoryUseCase {
    private val scope = CoroutineScope(dispatcher) + CoroutineName("AndroidGetCacheDirectoryUseCase")
    private val isInitialized = MutableStateFlow(false)
    private val cacheDirectory = CompletableDeferred<CacheDirectory?>()

    override suspend fun invoke(): CacheDirectory? {
        if (!isInitialized.value) {
            initialize()
        }

        return cacheDirectory.await()
    }

    private suspend fun initialize() = withContext(scope.coroutineContext) {
        isInitialized.value = true

        if (Environment.MEDIA_MOUNTED == Environment.getExternalStorageState()) {
            val externalCache = try {
                createCacheDirectory(context.externalCacheDir, cacheDirName)
            } catch (e: Exception) {
                DeviceLog.exception("Creating external cache directory failed", e)
                null
            }

            if (testCacheDirectory(externalCache)) {
                createNoMediaFile(externalCache)
                DeviceLog.debug("Unity Ads is using external cache directory: ${externalCache.absolutePath}")
                cacheDirectory.complete(CacheDirectory(externalCache, CacheDirectoryType.EXTERNAL))
                return@withContext
            }
        }

        DeviceLog.debug("External media not mounted")

        val internalCache = context.filesDir
        if (testCacheDirectory(internalCache)) {
            DeviceLog.debug("Unity Ads is using internal cache directory: ${internalCache.absolutePath}")
            cacheDirectory.complete(CacheDirectory(internalCache, CacheDirectoryType.INTERNAL))
            return@withContext
        }

        DeviceLog.error("Unity Ads failed to initialize cache directory")

        cacheDirectory.complete(null)
    }

    private fun createCacheDirectory(baseDir: File?, newDir: String): File {
        requireNotNull(baseDir) { "Base directory is null" }

        return File(baseDir, newDir).also(File::mkdirs).takeIf { it.isDirectory }
            ?: throw Exception("Could not create cache directory")
    }

    private fun testCacheDirectory(directory: File?): Boolean {
        contract { returns(true) implies (directory != null) }

        if (directory?.isDirectory != true) {
            return false
        }

        try {
            val text = "test"
            val testFile = File(directory, TEST_FILE_NAME)
            testFile.writeText(text)

            if (testFile.readText() != text) {
                DeviceLog.debug("Read content mismatch")
                return false
            }

            if (!testFile.delete()) {
                DeviceLog.debug("Failed to delete test file ${testFile.absoluteFile}")
                return false
            }
        } catch (e: Exception) {
            DeviceLog.debug("Unity Ads exception while testing cache directory ${directory.absolutePath}: ${e.message}")
            return false
        }

        return true
    }

    private fun createNoMediaFile(path: File?) {
        val noMediaFile = File(path, ".nomedia")
        try {
            val created = noMediaFile.createNewFile()
            if (created) {
                DeviceLog.debug("Successfully created .nomedia file")
            } else {
                DeviceLog.debug("Using existing .nomedia file")
            }
        } catch (e: Exception) {
            DeviceLog.exception("Failed to create .nomedia file", e)
        }
    }

    companion object {
        private const val TEST_FILE_NAME = "UnityAdsTest.txt"
    }
}