package com.unity3d.ads.core.data.datasource

import com.unity3d.ads.core.domain.privacy.FlattenerRulesUseCase
import com.unity3d.services.core.misc.JsonFlattener
import com.unity3d.services.core.misc.JsonStorage
import gateway.v1.DeveloperConsentOuterClass.DeveloperConsent
import gateway.v1.DeveloperConsentOuterClass.DeveloperConsentChoice
import gateway.v1.DeveloperConsentOuterClass.DeveloperConsentChoice.DEVELOPER_CONSENT_CHOICE_FALSE
import gateway.v1.DeveloperConsentOuterClass.DeveloperConsentChoice.DEVELOPER_CONSENT_CHOICE_TRUE
import gateway.v1.DeveloperConsentOuterClass.DeveloperConsentChoice.DEVELOPER_CONSENT_CHOICE_UNSPECIFIED
import gateway.v1.DeveloperConsentOuterClass.DeveloperConsentOption
import gateway.v1.DeveloperConsentOuterClass.DeveloperConsentType
import gateway.v1.DeveloperConsentOuterClass.DeveloperConsentType.DEVELOPER_CONSENT_TYPE_CUSTOM
import gateway.v1.DeveloperConsentOuterClass.DeveloperConsentType.DEVELOPER_CONSENT_TYPE_GDPR_CONSENT
import gateway.v1.DeveloperConsentOuterClass.DeveloperConsentType.DEVELOPER_CONSENT_TYPE_NON_BEHAVIORAL
import gateway.v1.DeveloperConsentOuterClass.DeveloperConsentType.DEVELOPER_CONSENT_TYPE_PIPL_CONSENT
import gateway.v1.DeveloperConsentOuterClass.DeveloperConsentType.DEVELOPER_CONSENT_TYPE_PRIVACY_CONSENT
import gateway.v1.DeveloperConsentOuterClass.DeveloperConsentType.DEVELOPER_CONSENT_TYPE_UNSPECIFIED
import gateway.v1.DeveloperConsentOuterClass.DeveloperConsentType.DEVELOPER_CONSENT_TYPE_USER_OVER_AGE_LIMIT
import gateway.v1.developerConsent
import gateway.v1.developerConsentOption
import org.json.JSONObject

class AndroidDeveloperConsentDataSource(
    private val flattenerRulesUseCase: FlattenerRulesUseCase,
    private val publicStorage: JsonStorage,
) : DeveloperConsentDataSource {
    override val developerConsent: DeveloperConsent
        get() = developerConsent {
            options.addAll(developerConsentList())
        }

    private fun developerConsentList(): List<DeveloperConsentOption> {
        val developerConsentList = mutableListOf<DeveloperConsentOption>()
        val data = fetchData()
        val keys = data.keys()
        while (keys.hasNext()) {
            val key = keys.next()
            val storedValue = data.get(key)
            val developerConsent = developerConsentOption {
                type = getDeveloperConsentType(key)
                if (type == DEVELOPER_CONSENT_TYPE_CUSTOM) { customType = key }
                value = getDeveloperConsentChoice(storedValue as Boolean?)
            }
            developerConsentList.add(developerConsent)
        }
        return developerConsentList
    }

    private fun fetchData(): JSONObject {
        publicStorage.data?.let {
            val flattener = JsonFlattener(publicStorage.data)
            return flattener.flattenJson(".", flattenerRulesUseCase())
        }?: return JSONObject()
    }

    private fun getDeveloperConsentType(type: String?): DeveloperConsentType {
        return when (type) {
            "user.nonBehavioral.value" -> DEVELOPER_CONSENT_TYPE_NON_BEHAVIORAL
            "pipl.consent.value" -> DEVELOPER_CONSENT_TYPE_PIPL_CONSENT
            "privacy.consent.value" -> DEVELOPER_CONSENT_TYPE_PRIVACY_CONSENT
            "gdpr.consent.value" -> DEVELOPER_CONSENT_TYPE_GDPR_CONSENT
            "privacy.useroveragelimit.value" -> DEVELOPER_CONSENT_TYPE_USER_OVER_AGE_LIMIT
            null -> DEVELOPER_CONSENT_TYPE_UNSPECIFIED
            else -> DEVELOPER_CONSENT_TYPE_CUSTOM
        }
    }

    private fun getDeveloperConsentChoice(choice: Boolean?): DeveloperConsentChoice {
        return when (choice) {
            true -> DEVELOPER_CONSENT_CHOICE_TRUE
            false -> DEVELOPER_CONSENT_CHOICE_FALSE
            else -> DEVELOPER_CONSENT_CHOICE_UNSPECIFIED
        }
    }
}