package com.unity3d.ads.adplayer

import android.util.Base64
import com.google.protobuf.ByteString
import com.unity3d.ads.core.data.model.ShowEvent
import com.unity3d.ads.core.data.model.ShowStatus
import com.unity3d.ads.core.data.model.exception.LoadException
import com.unity3d.ads.core.data.repository.DeviceInfoRepository
import com.unity3d.ads.core.data.repository.SessionRepository
import com.unity3d.ads.core.domain.ExecuteAdViewerRequest
import com.unity3d.ads.core.domain.SendDiagnosticEvent
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.BRIDGE_SEND_EVENT_FAILED
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON
import com.unity3d.ads.core.extensions.fromBase64
import com.unity3d.ads.core.extensions.toBase64
import com.unity3d.services.core.device.Storage
import com.unity3d.services.core.device.StorageEventInfo
import com.unity3d.services.core.network.mapper.toResponseHeadersMap
import com.unity3d.services.core.network.model.RequestType
import gateway.v1.AllowedPiiOuterClass.AllowedPii
import kotlinx.coroutines.CompletableDeferred
import kotlinx.coroutines.CoroutineDispatcher
import kotlinx.coroutines.CoroutineExceptionHandler
import kotlinx.coroutines.CoroutineName
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.filter
import kotlinx.coroutines.flow.launchIn
import kotlinx.coroutines.flow.map
import kotlinx.coroutines.flow.onCompletion
import kotlinx.coroutines.flow.onEach
import kotlinx.coroutines.flow.take
import kotlinx.coroutines.launch
import kotlinx.coroutines.plus
import org.json.JSONObject

private val SHOW_EVENTS = arrayOf(
    ExposedFunctionLocation.STARTED,
    ExposedFunctionLocation.CLICKED,
    ExposedFunctionLocation.COMPLETED,
    ExposedFunctionLocation.FAILED,
    ExposedFunctionLocation.CANCEL_SHOW_TIMEOUT
)

private val LOAD_EVENTS = arrayOf(
    ExposedFunctionLocation.LOAD_COMPLETE,
    ExposedFunctionLocation.LOAD_ERROR,
)

private val REQUEST_EVENTS = arrayOf(
    ExposedFunctionLocation.REQUEST_GET,
    ExposedFunctionLocation.REQUEST_POST,
    ExposedFunctionLocation.REQUEST_HEAD
)

/**
 * Class responsible for handling ad playback in a WebView.
 *
 * @property bridge The WebViewBridge used for communication with the WebView.
 * @property deviceInfoRepository The repository used for retrieving device information.
 * @property sessionRepository The repository used for managing session data.
 * @property dispatcher The CoroutineDispatcher used for running coroutines.
 */
internal class WebViewAdPlayer(
    private val bridge: WebViewBridge,
    private val deviceInfoRepository: DeviceInfoRepository,
    private val sessionRepository: SessionRepository,
    private val executeAdViewerRequest: ExecuteAdViewerRequest,
    private val dispatcher: CoroutineDispatcher,
    private val sendDiagnosticEvent: SendDiagnosticEvent,
) : AdPlayer {
    private val storageEventCallback: (StorageEventInfo) -> Unit = {
        scope.launch {
            bridge.sendEvent(OnStorageEvent(
                eventType = it.eventType,
                storageType = it.storageType,
                value = it.value
            ))
        }
    }

    private val scopeCancellationHandler = CoroutineExceptionHandler { _, _ ->
        Storage.removeStorageEventCallback(storageEventCallback)
    }

    override val scope = CoroutineScope(dispatcher) + CoroutineName("WebViewAdPlayer") + scopeCancellationHandler

    override val onShowEvent = bridge.onInvocation
        .filter { it.location in SHOW_EVENTS }
        .map {
            val event = when (it.location) {
                ExposedFunctionLocation.STARTED -> ShowEvent.Started
                ExposedFunctionLocation.CLICKED -> ShowEvent.Clicked
                ExposedFunctionLocation.COMPLETED -> ShowEvent.Completed(
                    when (it.parameters.first() as? String) {
                        "COMPLETED" -> ShowStatus.COMPLETED
                        "SKIPPED" -> ShowStatus.SKIPPED
                        else -> ShowStatus.ERROR
                    }
                )
                ExposedFunctionLocation.FAILED -> {
                    val jsonObject = it.parameters.first() as JSONObject
                    val errorCode = jsonObject.optInt("code")
                    val errorMessage = jsonObject.optString("message")
                    ShowEvent.Error(errorMessage, errorCode)
                }
                ExposedFunctionLocation.CANCEL_SHOW_TIMEOUT -> ShowEvent.CancelTimeout
                else -> throw IllegalStateException("Unexpected location: ${it.location}")
            }
            it.handle()
            event
        }

    override val loadEvent = CompletableDeferred<Unit>()

    override val updateCampaignState: Flow<Pair<ByteString, Int>> = bridge.onInvocation
        .filter { it.location == ExposedFunctionLocation.UPDATE_CAMPAIGN_STATE }
        .map {
            it.handle()
            val campaignJSON = it.parameters.first() as JSONObject
            val data = campaignJSON.optString("data")
            val dataByteString = data.fromBase64()
            val dataVersion = campaignJSON.optInt("dataVersion")
            dataByteString to dataVersion
        }

    override val markCampaignStateAsShown: Flow<Unit> = bridge.onInvocation
        .filter { it.location == ExposedFunctionLocation.MARK_CAMPAIGN_STATE_SHOWN }
        .map {
            it.handle()
        }

    private val onBroadcastEvents = bridge.onInvocation
        .filter { it.location == ExposedFunctionLocation.BROADCAST_EVENT }
        .map {
            it.handle()
            it.parameters.first() as JSONObject
        }

    val onRequestEvents = bridge.onInvocation
        .filter { it.location in REQUEST_EVENTS }
        .map {
            it.handle()

            val id = it.parameters.first() as String
            val url = it.parameters.getOrNull(1) as String?

            val type = when (it.location) {
                ExposedFunctionLocation.REQUEST_GET -> RequestType.GET
                ExposedFunctionLocation.REQUEST_POST -> RequestType.POST
                ExposedFunctionLocation.REQUEST_HEAD -> RequestType.HEAD
                else -> throw IllegalStateException("Unexpected location: ${it.location}")
            }

            try {
                val response = executeAdViewerRequest(type, it.parameters)
                val body = when (val responseBody = response.body) {
                    is String -> responseBody
                    is ByteArray -> String(responseBody, Charsets.UTF_8)
                    else -> null
                }
                val bridgeResponse = listOf(
                    id,
                    response.urlString,
                    body,
                    response.statusCode,
                    response.headers.toResponseHeadersMap()
                )
                bridge.sendEvent(OnWebRequestComplete(bridgeResponse))
            } catch (e: Exception) {
                val bridgeResponse = listOf(id, url, e.message ?: "")
                bridge.sendEvent(OnWebRequestFailed(bridgeResponse))
            }
        }

    init {
        Storage.addStorageEventCallback(storageEventCallback)

        // Listen to calls of `broadcastEvent`, and emit them to the broadcast event channel.
        onBroadcastEvents.onEach(AdPlayer.broadcastEventChannel::emit).launchIn(scope)

        // Listen to calls of `request`, and emit them to the request event channel.
        onRequestEvents.launchIn(scope)

        // Listen to the broadcast event channel, and send the events to the webview.
        AdPlayer.broadcastEventChannel.onEach(::onBroadcastEvent).launchIn(scope)

        // Listen for load event, and complete the deferred when it happens.
        bridge.onInvocation
            .filter { it.location in LOAD_EVENTS }
            .take(1)
            .onEach {
                it.handle()
                if (it.location == ExposedFunctionLocation.LOAD_ERROR) {
                    val jsonObject = it.parameters.first() as JSONObject
                    val errorCode = jsonObject.optInt("code")
                    val errorMessage = jsonObject.optString("message")
                    throw LoadException(errorCode, "WebView load error: $errorMessage")
                }
            }
            .onCompletion {
                if (it != null) {
                    loadEvent.completeExceptionally(it)
                } else {
                    loadEvent.complete(Unit)
                }
            }
            .launchIn(scope)
    }

    // region AdViewer API

    override suspend fun requestShow() {
        val dynamicDeviceInfo = deviceInfoRepository.dynamicDeviceInfo

        val showOptions = JSONObject().also {
            it.put("orientation", deviceInfoRepository.orientation)
            it.put("connectionType", deviceInfoRepository.connectionTypeStr)
            it.put("isMuted", deviceInfoRepository.ringerMode != 2)
            it.put("volume", dynamicDeviceInfo.android.volume)
            it.put("privacy", sessionRepository.getPrivacy().toBase64())
            it.put("privacyFsm", sessionRepository.getPrivacyFsm().toBase64())
            // TODO: Discuss with AdViewer and iOS how we should really pass around allowedPii structure.
            it.put("allowedPii", deviceInfoRepository.allowedPii.value.toByteString().toBase64())
        }

        bridge.request("webview", "show", showOptions)
    }

    // endregion AdViewer API

    // region Events

    private suspend fun sendEvent(getEvent: () -> WebViewEvent) {
        try {
            loadEvent.await()
            val event = getEvent()
            bridge.sendEvent(event)
        } catch (e: Throwable) {
            sendDiagnosticEvent(BRIDGE_SEND_EVENT_FAILED, tags = mapOf(REASON to e.message.toString()))
        }
    }

    override suspend fun sendMuteChange(isMuted: Boolean) = sendEvent {
        OnMuteChangeEvent(isMuted)
    }

    override suspend fun sendVisibilityChange(isVisible: Boolean) = sendEvent {
        OnVisibilityChangeEvent(isVisible)
    }

    override suspend fun sendVolumeChange(volume: Double) = sendEvent {
        OnVolumeChangeEvent(volume)
    }

    override suspend fun sendUserConsentChange(value: ByteString) = sendEvent {
        OnUserConsentChangeEvent(Base64.encodeToString(value.toByteArray(), Base64.NO_WRAP))
    }

    override suspend fun sendPrivacyFsmChange(value: ByteString) = sendEvent {
        OnPrivacyFsmChangeEvent(Base64.encodeToString(value.toByteArray(), Base64.NO_WRAP))
    }

    override suspend fun onBroadcastEvent(event: JSONObject) = sendEvent {
        val eventType = event.getString("eventType")
        val data = event.optString("data")

        OnBroadcastEvent(eventType, data)
    }

    override suspend fun onAllowedPiiChange(value: AllowedPii) = sendEvent {
        OnAllowedPiiChangeEvent(value)
    }

    // endregion Events
}
