package com.unity3d.ads.adplayer

import android.os.Build
import android.webkit.RenderProcessGoneDetail
import android.webkit.WebResourceRequest
import android.webkit.WebResourceResponse
import android.webkit.WebView
import androidx.annotation.RequiresApi
import androidx.webkit.WebResourceErrorCompat
import androidx.webkit.WebViewClientCompat
import androidx.webkit.WebViewFeature
import androidx.webkit.WebViewFeature.isFeatureSupported
import com.unity3d.ads.core.data.model.ErrorReason
import com.unity3d.ads.core.data.model.webResourceToErrorReason
import com.unity3d.ads.core.extensions.removeViewFromParent
import kotlinx.coroutines.*
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.update

data class WebViewClientError(
    val url: String? = null,
    val reason: ErrorReason,
    val statusCode: Int? = null
)

class AndroidWebViewClient : WebViewClientCompat() {
    private val mainScope = MainScope()
    private val loadErrors = MutableStateFlow(listOf<WebViewClientError>())
    private val _onLoadFinished = CompletableDeferred<List<WebViewClientError>>()
    val onLoadFinished: Deferred<List<WebViewClientError>> = _onLoadFinished

    override fun onPageFinished(view: WebView, url: String) {
        validatePage(url)
        super.onPageFinished(view, url)
        with(_onLoadFinished) { complete(loadErrors.value) }
    }

    @RequiresApi(Build.VERSION_CODES.LOLLIPOP)
    override fun onReceivedError(
        view: WebView,
        request: WebResourceRequest,
        error: WebResourceErrorCompat
    ) {
        super.onReceivedError(view, request, error)

        val reason = if (isFeatureSupported(WebViewFeature.WEB_RESOURCE_ERROR_GET_CODE)) {
            error.errorCode.webResourceToErrorReason()
        } else {
            ErrorReason.REASON_UNKNOWN
        }

        loadErrors.update {
            it + WebViewClientError(request.url.toString(), reason)
        }
    }
    override fun onReceivedHttpError(
        view: WebView,
        request: WebResourceRequest,
        errorResponse: WebResourceResponse
    ) {
        super.onReceivedHttpError(view, request, errorResponse)

        val error = if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
            WebViewClientError(
                url = request.url.toString(),
                reason = ErrorReason.REASON_WEB_ERROR_RECEIVED_HTTP,
                statusCode = errorResponse.statusCode
            )
        } else {
            WebViewClientError(reason = ErrorReason.REASON_WEB_ERROR_RECEIVED_HTTP)
        }

        loadErrors.update { it + error }
    }

    override fun shouldInterceptRequest(
        view: WebView,
        request: WebResourceRequest
    ): WebResourceResponse? {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP && request.url?.lastPathSegment == "favicon.ico") {
            return WebResourceResponse("image/png", null, null)
        }

        return super.shouldInterceptRequest(view, request)
    }

    override fun onRenderProcessGone(view: WebView?, detail: RenderProcessGoneDetail?): Boolean {

        mainScope.launch {
            view?.removeViewFromParent()
            view?.destroy()
            // todo: if during load and show -> rebuild webview to not compromise show
            // todo: add diagnostic for render process gone during load/show
        }

        loadErrors.update {
            it + WebViewClientError(view?.url.toString(), ErrorReason.REASON_WEBVIEW_RENDER_PROCESS_GONE)
        }

        with(_onLoadFinished) { complete(loadErrors.value) }

        return true // returning false will kill the app
    }

    private fun validatePage(url: String) {
        if (url == BLANK_PAGE) {
            loadErrors.update {
                it + WebViewClientError(url, ErrorReason.REASON_WEB_BLANK)
            }
        }
    }

    companion object {
        const val BLANK_PAGE = "about:blank"
    }
}