package com.unity3d.ads.core.data.datasource

import android.annotation.SuppressLint
import android.content.Context
import android.content.Intent
import android.content.IntentFilter
import android.database.ContentObserver
import android.media.AudioManager
import android.media.AudioManager.*
import android.net.ConnectivityManager
import android.net.NetworkCapabilities
import android.net.NetworkInfo
import android.os.BatteryManager
import android.os.Build
import android.os.Environment
import android.os.SystemClock
import android.provider.Settings
import android.telephony.TelephonyManager
import androidx.annotation.RequiresApi
import androidx.annotation.VisibleForTesting
import com.unity3d.ads.core.utils.getMemoryValueFromString
import com.unity3d.services.core.device.AdvertisingId
import com.unity3d.services.core.device.Device.MemoryInfoType
import com.unity3d.services.core.device.OpenAdvertisingId
import com.unity3d.services.core.log.DeviceLog
import gatewayprotocol.v1.DynamicDeviceInfoKt.android
import gatewayprotocol.v1.DynamicDeviceInfoOuterClass
import gatewayprotocol.v1.DynamicDeviceInfoOuterClass.ConnectionType
import gatewayprotocol.v1.NetworkCapabilityTransportsOuterClass.NetworkCapabilityTransports
import gatewayprotocol.v1.dynamicDeviceInfo
import kotlinx.coroutines.channels.awaitClose
import kotlinx.coroutines.channels.trySendBlocking
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.callbackFlow
import kotlinx.coroutines.flow.update
import java.io.File
import java.io.RandomAccessFile
import java.util.*
import java.util.regex.Pattern
import kotlin.math.roundToInt

/**
 * Class responsible for retrieving all device data that is expected to change during a session.
 *
 * Will be fetched on demand.
 */
@Suppress("DEPRECATION")
class AndroidDynamicDeviceInfoDataSource(
    val context: Context,
    private val lifecycleDataSource: LifecycleDataSource
) : DynamicDeviceInfoDataSource {
    private val reportedWarning = MutableStateFlow(emptyMap<String, Boolean>())

    /**
     * Fill out all the fields required for the [DynamicDeviceInfoOuterClass] protobuf
     */
    override fun fetch() = dynamicDeviceInfo {
        language = getLanguage()
        networkOperator = getNetworkOperator()
        networkOperatorName = getNetworkOperatorName()
        freeDiskSpace = getUsableSpace(context.getExternalFilesDir(null))
        freeRamMemory = getFreeMemory()
        wiredHeadset = isWiredHeadsetOn()
        timeZone = getTimeZone()
        timeZoneOffset = getTimeZoneOffset()
        limitedTracking = isLimitAdTrackingEnabled()
        limitedOpenAdTracking = isLimitOpenAdTrackingEnabled()
        batteryLevel = getBatteryLevel()
        batteryStatus = getBatteryStatus()
        connectionType = getConnectionType()
        android = fetchAndroidDynamicDeviceInfo()
        appActive = isAppActive()
        screenWidth = getScreenWidth()
        screenHeight = getScreenHeight()
    }

    private fun fetchAndroidDynamicDeviceInfo(): DynamicDeviceInfoOuterClass.DynamicDeviceInfo.Android = android {
        networkConnected = isActiveNetworkConnected()
        networkType = getNetworkType()
        networkMetered = getNetworkMetered()
        telephonyManagerNetworkType = getNetworkType()
        adbEnabled = isAdbEnabled()
        usbConnected = isUSBConnected()
        volume = getStreamVolume(STREAM_MUSIC)
        maxVolume = getStreamMaxVolume(STREAM_MUSIC)
        deviceElapsedRealtime = getElapsedRealtime()
        deviceUpTime = getUptime()
        airplaneMode = getAirplaneMode()
        chargingType = getChargingType()
        stayOnWhilePluggedIn = getStayOnWhilePluggedIn()
        sdCardPresent = getIsSdCardPresent()
        networkCapabilityTransports = getNetworkCapabilityTransports()
    }

    /**
     * @return the default language of the device
     */
    private fun getLanguage(): String = Locale.getDefault().toString()

    /**
     * @return the default timezone display name
     */
    private fun getTimeZone(): String {
        return try {
            TimeZone.getDefault().getDisplayName(false, TimeZone.SHORT, Locale.US)
        } catch (assertionError: AssertionError) {
            // This can occur on some flavours of Android 8.1 and is a workaround for an OS bug
            DeviceLog.error("Could not read timeZone information: %s", assertionError.message)
            ""
        }
    }

    /**
     * @return the offset of the default timezone
     */
    private fun getTimeZoneOffset(): Long =
        TimeZone.getDefault().getOffset(System.currentTimeMillis()).toLong() / 1000

    /**
     * @return if the device has a wifi connection
     */
    private fun isUsingWifi(): Boolean {
        val connectivity: ConnectivityManager = getConnectivityManager() ?: return false
        val telephony: TelephonyManager? = getTelephonyManager()

        // Skip if no connection, or background data disabled
        val info: NetworkInfo? = connectivity.activeNetworkInfo
        if (info == null || !connectivity.backgroundDataSetting || !info.isConnected || telephony == null
        ) {
            return false
        }
        val netType = info.type
        return netType == ConnectivityManager.TYPE_WIFI && info.isConnected
    }

    /**
     * @return if connection type is [ConnectionType.CONNECTION_TYPE_WIFI], [ConnectionType.CONNECTION_TYPE_CELLULAR] or [ConnectionType.CONNECTION_TYPE_NONE]
     */
    private fun getConnectionType(): ConnectionType {
        return if (isUsingWifi()) {
            ConnectionType.CONNECTION_TYPE_WIFI
        } else if (isActiveNetworkConnected()) {
            ConnectionType.CONNECTION_TYPE_CELLULAR
        } else {
            ConnectionType.CONNECTION_TYPE_UNSPECIFIED
        }
    }

    /**
     * https://developer.android.com/reference/android/telephony/TelephonyManager#getNetworkType()
     *
     * @return the current data network type.
     */
    @SuppressLint("MissingPermission")
    @Deprecated("This method was deprecated in API level 30. Use getDataNetworkType()")
    private fun getNetworkType(): Int {
        val telephony: TelephonyManager? = getTelephonyManager()

        try {
            return telephony?.networkType ?: -1
        } catch (ex: SecurityException) {
            if (reportedWarning.value["getNetworkType"] != true) {
                reportedWarning.update { it + ("getNetworkType" to true) }
                DeviceLog.warning("Unity Ads was not able to get current network type due to missing permission")
            }
        }

        return -1
    }

    private fun getNetworkMetered(): Boolean {
        val connectivity: ConnectivityManager? = getConnectivityManager()
        return connectivity?.isActiveNetworkMetered == true
    }

    /**
     * https://developer.android.com/reference/android/telephony/TelephonyManager#getNetworkOperator()
     *
     * Only when user is registered to a network. Result may be unreliable on CDMA networks.
     *
     * @return the numeric name (MCC+MNC) of current registered operator.
     */
    private fun getNetworkOperator(): String {
        val telephony: TelephonyManager? = getTelephonyManager()
        return telephony?.networkOperator ?: ""
    }

    /**
     * https://developer.android.com/reference/android/telephony/TelephonyManager#getNetworkOperatorName()
     *
     * Only when user is registered to a network. Result may be unreliable on CDMA networks.
     *
     * @return the alphabetic name of current registered operator.
     */
    private fun getNetworkOperatorName(): String {
        val telephony: TelephonyManager? = getTelephonyManager()
        return telephony?.networkOperatorName ?: ""
    }

    /**
     * https://developer.android.com/reference/android/telephony/TelephonyManager#getNetworkCountryIso()
     *
     * Result may be unreliable on CDMA networks
     *
     * @return the ISO-3166-1 alpha-2 country code equivalent of the MCC (Mobile Country Code) of the current registered operator or the cell nearby, if available.
     */
    fun getNetworkCountryISO(): String {
        val telephony: TelephonyManager? = getTelephonyManager()
        return telephony?.networkCountryIso ?: ""
    }

    /**
     * https://developer.android.com/reference/android/util/DisplayMetrics#widthPixels
     *
     * @return the screen width from [android.util.DisplayMetrics.widthPixels] or 0 if unavailable
     */
    private fun getScreenWidth(): Int {
        return context.resources?.displayMetrics?.widthPixels ?: -1
    }

    /**
     * https://developer.android.com/reference/android/util/DisplayMetrics#heightPixels
     *
     * @return the screen height from [android.util.DisplayMetrics.heightPixels] or 0 if unavailable
     */
    private fun getScreenHeight(): Int {
        return context.resources?.displayMetrics?.heightPixels ?: -1
    }

    /**
     * https://developer.android.com/reference/android/net/NetworkInfo#isConnected()
     *
     * Indicates whether network connectivity exists and it is possible to establish connections and pass data via the currently active default data network
     *
     * @return true if current active network connectivity exists, false otherwise.
     */
    private fun isActiveNetworkConnected(): Boolean {
        val connectivity: ConnectivityManager? = getConnectivityManager()
        val activeNetwork = connectivity?.activeNetworkInfo
        return activeNetwork != null && activeNetwork.isConnected
    }

    /**
     * https://developer.android.com/reference/android/media/AudioManager#isWiredHeadsetOn()
     *
     * Checks whether a wired headset is connected or not.
     * This is not a valid indication that audio playback is actually over the wired headset as audio routing depends on other conditions.
     *
     * @return true if a wired headset is connected. false if otherwise
     */
    private fun isWiredHeadsetOn(): Boolean {
        val audio: AudioManager? = getAudioManager()
        return audio?.isWiredHeadsetOn == true
    }

    /**
     * https://developer.android.com/reference/android/media/AudioManager#getRingerMode()
     *
     * RINGER_MODE_SILENT = 0
     * RINGER_MODE_VIBRATE = 1
     * RINGER_MODE_NORMAL = 2
     *
     * @return the current ringtone mode.
     */
    override fun getRingerMode(): Int {
        val audio: AudioManager? = getAudioManager()
        return audio?.ringerMode ?: -2
    }

    /**
     * https://developer.android.com/reference/android/media/AudioManager#getStreamVolume(int)
     *
     * @param streamType stream type whose volume index is returned.
     * @return the current volume index for a particular stream, -2 if not available
     */
    private fun getStreamVolume(streamType: Int): Double {
        val audio: AudioManager? = getAudioManager()
        return (audio?.getStreamVolume(streamType) ?: -2).toDouble()
    }

    /**
     * https://developer.android.com/reference/android/media/AudioManager#getStreamMaxVolume(int)
     *
     * @param streamType stream type whose maximum volume index is returned.
     * @return the maximum volume index for a particular stream, -2 if not available
     */
    fun getStreamMaxVolume(streamType: Int): Double {
        val audio: AudioManager? = getAudioManager()
        return (audio?.getStreamMaxVolume(streamType) ?: -2).toDouble()
    }

    /**
     * https://developer.android.com/reference/android/provider/Settings.System#SCREEN_BRIGHTNESS
     *
     *@return the screen backlight brightness between 0 and 255.
     */
    fun getScreenBrightness(): Int {
        return Settings.System.getInt(
            context.contentResolver,
            Settings.System.SCREEN_BRIGHTNESS,
            -1
        )
    }

    /**
     * https://developer.android.com/reference/java/io/File#getFreeSpace()
     *
     * @param file the [File] on which to calculate the free space available
     * @return the number of unallocated kilobytes in the partition named by this abstract path name.
     */
    @Deprecated("Legacy method, migrated from to .getUsableSpace()")
    private fun getFreeSpace(file: File?): Long {
        return if (file != null && file.exists()) {
            (file.freeSpace / 1024).toFloat().roundToInt().toLong()
        } else -1
    }

    /**
     * https://developer.android.com/reference/java/io/File#getUsableSpace()
     *
     * @param file the [File] on which to calculate the free space available
     * @return the number of unallocated kilobytes in the partition named by this abstract path name.
     */
    private fun getUsableSpace(file: File?): Long {
        return if (file != null && file.exists()) {
            (file.usableSpace / 1024).toFloat().roundToInt().toLong()
        } else -1
    }

    /**
     * https://developer.android.com/reference/android/os/BatteryManager#EXTRA_LEVEL
     * https://developer.android.com/reference/android/os/BatteryManager#EXTRA_SCALE
     *
     * EXTRA_LEVEL integer field containing the current battery level, from 0 to EXTRA_SCALE.
     * EXTRA_SCALE integer containing the maximum battery level.
     *
     * @return the battery level divided by the scale, 0 if not available
     */
    private fun getBatteryLevel(): Double {
        val i = context.registerReceiver(
            null, IntentFilter(
                Intent.ACTION_BATTERY_CHANGED
            )
        )
        if (i != null) {
            val level = i.getIntExtra(BatteryManager.EXTRA_LEVEL, 0)
            val scale = i.getIntExtra(BatteryManager.EXTRA_SCALE, 0)
            return level / scale.toDouble()
        }
        return -1.0
    }


    /**
     * https://developer.android.com/reference/android/content/Intent#ACTION_BATTERY_CHANGED
     *
     * BATTERY_STATUS_UNKNOWN = 1;
     * BATTERY_STATUS_CHARGING = 2;
     * BATTERY_STATUS_DISCHARGING = 3;
     * BATTERY_STATUS_NOT_CHARGING = 4;
     * BATTERY_STATUS_FULL = 5;
     *
     * @return the BATTERY_STATUS value or 0 if not available
     */
    private fun getBatteryStatus(): Int {
        val i = context.registerReceiver(
            null, IntentFilter(
                Intent.ACTION_BATTERY_CHANGED
            )
        )
        if (i != null) {
            return i.getIntExtra(BatteryManager.EXTRA_STATUS, 0)
        }
        return -1
    }

    /**
     * @return the total amount of physical RAM, in kilobytes.
     */
    fun getTotalMemory(): Long {
        return getMemoryInfo(MemoryInfoType.TOTAL_MEMORY)
    }

    /**
     * @return the amount of physical RAM, in kilobytes, left unused by the system
     */
    fun getFreeMemory(): Long {
        return getMemoryInfo(MemoryInfoType.FREE_MEMORY)
    }

    /**
     * @param infoType the [MemoryInfoType] for the amount to be returned
     * @return amount of memory depending on [MemoryInfoType] passed in
     */
    private fun getMemoryInfo(infoType: MemoryInfoType): Long {
        val lineNumber: Int = when (infoType) {
            MemoryInfoType.TOTAL_MEMORY -> 1
            MemoryInfoType.FREE_MEMORY -> 2
            else -> -1
        }
        var line: String? = null
        RandomAccessFile(DIRECTORY_MEM_INFO, DIRECTORY_MODE_READ).use {
            for (i in 0 until lineNumber) {
                line = it.readLine()
            }
        }
        return getMemoryValueFromString(line)
    }

    /**
     * https://developer.android.com/reference/android/provider/Settings.Global#ADB_ENABLED
     *
     * @return whether ADB over USB is enabled.
     */
    private fun isAdbEnabled(): Boolean {
        return adbStatus()
    }

    /**
     * https://developer.android.com/reference/android/provider/Settings.Global#ADB_ENABLED
     *
     * @return whether ADB over USB is enabled.
     */
    private fun adbStatus(): Boolean {
        var status: Boolean? = null
        try {
            status = 1 == Settings.Global.getInt(
                context.contentResolver,
                Settings.Global.ADB_ENABLED,
                0
            )
        } catch (e: Exception) {
            DeviceLog.exception("Problems fetching adb enabled status", e)
        }
        return status ?: false
    }

    /**
     * https://android.googlesource.com/platform/frameworks/base/+/master/core/java/android/hardware/usb/UsbManager.java
     *
     * @return whether USB is connected or disconnected
     */
    private fun isUSBConnected(): Boolean {
        val intent = context
            .registerReceiver(null, IntentFilter(INTENT_USB_STATE))
        if (intent != null) {
            return intent.getBooleanExtra(USB_EXTRA_CONNECTED, false)
        }
        return false
    }


    /**
     * https://developer.android.com/reference/android/os/SystemClock#uptimeMillis()
     *
     * @return milliseconds since boot, not counting time spent in deep sleep.
     */
    private fun getUptime(): Long {
        return SystemClock.uptimeMillis()
    }

    /**
     * https://developer.android.com/reference/android/os/SystemClock#elapsedRealtime()
     *
     * @return milliseconds since boot, including time spent in sleep.
     */
    private fun getElapsedRealtime(): Long {
        return SystemClock.elapsedRealtime()
    }


    /**
     * https://man7.org/linux/man-pages/man5/proc.5.html
     *
     * @return status information about the process.
     */
    fun getProcessInfo(): Map<String, String> {
        val retData = HashMap<String, String>()
        RandomAccessFile(DIRECTORY_PROCESS_INFO, DIRECTORY_MODE_READ).use {
            val statContent = it.readLine()
            retData[KEY_STAT_CONTENT] = statContent
        }
        return retData
    }

    /**
     * https://developers.google.com/android/reference/com/google/android/gms/ads/identifier/AdvertisingIdClient.Info#isLimitAdTrackingEnabled()
     *
     * When the returned value is true, the returned value of getId() will always be 00000000-0000-0000-0000-000000000000 starting with Android 12.
     *
     * @return whether the user has limit ad tracking enabled or not.
     */
    private fun isLimitAdTrackingEnabled(): Boolean {
        return AdvertisingId.getLimitedAdTracking()
    }

    private fun isLimitOpenAdTrackingEnabled(): Boolean {
        return OpenAdvertisingId.getLimitedOpenAdTracking()
    }

    /**
     * @return if the application is in the foreground of background
     */
    private fun isAppActive() = lifecycleDataSource.appIsForeground()

    /**
     * @return a current timestamp, epoch in milliseconds
     */
    private fun getEventTimeStamp(): Long = System.currentTimeMillis() / 1000

    /**
     * https://developer.android.com/reference/android/telephony/TelephonyManager
     *
     * @return the apps [TelephonyManager]
     */
    private fun getTelephonyManager(): TelephonyManager? =
        context.getSystemService(Context.TELEPHONY_SERVICE) as? TelephonyManager

    /**
     * https://developer.android.com/reference/android/net/ConnectivityManager
     *
     * @return the apps [ConnectivityManager]
     */
    private fun getConnectivityManager(): ConnectivityManager? =
        context.getSystemService(Context.CONNECTIVITY_SERVICE) as? ConnectivityManager

    /**
     * https://developer.android.com/reference/android/media/AudioManager
     *
     * @return the apps [AudioManager]
     */
    private fun getAudioManager(): AudioManager? =
        context.getSystemService(Context.AUDIO_SERVICE) as? AudioManager

    /**
     * Does best attempt to determine if the device has internet connection
     * Note: not 100% accurate
     *
     * @see [hasInternetConnection] for API < 23
     * @see [hasInternetConnectionM] for API >= 23
     *
     * @return true if we estimate the device has internet connection, false otherwise
     */
    override fun hasInternet(): Boolean {
        return if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
            hasInternetConnectionM()
        } else {
            hasInternetConnection()
        }
    }

    /**
     * https://developer.android.com/reference/android/net/NetworkInfo#isConnected()
     *
     * Indicates whether network connectivity exists and it is possible to establish connections and pass data.
     */
    private fun hasInternetConnection(): Boolean {
        val connectivityManager = getConnectivityManager() ?: return false
        val networkInfo = connectivityManager.activeNetworkInfo
        return networkInfo?.isConnected == true
    }

    /**
     * https://developer.android.com/reference/android/net/NetworkCapabilities#NET_CAPABILITY_VALIDATED
     *
     * Indicates that connectivity on this network was successfully validated.
     * For example, for a network with NET_CAPABILITY_INTERNET, it means that Internet connectivity was successfully detected.
     */
    @RequiresApi(api = Build.VERSION_CODES.M)
    private fun hasInternetConnectionM(): Boolean {
        val connectivityManager = getConnectivityManager() ?: return false
        val network = connectivityManager.activeNetwork
        val capabilities = connectivityManager.getNetworkCapabilities(network)
        return capabilities != null
            && capabilities.hasCapability(NetworkCapabilities.NET_CAPABILITY_INTERNET)
            && capabilities.hasCapability(NetworkCapabilities.NET_CAPABILITY_VALIDATED)
    }

    override val volumeSettingsChange = callbackFlow {
        var currentVolume = getStreamVolume(STREAM_MUSIC)
        trySendBlocking(VolumeSettingsChange.VolumeChange(currentVolume))

        var currentRingerMode = getRingerMode()
        trySendBlocking(VolumeSettingsChange.MuteChange(currentRingerMode == RINGER_MODE_SILENT))

        val contentObserver = object : ContentObserver(null) {
            override fun onChange(selfChange: Boolean) {
                super.onChange(selfChange)

                val newVolume = getStreamVolume(STREAM_MUSIC)
                if (newVolume != currentVolume) {
                    currentVolume = newVolume
                    trySendBlocking(VolumeSettingsChange.VolumeChange(newVolume))
                }

                val newRingerMode = getRingerMode()
                if (newRingerMode != currentRingerMode) {
                    currentRingerMode = newRingerMode
                    trySendBlocking(VolumeSettingsChange.MuteChange(newRingerMode == RINGER_MODE_SILENT))
                }
            }
        }

        context.contentResolver.registerContentObserver(
            Settings.System.CONTENT_URI,
            true,
            contentObserver
        )

        awaitClose {
            context.contentResolver.unregisterContentObserver(contentObserver)
        }
    }

    override fun getOrientation(): String {
        return if (getScreenHeight() > getScreenWidth()) {
            "portrait"
        } else {
            "landscape"
        }
        // Right now, we inject the ApplicationContext which is unreliable to figure our the game orientation.
        // Once we have the Activity provided down, we can fallback to this previous implementation.
        // For now, we will use logic based on width and height calculation.
        /*val activity = context as? Activity
        return when (activity?.windowManager?.defaultDisplay?.rotation) {
            Surface.ROTATION_0 -> "portrait"
            Surface.ROTATION_90 -> "landscape"
            Surface.ROTATION_180 -> "reversePortrait"
            Surface.ROTATION_270 -> "reverseLandscape"
            else -> null
        } ?: when (context.resources.configuration.orientation) {
            ORIENTATION_PORTRAIT -> "portrait"
            ORIENTATION_LANDSCAPE -> "landscape"
            else -> "unknown"
        }*/
    }

    override fun getConnectionTypeStr(): String {
        return when (getConnectionType()) {
            ConnectionType.CONNECTION_TYPE_WIFI -> "wifi"
            ConnectionType.CONNECTION_TYPE_CELLULAR -> "cellular"
            ConnectionType.CONNECTION_TYPE_UNSPECIFIED -> "none"
            else -> "none"
        }
    }

    override fun getCurrentUiTheme(): Int {
        return context.resources.configuration.uiMode
    }

    override fun getLocaleList(): List<String> {
        return if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
            val locales = context.resources.configuration.locales
            List(locales.size()) { locales[it].toString() }
        } else {
            Locale.getAvailableLocales().map(Locale::toString)
        }
    }

    private fun getAirplaneMode(): Boolean {
        return try {
            Settings.Global.getInt(context.contentResolver, Settings.Global.AIRPLANE_MODE_ON, 0) != 0
        } catch (e: Throwable) {
            DeviceLog.error("Problems fetching airplane mode status", e.message)
            false
        }
    }

    fun getChargingType(): Int {
        val intentFilter = IntentFilter(Intent.ACTION_BATTERY_CHANGED)
        val batteryStatus = context.registerReceiver(null, intentFilter)
        return batteryStatus?.getIntExtra(BatteryManager.EXTRA_PLUGGED, -1) ?: -1
    }

    fun getStayOnWhilePluggedIn(): Boolean {
        return try {
            Settings.Global.getInt(context.contentResolver, Settings.Global.STAY_ON_WHILE_PLUGGED_IN, 0) != 0
        } catch (e: Throwable) {
            DeviceLog.error("Problems fetching stay on while plugged in status", e.message)
            false
        }
    }

    fun getIsSdCardPresent(): Boolean {
        return Environment.getExternalStorageState() == Environment.MEDIA_MOUNTED
    }

    fun getNetworkCapabilityTransports(): NetworkCapabilityTransports {
        val result = NetworkCapabilityTransports.newBuilder()

        if (Build.VERSION.SDK_INT < Build.VERSION_CODES.S) {
            return result.build()
        }
        val connectivityManager = context.getSystemService(Context.CONNECTIVITY_SERVICE) as? ConnectivityManager
        val activeNetwork = connectivityManager?.activeNetwork ?: return result.build()
        val networkCapabilities = connectivityManager.getNetworkCapabilities(activeNetwork) ?: return result.build()

        networkCapabilities.hasTransport(NetworkCapabilities.TRANSPORT_WIFI)
            .let(result::setWifi)
        networkCapabilities.hasTransport(NetworkCapabilities.TRANSPORT_CELLULAR)
            .let(result::setCellular)
        networkCapabilities.hasTransport(NetworkCapabilities.TRANSPORT_VPN)
            .let(result::setVpn)
        networkCapabilities.hasTransport(NetworkCapabilities.TRANSPORT_ETHERNET)
            .let(result::setEthernet)
        networkCapabilities.hasTransport(NetworkCapabilities.TRANSPORT_WIFI_AWARE)
            .let(result::setWifiAware)
        networkCapabilities.hasTransport(NetworkCapabilities.TRANSPORT_LOWPAN)
            .let(result::setLowpan)
        networkCapabilities.hasTransport(NetworkCapabilities.TRANSPORT_BLUETOOTH)
            .let(result::setBluetooth)

        return result.build()
    }

    companion object {

        const val DIRECTORY_MEM_INFO: String = "/proc/meminfo"
        const val DIRECTORY_PROCESS_INFO: String = "/proc/self/stat"
        const val DIRECTORY_MODE_READ: String = "r"

        const val KEY_STAT_CONTENT: String = "stat"

        const val INTENT_USB_STATE: String = "android.hardware.usb.action.USB_STATE"
        const val USB_EXTRA_CONNECTED: String = "connected"
    }
}

sealed class VolumeSettingsChange {
    data class MuteChange(val isMuted: Boolean) : VolumeSettingsChange()
    data class VolumeChange(val volume: Double) : VolumeSettingsChange()
}