package com.unity3d.services


import com.google.protobuf.ByteString
import com.unity3d.ads.core.configuration.AlternativeFlowReader
import com.unity3d.ads.core.data.model.CoroutineOpportunity
import com.unity3d.ads.core.domain.SendDiagnosticEvent
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.COROUTINE_NAME
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.IMPRESSION_OPPORTUNITY_ID
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON_DEBUG
import com.unity3d.ads.core.extensions.getShortenedStackTrace
import com.unity3d.ads.core.extensions.retrieveUnityCrashValue
import com.unity3d.ads.core.extensions.toUUID
import com.unity3d.services.UnityAdsConstants.ErrorHandler.ERROR_HANDLER_STACK_TRACE_LINES_MAX
import com.unity3d.services.core.log.DeviceLog
import com.unity3d.services.core.request.metrics.Metric
import com.unity3d.services.core.request.metrics.SDKMetricsSender
import kotlinx.coroutines.CoroutineDispatcher
import kotlinx.coroutines.CoroutineExceptionHandler
import kotlinx.coroutines.CoroutineName
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.launch
import kotlinx.coroutines.plus
import kotlin.coroutines.CoroutineContext


class SDKErrorHandler(
    private val ioDispatcher: CoroutineDispatcher,
    private val alternativeFlowReader: AlternativeFlowReader,
    private val sendDiagnosticEvent: SendDiagnosticEvent,
    private val sdkMetricsSender: SDKMetricsSender
) : CoroutineExceptionHandler {
    private val scope = CoroutineScope(ioDispatcher) + CoroutineName("SDKErrorHandler")

    override val key = CoroutineExceptionHandler.Key

    override fun handleException(context: CoroutineContext, exception: Throwable) {
        val coroutineName = retrieveCoroutineName(context)
        val opportunityId = retrieveOpportunityId(context)
        val name: String = when (exception) {
            is NullPointerException -> "native_exception_npe"
            is OutOfMemoryError -> "native_exception_oom"
            is IllegalStateException -> "native_exception_ise"
            is SecurityException -> "native_exception_se"
            is RuntimeException -> "native_exception_re"
            else -> "native_exception"
        }

        val isAlternativeFlowEnabled = alternativeFlowReader()
        val crashValue = exception.retrieveUnityCrashValue()
        DeviceLog.error("Unity Ads SDK encountered an exception: $crashValue")
        if (isAlternativeFlowEnabled) {
            val stackTrace = exception.getShortenedStackTrace(ERROR_HANDLER_STACK_TRACE_LINES_MAX)
            sendDiagnostic(name, crashValue, coroutineName, stackTrace, opportunityId)
        } else {
            sendMetric(Metric(name, crashValue))
        }

    }

    private fun sendDiagnostic(name: String, reason: String, scopeName: String, stackTrace: String, opportunityId: ByteString?) {
        scope.launch {
            sendDiagnosticEvent(
                event = name,
                tags = buildMap {
                    put(REASON, reason)
                    put(REASON_DEBUG, stackTrace)
                    put(COROUTINE_NAME, scopeName)
                    opportunityId?.let {
                        put(IMPRESSION_OPPORTUNITY_ID, it.toUUID().toString())
                    }
                }
            )
        }
    }

    private fun sendMetric(metric: Metric) = sdkMetricsSender.sendMetric(metric)

    private fun retrieveCoroutineName(context: CoroutineContext) = context[CoroutineName]?.name ?: "unknown"

    private fun retrieveOpportunityId(context: CoroutineContext) = context[CoroutineOpportunity]?.value

    companion object {
        const val UNKNOWN_FILE = "unknown"
        const val UNITY_PACKAGE = "com.unity3d"
    }
}