package com.unity3d.ads.core.domain

import android.content.Context
import android.content.Intent
import com.unity3d.ads.adplayer.AndroidFullscreenWebViewAdPlayer
import com.unity3d.ads.adplayer.DisplayMessage
import com.unity3d.ads.core.data.model.AdObject
import com.unity3d.ads.core.extensions.toUUID
import gatewayprotocol.v1.DiagnosticEventRequestOuterClass.DiagnosticAdType
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.currentCoroutineContext
import kotlinx.coroutines.flow.filter
import kotlinx.coroutines.flow.first
import kotlinx.coroutines.flow.onSubscription
import kotlinx.coroutines.launch

class AndroidHandleOpenUrl(
    private val context: Context,
    private val intentCreation: IntentCreation
) : HandleOpenUrl {
    override suspend fun invoke(
        adObject: AdObject,
        url: String,
        packageName: String?,
        action: String?,
        extras: Map<String, Any?>?,
        useActivityForResult: Boolean
    ): Boolean {
        val intent = intentCreation(
            url = url,
            packageName = packageName,
            action = action,
            extras = extras
        )
        return if (adObject.adType == DiagnosticAdType.DIAGNOSTIC_AD_TYPE_BANNER) {
            // Banners handle open url directly from the context
            intent.setFlags(Intent.FLAG_ACTIVITY_NEW_TASK)
            runCatching { context.startActivity(intent) }.isSuccess
        } else {
            val result = AndroidFullscreenWebViewAdPlayer.displayMessages.onSubscription {
                CoroutineScope(currentCoroutineContext()).launch {
                    AndroidFullscreenWebViewAdPlayer.displayMessages.emit(
                        DisplayMessage.OpenUrl(
                            opportunityId = adObject.opportunityId.toUUID().toString(),
                            intent = intent,
                            useActivityForResult = useActivityForResult
                        )
                    )
                }
            }.filter {
                it is DisplayMessage.OpenUrlResult && it.opportunityId == adObject.opportunityId.toUUID().toString()
            }.first() as DisplayMessage.OpenUrlResult
            result.success
        }
    }
}