package com.unity3d.ads.adplayer

import android.content.Intent
import android.os.Bundle
import android.view.KeyEvent
import android.view.ViewGroup
import android.webkit.WebView
import androidx.activity.ComponentActivity
import androidx.activity.addCallback
import androidx.activity.result.contract.ActivityResultContracts
import androidx.lifecycle.lifecycleScope
import com.unity3d.ads.core.data.repository.AdRepository
import com.unity3d.ads.core.domain.SendDiagnosticEvent
import com.unity3d.ads.core.extensions.toBuiltInMap
import com.unity3d.ads.core.extensions.toByteString
import com.unity3d.services.core.di.IServiceComponent
import com.unity3d.services.core.di.inject
import com.unity3d.services.core.domain.ISDKDispatchers
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.ensureActive
import kotlinx.coroutines.flow.filter
import kotlinx.coroutines.flow.launchIn
import kotlinx.coroutines.flow.onEach
import kotlinx.coroutines.flow.onSubscription
import kotlinx.coroutines.isActive
import kotlinx.coroutines.launch
import kotlinx.coroutines.suspendCancellableCoroutine
import org.json.JSONObject
import java.util.UUID
import kotlin.coroutines.cancellation.CancellationException
import kotlin.coroutines.resume

/**
 * An activity that displays a fullscreen webview.
 */
class FullScreenWebViewDisplay : ComponentActivity(), IServiceComponent {
    private var opportunityId = ""
    private var showOptions: Map<String, Any?>? = null
    private val sendDiagnosticEvent by inject<SendDiagnosticEvent>()
    private val adObject by lazy {
        val adRepository by inject<AdRepository>()
        runCatching {
            adRepository.getAd(UUID.fromString(opportunityId).toByteString())
        }.getOrNull()
    }
    private val dispatchers by inject<ISDKDispatchers>()

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)

        opportunityId = intent.getStringExtra("opportunityId") ?: "not_provided"

        if (opportunityId == "not_provided") {
            setResult(RESULT_CANCELED)
            // Use CoroutineScope for launching coroutine outside lifecycleScope if activity might finish immediately
            CoroutineScope(dispatchers.default).launch {
                AndroidFullscreenWebViewAdPlayer.displayMessages.emit(DisplayMessage.DisplayError(opportunityId, "Opportunity ID not found"))
            }
            finish()
            return
        }

        val adPlayer = adObject?.adPlayer
        if (adPlayer?.scope?.isActive != true) {
            setResult(RESULT_CANCELED)
            CoroutineScope(dispatchers.default).launch {
                AndroidFullscreenWebViewAdPlayer.displayMessages.emit(DisplayMessage.DisplayError(opportunityId, "AdPlayer is not active. Could be because show was called while the app was in background."))
            }
            finish()
            return
        }

        intent.hasExtra("orientation")
            .takeIf { it }
            ?.let { requestedOrientation = intent.getIntExtra("orientation", -1) }

        showOptions = intent.getStringExtra("showOptions")?.let {
            runCatching { JSONObject(it).toBuiltInMap() }.getOrNull()
        }

        lifecycleScope.launch {
            listenToAdPlayerEvents()
        }

        onBackPressedDispatcher.addCallback(this) {
            // Disable back button for predictive back gesture
        }
    }

    private suspend fun listenToAdPlayerEvents() = suspendCancellableCoroutine { continuation ->
        AndroidFullscreenWebViewAdPlayer.displayMessages
            .onSubscription {
                lifecycleScope.launch(dispatchers.default) {
                    AndroidFullscreenWebViewAdPlayer.displayMessages.emit(DisplayMessage.WebViewInstanceRequest(opportunityId))
                    ensureActive()
                    continuation.resume(Unit)
                }
            }
            .filter { it.opportunityId == opportunityId }
            .onEach {
                runCatching {
                    when (it) {
                        is DisplayMessage.DisplayFinishRequest -> finish()
                        is DisplayMessage.WebViewInstanceResponse -> loadWebView(it.webView)
                        is DisplayMessage.SetOrientation -> requestedOrientation = it.orientation
                        is DisplayMessage.OpenUrl -> openUrl(it.opportunityId, it.intent, it.useActivityForResult)
                        else -> Unit
                    }
                }.getOrElse {
                    lifecycleScope.launch(dispatchers.default) {
                        AndroidFullscreenWebViewAdPlayer.displayMessages.emit(
                            DisplayMessage.DisplayError(
                                opportunityId,
                                it.message ?: "Unknown error"
                            )
                        )
                    }
                }
            }
            .launchIn(lifecycleScope)
    }

    private fun openUrl(opportunityId: String, intent: Intent, useActivityForResult: Boolean = false
    ) {
        val result = runCatching {
            if (intent.resolveActivity(packageManager) != null && useActivityForResult) {
                startForResult.launch(intent)
            } else {
                intent.flags = Intent.FLAG_ACTIVITY_NEW_TASK
                startActivity(intent)
            }
        }.isSuccess

        lifecycleScope.launch(dispatchers.default) {
            AndroidFullscreenWebViewAdPlayer.displayMessages.emit(
                DisplayMessage.OpenUrlResult(
                    opportunityId,
                    result
                )
            )
        }
    }

    private val startForResult = registerForActivityResult(ActivityResultContracts.StartActivityForResult()) { _ ->
        // Handle the result here if needed.
        // This is just required when useActivityForResult is true to return to this activity.
    }

    private fun loadWebView(webView: WebView) {
        CoroutineScope(dispatchers.main).launch {
            try {
                (webView.parent as? ViewGroup)?.removeView(webView)
                setContentView(webView)
                CoroutineScope(dispatchers.default).launch {
                    AndroidFullscreenWebViewAdPlayer.displayMessages.emit(DisplayMessage.DisplayReady(opportunityId, showOptions))
                }
            } catch (error: Throwable) {
                if (error is CancellationException) return@launch

                CoroutineScope(dispatchers.default).launch {
                    AndroidFullscreenWebViewAdPlayer.displayMessages.emit(DisplayMessage.DisplayError(opportunityId, "WebView failed to attach to FullScreenWebViewDisplay."))
                }

                setResult(RESULT_CANCELED)
                finish()
            }
        }
    }

    override fun onKeyDown(keyCode: Int, event: KeyEvent?): Boolean {
        // We keep this for compatibility (older Android OS or Jetpack compose versions), even though back button is disabled via onBackPressedDispatcher
        return keyCode == KeyEvent.KEYCODE_BACK
    }

    override fun onResume() {
        super.onResume()

        CoroutineScope(dispatchers.default).launch {
            AndroidFullscreenWebViewAdPlayer.displayMessages.emit(DisplayMessage.VisibilityChanged(opportunityId, true))
        }
    }

    override fun onPause() {
        super.onPause()

        CoroutineScope(dispatchers.default).launch {
            AndroidFullscreenWebViewAdPlayer.displayMessages.emit(DisplayMessage.VisibilityChanged(opportunityId, false))
        }

        if (isFinishing) {
            CoroutineScope(dispatchers.default).launch {
                AndroidFullscreenWebViewAdPlayer.displayMessages.emit(DisplayMessage.DisplayDestroyed(opportunityId))
            }
        }
    }

    override fun onDestroy() {
        if (isFinishing) {
            CoroutineScope(dispatchers.default).launch {
                AndroidFullscreenWebViewAdPlayer.displayMessages.emit(DisplayMessage.DisplayDestroyed(opportunityId))
            }
        }
        super.onDestroy()
    }

    override fun onWindowFocusChanged(hasFocus: Boolean) {
        super.onWindowFocusChanged(hasFocus)

        CoroutineScope(dispatchers.default).launch {
            AndroidFullscreenWebViewAdPlayer.displayMessages.emit(DisplayMessage.FocusChanged(opportunityId, hasFocus))
        }
    }
}
