package com.unity3d.ads.adplayer

import android.annotation.SuppressLint
import android.content.Context
import android.view.InputEvent
import android.view.MotionEvent
import android.view.ViewGroup
import android.webkit.JavascriptInterface
import android.webkit.WebView
import androidx.annotation.VisibleForTesting
import androidx.core.view.ViewCompat
import androidx.core.view.WindowInsetsCompat
import com.unity3d.ads.adplayer.model.ErrorReason
import com.unity3d.ads.adplayer.model.WebViewBridgeInterface
import com.unity3d.ads.adplayer.model.WebViewClientError
import com.unity3d.ads.core.domain.SendWebViewClientErrorDiagnostics
import kotlinx.coroutines.*
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.asStateFlow
import kotlinx.coroutines.flow.filter
import kotlinx.coroutines.flow.launchIn
import kotlinx.coroutines.flow.onEach
import org.json.JSONArray
import kotlin.math.max

@SuppressLint("ClickableViewAccessibility")
class AndroidWebViewContainer(
    val webView: WebView,
    private val webViewClient: AndroidWebViewClient,
    private val sendWebViewClientErrorDiagnostics: SendWebViewClientErrorDiagnostics,
    mainDispatcher: CoroutineDispatcher,
    defaultDispatcher: CoroutineDispatcher,
    adPlayerScope: CoroutineScope,
    private val context: Context,
) : WebViewContainer {
    companion object {
        @VisibleForTesting
        val SHARED_STRING_BUILDER = StringBuilder(4096) // Memory usage would be around 8192 bytes
        private const val JS_INSTRUCTION = "javascript:window.nativebridge."
        private const val JS_OPENING_PARENTHESES = "("
        private const val JS_CLOSING_PARENTHESES = ")"
        private const val JS_SEMICOLON = ";"
    }

    val scope = adPlayerScope + mainDispatcher + CoroutineName("AndroidWebViewContainer")

    val _lastInputEvent = MutableStateFlow<InputEvent?>(null)
    override val lastInputEvent = _lastInputEvent.asStateFlow()

    init {
        webViewClient.isRenderProcessGone
            .filter { it }
            .onEach { onRenderProcessGone() }
            .launchIn(scope + defaultDispatcher)

        webView.setOnTouchListener { _, event ->
            when (event.actionMasked) {
                MotionEvent.ACTION_UP,
                MotionEvent.ACTION_DOWN,
                MotionEvent.ACTION_POINTER_UP,
                MotionEvent.ACTION_POINTER_DOWN -> {
                    _lastInputEvent.value = event
                }
            }

            false
        }

        applySafeAreaInsets()
    }

    private suspend fun onRenderProcessGone() {
        destroy()
        sendWebViewClientErrorDiagnostics(
            listOf(
                WebViewClientError(
                    "Render process gone",
                    ErrorReason.REASON_WEBVIEW_RENDER_PROCESS_GONE
                )
            )
        )
    }

    override suspend fun loadUrl(url: String) {
        withContext(scope.coroutineContext) {
            webView.loadUrl(url)
        }

        val loadResult = webViewClient.onLoadFinished.await()
        if (loadResult.isNotEmpty()) {
            destroy()
            sendWebViewClientErrorDiagnostics(loadResult)
            throw LoadWebViewError(loadResult)
        }
    }

    override suspend fun evaluateJavascript(handlerType: HandlerType, arguments: JSONArray) {
        try {
            withContext(scope.coroutineContext) {
                val argumentsString = arguments.toString()

                // Example: `javascript:window.nativebridge.handlerType(argumentsString);`
                val neededSize = JS_INSTRUCTION.length +
                    handlerType.jsPath.length +
                    JS_OPENING_PARENTHESES.length +
                    argumentsString.length +
                    JS_CLOSING_PARENTHESES.length +
                    JS_SEMICOLON.length

                val selectedStringBuilder = if (neededSize > SHARED_STRING_BUILDER.capacity()) {
                    StringBuilder(neededSize)
                } else {
                    SHARED_STRING_BUILDER.also(StringBuilder::clear)
                }

                with(selectedStringBuilder) {
                    append(JS_INSTRUCTION)
                    append(handlerType.jsPath)
                    append(JS_OPENING_PARENTHESES)
                    append(argumentsString)
                    append(JS_CLOSING_PARENTHESES)
                    append(JS_SEMICOLON)
                }

                webView.evaluateJavascript(selectedStringBuilder.toString(), null)
            }
        } catch (e: CancellationException) {
            // ignore cancellation exception from the coroutine scope, any other exception should be rethrown
        }
    }

    override suspend fun addJavascriptInterface(webViewBridgeInterface: WebViewBridge, name: String) {
        try {
            withContext(scope.coroutineContext) {
                val wrapper = object : WebViewBridgeInterface {
                    @JavascriptInterface
                    override fun handleInvocation(message: String) = webViewBridgeInterface.handleInvocation(message)

                    @JavascriptInterface
                    override fun handleCallback(callbackId: String, callbackStatus: String, rawParameters: String) =
                        webViewBridgeInterface.handleCallback(callbackId, callbackStatus, rawParameters)
                }
                webView.addJavascriptInterface(wrapper, name)
            }
        } catch (e: CancellationException) {
            // ignore cancellation exception from the coroutine scope, any other exception should be rethrown
        }

    }

    override suspend fun destroy() {
        withContext(scope.coroutineContext + NonCancellable) {
            (webView.parent as? ViewGroup)?.removeView(webView)
            webView.destroy()
        }

        scope.cancel()
    }

    private fun applySafeAreaInsets() {
        ViewCompat.setOnApplyWindowInsetsListener(webView) { v, insets ->
            val systemBarsInsets = insets.getInsets(WindowInsetsCompat.Type.systemBars())
            val displayCutout = insets.getInsets(WindowInsetsCompat.Type.displayCutout())

            val density = runCatching { context.resources.displayMetrics.density }.getOrDefault(1f)
            fun Int.toPx() = (this / density).toInt()

            // Calculate combined insets, explicitly considering cutouts
            val left = max(systemBarsInsets.left, displayCutout.left).toPx()
            val top = max(systemBarsInsets.top, displayCutout.top).toPx()
            val right = max(systemBarsInsets.right, displayCutout.right).toPx()
            val bottom = max(systemBarsInsets.bottom, displayCutout.bottom).toPx()

            val js = """
                (function() {
                    const root = document.documentElement;
                    root.style.setProperty('--safe-area-inset-left', '${left}px');
                    root.style.setProperty('--safe-area-inset-right', '${right}px');
                    root.style.setProperty('--safe-area-inset-top', '${top}px');
                    root.style.setProperty('--safe-area-inset-bottom', '${bottom}px');
                })();
            """.trimIndent()

            scope.launch {
                runCatching { webView.evaluateJavascript(js, null) }
            }

            insets
        }
    }
}