package com.unity3d.ads.core.domain

import android.content.Context
import com.google.android.gms.net.CronetProviderInstaller
import com.unity3d.ads.core.configuration.AlternativeFlowReader
import com.unity3d.ads.core.configuration.MediationTraitsMetadataReader
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.SYSTEM_CRONET_ENGINE_ERROR
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.SYSTEM_CRONET_FAILURE
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.SYSTEM_CRONET_INSTALL_ERROR
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.SYSTEM_CRONET_SUCCESS
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.SYSTEM_CRONET_TIMEOUT
import com.unity3d.services.core.di.ServiceProvider.CDN_CREATIVES_HOST
import com.unity3d.services.core.di.ServiceProvider.CDN_CREATIVES_PORT
import com.unity3d.services.core.di.ServiceProvider.GATEWAY_HOST
import com.unity3d.services.core.di.ServiceProvider.GATEWAY_PORT
import com.unity3d.services.core.di.ServiceProvider.HTTP_CLIENT_FETCH_TIMEOUT
import com.unity3d.services.core.domain.ISDKDispatchers
import com.unity3d.services.core.network.core.CronetClient
import com.unity3d.services.core.network.core.CronetEngineBuilderFactory
import com.unity3d.services.core.network.core.HttpClient
import com.unity3d.services.core.network.core.LegacyHttpClient
import com.unity3d.services.core.network.core.OkHttp3Client
import kotlinx.coroutines.suspendCancellableCoroutine
import kotlinx.coroutines.withTimeoutOrNull
import okhttp3.OkHttpClient
import kotlin.coroutines.resume
import kotlin.time.DurationUnit
import kotlin.time.ExperimentalTime
import kotlin.time.TimeSource

enum class HttpClientSelection {
    CRONET_WITHOUT_QUIC,
    CRONET_WITH_QUIC,
    LEGACY,
    OKHTTP3,
}

class AndroidHttpClientProvider(
    private val alternativeFlowReader: AlternativeFlowReader,
    private val dispatchers: ISDKDispatchers,
    private val sendDiagnosticEvent: SendDiagnosticEvent,
    private val context: Context,
    private val cronetEngineBuilderFactory: CronetEngineBuilderFactory,
    private val mediationTraitsMetadataReader: MediationTraitsMetadataReader,
) : HttpClientProvider {
    @OptIn(ExperimentalTime::class)
    override suspend fun invoke(): HttpClient {
        if (!alternativeFlowReader()) {
            return LegacyHttpClient(dispatchers)
        }

        val selectedHttpClient = mediationTraitsMetadataReader.getStringTrait(MediationTraitsMetadataReader.USE_HTTP_CLIENT)?.let {
            runCatching { HttpClientSelection.valueOf(it.uppercase()) }.getOrNull()
        } ?: HttpClientSelection.CRONET_WITHOUT_QUIC

        val startTime = TimeSource.Monotonic.markNow()

        val client = withTimeoutOrNull(HTTP_CLIENT_FETCH_TIMEOUT) {
            when (selectedHttpClient) {
                HttpClientSelection.OKHTTP3 -> getOkHttp3Client()
                HttpClientSelection.LEGACY -> LegacyHttpClient(dispatchers)
                else -> buildNetworkClient(context, dispatchers, selectedHttpClient)
            }
        }

        val elapsedTime = startTime.elapsedNow().toDouble(DurationUnit.MILLISECONDS)

        if ("CRONET" in selectedHttpClient.name) {
            val diagnosticResult = when (client) {
                null -> SYSTEM_CRONET_TIMEOUT
                is CronetClient -> SYSTEM_CRONET_SUCCESS
                else -> SYSTEM_CRONET_FAILURE
            }
            sendDiagnosticEvent(diagnosticResult, elapsedTime)
        }

        return client ?: getOkHttp3Client()
    }

    private suspend fun buildNetworkClient(
        context: Context, dispatchers: ISDKDispatchers, selectedHttpClient: HttpClientSelection
    ): HttpClient = suspendCancellableCoroutine { continuation ->
        CronetProviderInstaller.installProvider(context).addOnCompleteListener {
            if (it.isSuccessful) {
                try {
                    val cronetEngine =
                        cronetEngineBuilderFactory.createCronetEngineBuilder(context).apply {
                            if (selectedHttpClient == HttpClientSelection.CRONET_WITH_QUIC) {
                                enableQuic(true)
                                addQuicHint(GATEWAY_HOST, GATEWAY_PORT, GATEWAY_PORT)
                                addQuicHint(CDN_CREATIVES_HOST, CDN_CREATIVES_PORT, CDN_CREATIVES_PORT)
                            }
                        }.build()
                    continuation.resume(CronetClient(cronetEngine, dispatchers))
                } catch (e: Throwable) {
                    sendDiagnosticEvent(SYSTEM_CRONET_ENGINE_ERROR, tags = mapOf(
                        REASON to (e.message ?: "Errored without message."),
                    ))

                    continuation.resume(getOkHttp3Client())
                }
            } else {
                sendDiagnosticEvent(SYSTEM_CRONET_INSTALL_ERROR, tags = mapOf(
                    REASON to (it.exception?.message ?: "Errored without message."),
                ))
                continuation.resume(getOkHttp3Client())
            }
        }
    }

    fun getOkHttp3Client(): OkHttp3Client = OkHttp3Client(dispatchers, OkHttpClient())
}