package com.unity3d.ads.core.domain.exposure

import android.util.Base64
import com.google.protobuf.ByteString
import com.google.protobuf.kotlin.toByteString
import com.unity3d.ads.adplayer.AndroidFullscreenWebViewAdPlayer
import com.unity3d.ads.adplayer.DisplayMessage
import com.unity3d.ads.adplayer.ExposedFunction
import com.unity3d.ads.adplayer.model.OnWebRequestComplete
import com.unity3d.ads.adplayer.model.OnWebRequestFailed
import com.unity3d.ads.core.data.model.AdData
import com.unity3d.ads.core.data.model.AdDataRefreshToken
import com.unity3d.ads.core.data.model.AdObject
import com.unity3d.ads.core.data.model.CacheResult
import com.unity3d.ads.core.data.model.ImpressionConfig
import com.unity3d.ads.core.data.repository.CampaignRepository
import com.unity3d.ads.core.data.repository.DeviceInfoRepository
import com.unity3d.ads.core.data.repository.SessionRepository
import com.unity3d.ads.core.domain.AndroidGetAdPlayerContext
import com.unity3d.ads.core.domain.CacheFile
import com.unity3d.ads.core.domain.ExecuteAdViewerRequest
import com.unity3d.ads.core.domain.GetIsFileCache
import com.unity3d.ads.core.domain.HandleInvocationsFromAdViewer
import com.unity3d.ads.core.domain.HandleInvocationsFromAdViewer.Companion.KEY_AD_STRING
import com.unity3d.ads.core.domain.HandleInvocationsFromAdViewer.Companion.KEY_AD_TYPE
import com.unity3d.ads.core.domain.HandleInvocationsFromAdViewer.Companion.KEY_AD_UNIT_ID
import com.unity3d.ads.core.domain.HandleInvocationsFromAdViewer.Companion.KEY_DOWNLOAD_PRIORITY
import com.unity3d.ads.core.domain.HandleInvocationsFromAdViewer.Companion.KEY_DOWNLOAD_URL
import com.unity3d.ads.core.domain.HandleInvocationsFromAdViewer.Companion.KEY_OMJS_SERVICE
import com.unity3d.ads.core.domain.HandleInvocationsFromAdViewer.Companion.KEY_OMJS_SESSION
import com.unity3d.ads.core.domain.HandleInvocationsFromAdViewer.Companion.KEY_OM_PARTNER
import com.unity3d.ads.core.domain.HandleInvocationsFromAdViewer.Companion.KEY_OM_PARTNER_VERSION
import com.unity3d.ads.core.domain.HandleInvocationsFromAdViewer.Companion.KEY_OM_VERSION
import com.unity3d.ads.core.domain.HandleInvocationsFromAdViewer.Companion.KEY_PLACEMENT_NAME
import com.unity3d.ads.core.domain.HandleInvocationsFromAdViewer.Companion.KEY_QUERY_ID
import com.unity3d.ads.core.domain.HandleInvocationsFromAdViewer.Companion.KEY_VIDEO_LENGTH
import com.unity3d.ads.core.domain.HandleOpenUrl
import com.unity3d.ads.core.domain.Refresh
import com.unity3d.ads.core.domain.SendDiagnosticEvent
import com.unity3d.ads.core.domain.SendPrivacyUpdateRequest
import com.unity3d.ads.core.domain.attribution.AndroidAttribution
import com.unity3d.ads.core.domain.events.GetOperativeEventApi
import com.unity3d.ads.core.domain.offerwall.GetIsOfferwallAdReady
import com.unity3d.ads.core.domain.offerwall.LoadOfferwallAd
import com.unity3d.ads.core.domain.om.AndroidOmInteraction
import com.unity3d.ads.core.domain.om.GetOmData
import com.unity3d.ads.core.domain.om.IsOMActivated
import com.unity3d.ads.core.domain.om.OmFinishSession
import com.unity3d.ads.core.domain.om.OmImpressionOccurred
import com.unity3d.ads.core.domain.scar.LoadScarAd
import com.unity3d.ads.core.extensions.fromBase64
import com.unity3d.ads.core.extensions.toBase64
import com.unity3d.ads.core.utils.ContinuationFromCallback
import com.unity3d.services.core.network.model.RequestType
import com.unity3d.services.UnityAdsConstants.Cache.CACHE_SCHEME
import com.unity3d.services.UnityAdsConstants.DefaultUrls.AD_CACHE_DOMAIN
import com.unity3d.services.UnityAdsConstants.OpenMeasurement.OM_JS_URL_SERVICE
import com.unity3d.services.UnityAdsConstants.OpenMeasurement.OM_JS_URL_SESSION
import com.unity3d.services.core.api.Storage
import com.unity3d.services.core.network.mapper.toResponseHeadersMap
import gatewayprotocol.v1.OperativeEventRequestOuterClass
import gatewayprotocol.v1.campaign
import gatewayprotocol.v1.copy
import kotlinx.coroutines.flow.update
import org.json.JSONArray
import org.json.JSONObject
import kotlin.coroutines.suspendCoroutine
import kotlin.time.Duration.Companion.seconds

internal fun getAdContext(
    getAndroidAdPlayerContext: AndroidGetAdPlayerContext,
    adData: AdData,
    impressionConfig: ImpressionConfig,
    adDataRefreshToken: AdDataRefreshToken,
    isOMActivated: IsOMActivated,
    adObject: AdObject,
    ) = ExposedFunction {
    buildMap {
        put(HandleInvocationsFromAdViewer.KEY_AD_DATA, adData.data)
        put(HandleInvocationsFromAdViewer.KEY_IMPRESSION_CONFIG, impressionConfig.data)
        put(HandleInvocationsFromAdViewer.KEY_AD_DATA_REFRESH_TOKEN, adDataRefreshToken.data)
        put(HandleInvocationsFromAdViewer.KEY_NATIVE_CONTEXT, getAndroidAdPlayerContext())
        put(HandleInvocationsFromAdViewer.KEY_TRACKING_TOKEN, adObject.trackingToken.toBase64())
        adObject.loadOptions.data?.let { loadOptions ->
            // Filter out adMarkup and objectId from loadOptions
            if (loadOptions.length() != 0) {
                put(HandleInvocationsFromAdViewer.KEY_LOAD_OPTIONS, loadOptions.keys().asSequence().fold(JSONObject()) { acc, key ->
                    if (key == "adMarkup" || key == "objectId") return@fold acc
                    acc.put(key, loadOptions[key])
                })
            }
        }
        if (isOMActivated()) {
            put(HandleInvocationsFromAdViewer.KEY_OMID, mapOf (
                KEY_OMJS_SESSION to OM_JS_URL_SESSION,
                KEY_OMJS_SERVICE to OM_JS_URL_SERVICE
            ))
        }
        put(HandleInvocationsFromAdViewer.KEY_IS_HEADER_BIDDING, adObject.isHeaderBidding)
    }
}

internal fun getConnectionType(deviceInfoRepository: DeviceInfoRepository) = ExposedFunction {
    deviceInfoRepository.dynamicDeviceInfo.connectionType
}

internal fun getDeviceVolume(deviceInfoRepository: DeviceInfoRepository) = ExposedFunction {
    deviceInfoRepository.dynamicDeviceInfo.android.volume
}

internal fun getDeviceMaxVolume(deviceInfoRepository: DeviceInfoRepository) = ExposedFunction {
    deviceInfoRepository.dynamicDeviceInfo.android.maxVolume
}

internal fun getScreenHeight(deviceInfoRepository: DeviceInfoRepository) = ExposedFunction {
    deviceInfoRepository.dynamicDeviceInfo.screenHeight
}

internal fun getScreenWidth(deviceInfoRepository: DeviceInfoRepository) = ExposedFunction {
    deviceInfoRepository.dynamicDeviceInfo.screenWidth
}

internal fun openUrl(handleOpenUrl: HandleOpenUrl) = ExposedFunction {
    val url = it[0] as String
    val params = it.getOrNull(1) as? JSONObject
    val packageName = params?.optString(HandleInvocationsFromAdViewer.KEY_PACKAGE_NAME)

    handleOpenUrl(url, packageName)
}

internal fun setOrientation(adObject: AdObject) = ExposedFunction { args ->
    val orientation = args[0] as Int
    AndroidFullscreenWebViewAdPlayer.displayMessages.emit(
        DisplayMessage.SetOrientation(
            opportunityId = adObject.opportunityId.toStringUtf8(),
            orientation = orientation
        )
    )
}

internal fun sendOperativeEvent(getOperativeEventApi: GetOperativeEventApi, adObject: AdObject) = ExposedFunction {
    getOperativeEventApi(
        adObject = adObject,
        operativeEventType = OperativeEventRequestOuterClass.OperativeEventType.OPERATIVE_EVENT_TYPE_SPECIFIED_BY_AD_PLAYER,
        additionalEventData = Base64.decode(it[0] as String, Base64.NO_WRAP).toByteString()
    )
}

internal fun writeStorage() = ExposedFunction { args ->
    suspendCoroutine {
        Storage.write(
            args[0] as String,
            ContinuationFromCallback(it)
        )
    }
}

internal fun readStorage() = ExposedFunction { args ->
    suspendCoroutine {
        Storage.read(
            args[0] as String,
            ContinuationFromCallback(it)
        )
    }

}

internal fun deleteStorage() = ExposedFunction { args ->
    suspendCoroutine {
        Storage.delete(
            args[0] as String,
            args[1] as String,
            ContinuationFromCallback(it)
        )
    }
}

internal fun clearStorage() = ExposedFunction { args ->
    suspendCoroutine {
        Storage.clear(
            args[0] as String,
            ContinuationFromCallback(it)
        )
    }
}

internal fun getKeysStorage() = ExposedFunction { args ->
    suspendCoroutine {
        Storage.getKeys(
            args[0] as String,
            args[1] as String,
            args[2] as Boolean,
            ContinuationFromCallback(it)
        )
    }
}

internal fun getStorage() = ExposedFunction { args ->
    suspendCoroutine {
        Storage.get(
            args[0] as String,
            args[1] as String,
            ContinuationFromCallback(it)
        )
    }
}

internal fun setStorage() = ExposedFunction { args ->
    suspendCoroutine {
        Storage.set(
            args[0] as String,
            args[1] as String,
            args[2],
            ContinuationFromCallback(it)
        )
    }
}

internal fun getPrivacyFsm(sessionRepository: SessionRepository) = ExposedFunction {
    sessionRepository.getPrivacyFsm().toBase64()
}

internal fun setPrivacyFsm(sessionRepository: SessionRepository) = ExposedFunction {
    sessionRepository.setPrivacyFsm(
        Base64.decode(it[0] as String, Base64.NO_WRAP).toByteString()
    )
}

internal fun getPrivacy(sessionRepository: SessionRepository) = ExposedFunction {
    sessionRepository.getPrivacy().toBase64()
}

internal fun setPrivacy(sessionRepository: SessionRepository) = ExposedFunction {
    sessionRepository.setPrivacy(
        Base64.decode(it[0] as String, Base64.NO_WRAP).toByteString()
    )
}

internal fun getAllowedPii(deviceInfoRepository: DeviceInfoRepository) = ExposedFunction {
    Base64.encodeToString(deviceInfoRepository.allowedPii.value.toByteArray(), Base64.NO_WRAP)
}

internal fun setAllowedPii(deviceInfoRepository: DeviceInfoRepository) = ExposedFunction {
    val allowedPiiUpdated = it[0] as JSONObject

    deviceInfoRepository.allowedPii.update { allowedPii ->
        allowedPii.copy {
            (allowedPiiUpdated.opt("idfa") as? Boolean)?.let(::idfa::set)
            (allowedPiiUpdated.opt("idfv") as? Boolean)?.let(::idfv::set)
            (allowedPiiUpdated.opt("appset_id") as? Boolean)?.let(::appsetId::set)
        }
    }
}

internal fun getSessionToken(sessionRepository: SessionRepository) = ExposedFunction {
    sessionRepository.sessionToken.toBase64()
}

internal fun markCampaignStateShown(campaignRepository: CampaignRepository, adObject: AdObject) = ExposedFunction {
    campaignRepository.setShowTimestamp(adObject.opportunityId)
}

internal fun refreshAdData(refresh: Refresh, adObject: AdObject) = ExposedFunction {
    val refreshTokenByteString = if (it.isEmpty()) {
        ByteString.EMPTY
    } else {
        val refreshTokenJson = it[0] as JSONObject
        val refreshToken = refreshTokenJson.optString(HandleInvocationsFromAdViewer.KEY_AD_DATA_REFRESH_TOKEN)
        refreshToken.fromBase64()
    }

    val adRefreshResponse = refresh(refreshTokenByteString, adObject.opportunityId)

    if (adRefreshResponse.hasError()) throw IllegalArgumentException("Refresh failed")

    buildMap {
        put(HandleInvocationsFromAdViewer.KEY_AD_DATA, adRefreshResponse.adData.toBase64())
        put(HandleInvocationsFromAdViewer.KEY_AD_DATA_REFRESH_TOKEN, adRefreshResponse.adDataRefreshToken.toBase64())
        put(HandleInvocationsFromAdViewer.KEY_TRACKING_TOKEN, adRefreshResponse.trackingToken.toBase64())
    }
}

internal fun updateTrackingToken(adObject: AdObject) = ExposedFunction {
    val updateTrackingToken = it[0] as JSONObject
    val token = updateTrackingToken.optString("trackingToken")

    if (!token.isNullOrEmpty()) {
        adObject.trackingToken = token.fromBase64()
    }
}

internal fun sendPrivacyUpdateRequest(sendPrivacyUpdateRequest: SendPrivacyUpdateRequest) = ExposedFunction {
    val privacyUpdateRequest = it[0] as JSONObject

    val privacyUpdateContentBase64 = privacyUpdateRequest.optString(HandleInvocationsFromAdViewer.KEY_PRIVACY_UPDATE_CONTENT)
    val privacyUpdateVersion = privacyUpdateRequest.optInt(HandleInvocationsFromAdViewer.KEY_PRIVACY_UPDATE_VERSION)

    val response = sendPrivacyUpdateRequest(privacyUpdateVersion, privacyUpdateContentBase64.fromBase64())

    buildMap {
        put(HandleInvocationsFromAdViewer.KEY_PRIVACY_UPDATE_VERSION, response.version)
        put(HandleInvocationsFromAdViewer.KEY_PRIVACY_UPDATE_CONTENT, response.content.toBase64())
    }
}

internal fun sendDiagnosticEvent(sendDiagnosticEvent: SendDiagnosticEvent, adObject: AdObject) = ExposedFunction {
    val event = it[0] as String
    val tags = it[1] as JSONObject
    val tagsMap = buildMap {
        tags.keys().forEach { key ->
            put(key, tags.getString(key))
        }
    }
    val value = it.getOrNull(2)?.toString()?.toDouble()
    sendDiagnosticEvent(event = event, value = value, tags = tagsMap, adObject = adObject)
}

internal fun incrementBannerImpressionCount(sessionRepository: SessionRepository) = ExposedFunction {
    sessionRepository.incrementBannerImpressionCount()
}

internal fun download(cacheFile: CacheFile, adObject: AdObject) = ExposedFunction {
    val json = it[0] as JSONObject
    val url = json.getString(KEY_DOWNLOAD_URL)
    val headers = it.getOrNull(2) as JSONArray?
    val priority = json.optInt(KEY_DOWNLOAD_PRIORITY, 0)

    when (val result = cacheFile(url, adObject, headers, priority)) {
        is CacheResult.Success -> "$CACHE_SCHEME://$AD_CACHE_DOMAIN/${result.cachedFile.name}"
        is CacheResult.Failure -> error(result.error.name)
    }
}

internal fun isFileCached(getIfFileCache: GetIsFileCache) = ExposedFunction {
    val fileUrl = it[0] as String
    getIfFileCache(fileUrl)
}

internal fun omStartSession(omStartSession: AndroidOmInteraction, adObject: AdObject) = ExposedFunction {
    val options = it[0] as JSONObject
    omStartSession(adObject, options)
}

internal fun omFinishSession(omFinishSession: OmFinishSession, adObject: AdObject) = ExposedFunction {
    omFinishSession(adObject)
}

internal fun omImpression(omImpressionOccurred: OmImpressionOccurred, adObject: AdObject) = ExposedFunction {
    val signalLoaded = it[0] as Boolean
    omImpressionOccurred(adObject, signalLoaded)
}

internal fun omGetData(getOmData: GetOmData) = ExposedFunction {
    val data = getOmData()
    buildMap {
        put(KEY_OM_VERSION, data.version)
        put(KEY_OM_PARTNER, data.partnerName)
        put(KEY_OM_PARTNER_VERSION, data.partnerVersion)
    }
}

internal fun isAttributionAvailable(androidAttribution: AndroidAttribution) = ExposedFunction {
    androidAttribution.isAvailable()
}

internal fun attributionRegisterView(androidAttribution: AndroidAttribution, adObject: AdObject) = ExposedFunction { args ->
    androidAttribution.registerView(args[0] as String, adObject)
}

internal fun attributionRegisterClick(androidAttribution: AndroidAttribution, adObject: AdObject) = ExposedFunction { args ->
    androidAttribution.registerClick(args[0] as String, adObject)
}

internal fun loadScarAd(loadScarAd: LoadScarAd, adObject: AdObject) = ExposedFunction { args ->
    val params = args[0] as JSONObject
    val adType = params.optString(KEY_AD_TYPE)
    val adUnitId = params.optString(KEY_AD_UNIT_ID)
    val adString = params.optString(KEY_AD_STRING)
    val queryId = params.optString(KEY_QUERY_ID)
    val videoLength = params.optInt(KEY_VIDEO_LENGTH)
    adObject.isScarAd = true
    adObject.scarAdUnitId = adUnitId
    adObject.scarQueryId = queryId
    adObject.scarAdString = adString
    loadScarAd(adType, adObject.placementId, adUnitId, adString, queryId, videoLength)
}

internal fun showScarAd() = ExposedFunction { }

internal fun hbTokenIncrementWins(sessionRepository: SessionRepository) = ExposedFunction {
    sessionRepository.incrementTokenWinsCount()
}

internal fun hbTokenIncrementStarts(sessionRepository: SessionRepository) = ExposedFunction {
    sessionRepository.incrementTokenStartsCount()
}

internal fun hbTokenReset(sessionRepository: SessionRepository) = ExposedFunction {
    sessionRepository.resetTokenCounters()
}

internal fun loadOfferwallAd(loadOfferwallAd: LoadOfferwallAd, adObject: AdObject) = ExposedFunction { args ->
    val params = args[0] as JSONObject
    val placementName = params.optString(KEY_PLACEMENT_NAME)

    adObject.isOfferwallAd = true
    adObject.offerwallPlacementName = placementName

    loadOfferwallAd(placementName)
}

internal fun showOfferwallAd() = ExposedFunction { }

internal fun isOfferwallAdReady(getIsOfferwallAdReady: GetIsOfferwallAdReady) = ExposedFunction { args ->
    val params = args[0] as JSONObject
    val placementName = params.optString(KEY_PLACEMENT_NAME)

    getIsOfferwallAdReady(placementName)
}

internal fun request(type: RequestType, executeAdViewerRequest: ExecuteAdViewerRequest) = ExposedFunction { args ->
    val id = args.first() as String
    val url = args.getOrNull(1) as String?

    try {
        val response = executeAdViewerRequest(type, args)
        val body = when (val responseBody = response.body) {
            is String -> responseBody
            is ByteArray -> String(responseBody, Charsets.UTF_8)
            else -> null
        }

        OnWebRequestComplete(listOf(
            id,
            response.urlString,
            body,
            response.statusCode,
            response.headers.toResponseHeadersMap()
        ))
    } catch (e: Exception) {
        OnWebRequestFailed(listOf(id, url, e.message ?: ""))
    }
}


internal fun setOpportunityTTL(adObject: AdObject) = ExposedFunction { args ->
    val ttl = args[0] as Int
    adObject.ttl.value = ttl.seconds
    Unit
}

internal fun updateCampaignState(campaignRepository: CampaignRepository, adObject: AdObject) = ExposedFunction { args ->
    val update = args.firstOrNull() as? JSONObject
    requireNotNull(update) { "Update campaign state requires a JSONObject" }

    val data = update.optString("data")
    require(!data.isNullOrBlank()) { "Update campaign state requires a data string" }

    val dataVersion = update.optInt("dataVersion")
    require(dataVersion != 0) { "Update campaign state requires a dataVersion integer" }

    val opportunityId = adObject.opportunityId
    val placementId = adObject.placementId

    val byteStringData = data.fromBase64()
    require(!byteStringData.isEmpty) { "Update campaign state requires a non-empty data byte string" }

    val state = campaignRepository.getCampaign(opportunityId)
        ?.copy {
            this.data = byteStringData
            this.dataVersion = dataVersion
        } ?: campaign {
        this.data = byteStringData
        this.dataVersion = dataVersion
        this.placementId = placementId
        this.impressionOpportunityId = opportunityId
    }

    campaignRepository.setCampaign(opportunityId, state)
}
