package com.unity3d.services.ads.offerwall

import com.unity3d.ads.core.domain.offerwall.OfferwallEventData
import com.unity3d.services.core.log.DeviceLog
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.flow.MutableSharedFlow
import kotlinx.coroutines.flow.asSharedFlow
import kotlinx.coroutines.launch
import java.lang.reflect.InvocationHandler
import java.lang.reflect.Proxy

class OfferwallAdapterBridge(private val scope: CoroutineScope) {

    private val _offerwallEventFlow = MutableSharedFlow<OfferwallEventData>()
    val offerwallEventFlow = _offerwallEventFlow.asSharedFlow()
    private val placementsMap = hashMapOf<String, Any?>()

    fun getVersion() = try {
        val tapjoyClass = Class.forName("com.tapjoy.Tapjoy")
        val getVersionMethod = tapjoyClass.getDeclaredMethod("getVersion")
        (getVersionMethod.invoke(null) as String?) ?: "0.0.0"
    } catch (ex: Exception) {
        "0.0.0"
    }

    fun isConnected() = try {
        val tapjoyClass = Class.forName("com.tapjoy.Tapjoy")
        val getIsConnectedMethod = tapjoyClass.getDeclaredMethod("isConnected")
        (getIsConnectedMethod.invoke(null) as Boolean?) ?: false
    } catch (ex: Exception) {
        false
    }

    fun isAdReady(placementName: String): Boolean {
        val placementInstance = getPlacement(placementName)
        val isContentReadyMethod = placementInstance?.javaClass?.getDeclaredMethod("isContentReady")
        return (isContentReadyMethod?.invoke(placementInstance) as Boolean?) ?: false
    }

    fun loadAd(placementName: String) {
        val placementInstance = getPlacement(placementName)
        placementsMap[placementName] = placementInstance
        val requestContentMethod = placementInstance?.javaClass?.getDeclaredMethod("requestContent")
        requestContentMethod?.invoke(placementInstance)
    }

    fun showAd(placementName: String) {
        val placementInstance = placementsMap[placementName]
        val isContentAvailableMethod = placementInstance?.javaClass?.getDeclaredMethod("isContentAvailable")
        val placementAvailable = (isContentAvailableMethod?.invoke(placementInstance) as Boolean?) ?: false
        if (placementAvailable) {
            val showContentMethod = placementInstance?.javaClass?.getDeclaredMethod("showContent")
            showContentMethod?.invoke(placementInstance)
        } else {
            scope.launch {
                _offerwallEventFlow.emit(
                    OfferwallEventData(
                        OfferwallEvent.SHOW_FAILED,
                        placementName = placementName
                    )
                )
            }
        }
        placementsMap.remove(placementName)
    }

    private fun tapjoyPlacementListener(): InvocationHandler {
        return InvocationHandler { proxy, method, args ->
            when (method.name) {
                "onRequestSuccess" -> {
                    val placementObj = args[0]
                    val placementName = getPlacementName(placementObj)
                    val placementAvailable = (placementObj.javaClass.getDeclaredMethod("isContentAvailable").invoke(placementObj) as Boolean?) ?: false
                    DeviceLog.debug("Placement request succeeded: $placementName - isContentAvailable: $placementAvailable")
                    scope.launch {
                        if (placementAvailable) {
                            _offerwallEventFlow.emit(OfferwallEventData(
                                OfferwallEvent.REQUEST_SUCCESS,
                                placementName = placementName ?: ""
                            ))
                        } else {
                            placementsMap.remove(placementName ?: "")
                            _offerwallEventFlow.emit(
                                OfferwallEventData(
                                    OfferwallEvent.REQUEST_FAILED,
                                    placementName = placementName ?: ""
                                )
                            )
                        }
                    }
                }

                "onRequestFailure" -> {
                    val placementName = getPlacementName(args[0])
                    placementsMap.remove(placementName ?: "")
                    val errorObj = args[1]
                    val errorCode = errorObj.javaClass.getField("code").get(errorObj) as Int?
                    val errorMessage = errorObj.javaClass.getField("message").get(errorObj) as String?
                    DeviceLog.error("Placement request failed: $placementName - errorCode: $errorCode, errorMessage: $errorMessage")
                    scope.launch {
                        _offerwallEventFlow.emit(
                            OfferwallEventData(
                                OfferwallEvent.REQUEST_FAILED,
                                placementName = placementName ?: ""
                            )
                        )
                    }
                }

                "onContentReady" -> {
                    val placementName = getPlacementName(args[0])
                    DeviceLog.debug("Placement content ready: $placementName")
                    scope.launch {
                        _offerwallEventFlow.emit(
                            OfferwallEventData(
                                OfferwallEvent.ON_CONTENT_READY,
                                placementName = placementName ?: ""
                            )
                        )
                    }
                }

                "onContentShow" -> {
                    val placementName = getPlacementName(args[0])
                    DeviceLog.debug("Placement content shown: $placementName")
                    scope.launch {
                        _offerwallEventFlow.emit(
                            OfferwallEventData(
                                OfferwallEvent.ON_CONTENT_SHOW,
                                placementName = placementName ?: ""
                            )
                        )
                    }
                }

                "onContentDismiss" -> {
                    val placementName = getPlacementName(args[0])
                    DeviceLog.debug("Placement content dismissed: $placementName")
                    scope.launch {
                        _offerwallEventFlow.emit(
                            OfferwallEventData(
                                OfferwallEvent.ON_CONTENT_DISMISS,
                                placementName = placementName ?: ""
                            )
                        )
                    }
                }
            }
            null
        }
    }

    private fun getPlacementName(placementObj: Any): String? {
        return placementObj.javaClass.getDeclaredMethod("getName").invoke(placementObj) as String?
    }

    private fun getPlacement(placementName: String): Any? {
        // Obtain the Tapjoy class using reflection
        val tapjoyClass = Class.forName("com.tapjoy.Tapjoy")

        // Obtain the TJPlacement getPlacement method
        val getPlacementMethod = tapjoyClass.getDeclaredMethod(
            "getPlacement",
            String::class.java,
            Class.forName("com.tapjoy.TJPlacementListener")
        )

        // Create an instance of TJPlacementListener using Proxy
        val placementListenerInstance = Proxy.newProxyInstance(
            Class.forName("com.tapjoy.TJPlacementListener").classLoader,
            arrayOf(Class.forName("com.tapjoy.TJPlacementListener")),
            tapjoyPlacementListener()
        )

        // Create the placement instance
        return getPlacementMethod.invoke(null, placementName, placementListenerInstance)
    }
}