package com.unity3d.ads.adplayer

import android.os.Bundle
import android.view.KeyEvent
import android.webkit.WebView
import androidx.activity.ComponentActivity
import androidx.lifecycle.lifecycleScope
import com.unity3d.ads.core.domain.SendDiagnosticEvent
import com.unity3d.ads.core.extensions.toBuiltInMap
import com.unity3d.services.core.di.IServiceComponent
import com.unity3d.services.core.di.ServiceProvider
import com.unity3d.services.core.di.inject
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.flow.filter
import kotlinx.coroutines.flow.launchIn
import kotlinx.coroutines.flow.onEach
import kotlinx.coroutines.launch
import org.json.JSONObject

/**
 * An activity that displays a fullscreen webview.
 */
class FullScreenWebViewDisplay : ComponentActivity(), IServiceComponent {
    private var opportunityId = ""
    private var showOptions: Map<String, Any?>? = null

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        val sendDiagnosticEvent by inject<SendDiagnosticEvent>()

        sendDiagnosticEvent(SendDiagnosticEvent.SHOW_AD_VIEWER_FULLSCREEN_INTENT_CREATION_STARTS)

        opportunityId = intent.getStringExtra("opportunityId")
            ?: run {
                setResult(RESULT_CANCELED)
                finish()
                lifecycleScope.launch {
                    AndroidFullscreenWebViewAdPlayer.displayMessages.emit(DisplayMessage.DisplayError(opportunityId, "Opportunity ID not found"))
                }
                sendDiagnosticEvent(SendDiagnosticEvent.SHOW_AD_VIEWER_FULLSCREEN_INTENT_CREATION_FAILS)
                return
            }
        intent.hasExtra("orientation")
            .takeIf { it }
            ?.let { requestedOrientation = intent.getIntExtra("orientation", -1) }

        showOptions = intent.getStringExtra("showOptions")?.let { JSONObject(it).toBuiltInMap() }
        listenToAdPlayerEvents()
        sendDiagnosticEvent(SendDiagnosticEvent.SHOW_AD_VIEWER_FULLSCREEN_INTENT_CREATION_SUCCESS)
        lifecycleScope.launch {
            AndroidFullscreenWebViewAdPlayer.displayMessages.emit(DisplayMessage.WebViewInstanceRequest(opportunityId))
        }
    }

    private fun listenToAdPlayerEvents() {
        AndroidFullscreenWebViewAdPlayer.displayMessages
            .filter { it.opportunityId == opportunityId }
            .onEach {
                when (it) {
                    is DisplayMessage.DisplayFinishRequest -> finish()
                    is DisplayMessage.WebViewInstanceResponse -> loadWebView(it.webView)
                    is DisplayMessage.SetOrientation -> requestedOrientation = it.orientation
                    else -> Unit
                }
            }
            .launchIn(lifecycleScope)
    }

    private fun loadWebView(webView: WebView) {
        lifecycleScope.launch {
            setContentView(webView)
            AndroidFullscreenWebViewAdPlayer.displayMessages.emit(DisplayMessage.DisplayReady(opportunityId, showOptions))
        }
    }

    override fun onKeyDown(keyCode: Int, event: KeyEvent?): Boolean {
        return keyCode == KeyEvent.KEYCODE_BACK
    }

    override fun onResume() {
        super.onResume()

        lifecycleScope.launch {
            AndroidFullscreenWebViewAdPlayer.displayMessages.emit(DisplayMessage.VisibilityChanged(opportunityId, true))
        }
    }

    override fun onPause() {
        super.onPause()

        lifecycleScope.launch {
            AndroidFullscreenWebViewAdPlayer.displayMessages.emit(DisplayMessage.VisibilityChanged(opportunityId, false))
        }
    }

    override fun onDestroy() {
        super.onDestroy()
        val sendDiagnosticEvent by inject<SendDiagnosticEvent>()

        sendDiagnosticEvent(SendDiagnosticEvent.SHOW_AD_VIEWER_FULLSCREEN_INTENT_DESTROYED)

        CoroutineScope(Dispatchers.Main).launch {
            AndroidFullscreenWebViewAdPlayer.displayMessages.emit(DisplayMessage.DisplayDestroyed(opportunityId))
        }
    }

    override fun onWindowFocusChanged(hasFocus: Boolean) {
        super.onWindowFocusChanged(hasFocus)

        lifecycleScope.launch {
            AndroidFullscreenWebViewAdPlayer.displayMessages.emit(DisplayMessage.FocusChanged(opportunityId, hasFocus))
        }
    }
}