package com.unity3d.ads.adplayer

import android.webkit.WebView

/**
 * This sealed class represents a message that can be sent between the AdPlayer and the Display.
 *
 * @property opportunityId The id of the opportunity that the message is related to.
 */
sealed class DisplayMessage(val opportunityId: String) {
    // region Display -> AdPlayer

    /**
     * Represents a message that is sent when the display is ready to be shown.
     * Display -> AdPlayer
     *
     * @param opportunityId The id of the opportunity that the message is related to.
     */
    class DisplayReady(opportunityId: String, val showOptions: Map<String, Any>? = null) : DisplayMessage(opportunityId)

    /**
     * Represents a message that is sent when the display needs to receive a WebView instance to be shown.
     * Display -> AdPlayer
     *
     * @param opportunityId The id of the opportunity that the message is related to.
     */
    class WebViewInstanceRequest(opportunityId: String) : DisplayMessage(opportunityId)

    /**
     * Represents a message that is sent when the display visibility has changed.
     * Display -> AdPlayer
     *
     * @param opportunityId The id of the opportunity that the message is related to.
     */
    class VisibilityChanged(opportunityId: String, val isVisible: Boolean) : DisplayMessage(opportunityId)

    /**
     * Represents a message that is sent when the display focus has changed.
     * Display -> AdPlayer
     *
     * @param opportunityId The id of the opportunity that the message is related to.
     */
    class FocusChanged(opportunityId: String, val isFocused: Boolean) : DisplayMessage(opportunityId)

    /**
     * Represents a message that is sent when the display is destroyed.
     * Display -> AdPlayer
     *
     * @param opportunityId The id of the opportunity that the message is related to.
     */
    class DisplayDestroyed(opportunityId: String) : DisplayMessage(opportunityId)

    /**
     * Represents a message that is sent when the Display encounters an error.
     * AdPlayer -> Display
     *
     * @param opportunityId The id of the opportunity that the message is related to.
     */
    class DisplayError(opportunityId: String, val reason: String) : DisplayMessage(opportunityId)

    // endregion Display -> AdPlayer

    // region AdPlayer -> Display

    /**
     * Represents a message that is sent when the display requested a WebView instance and the AdPlayer has one to be shown.
     * AdPlayer -> Display
     *
     * @param opportunityId The id of the opportunity that the message is related to.
     * @property webView The WebView instance to be shown.
     */
    class WebViewInstanceResponse(opportunityId: String, val webView: WebView) : DisplayMessage(opportunityId)

    /**
     * Represents a message that is sent when the AdPlayer requests the display to be finished.
     * AdPlayer -> Display
     *
     * @param opportunityId The id of the opportunity that the message is related to.
     */
    class DisplayFinishRequest(opportunityId: String) : DisplayMessage(opportunityId)

    // endregion AdPlayer -> Display
}