package com.unity3d.services.core.di

import ByteStringStoreOuterClass.ByteStringStore
import UniversalRequestStoreOuterClass
import WebviewConfigurationStore
import android.content.Context
import android.os.Handler
import android.os.Looper
import androidx.datastore.core.DataMigration
import androidx.datastore.core.DataStore
import androidx.datastore.core.DataStoreFactory
import androidx.datastore.core.handlers.ReplaceFileCorruptionHandler
import androidx.datastore.dataStoreFile
import com.google.android.gms.net.CronetProviderInstaller
import com.google.protobuf.ByteString
import com.unity3d.ads.core.configuration.AlternativeFlowReader
import com.unity3d.ads.core.data.datasource.AndroidByteStringDataSource
import com.unity3d.ads.core.data.datasource.AndroidStaticDeviceInfoDataSource
import com.unity3d.ads.core.data.datasource.ByteStringDataSource
import com.unity3d.ads.core.data.datasource.DefaultByteStringMigration
import com.unity3d.ads.core.data.datasource.ForcefulPreservingByteStringPreferenceMigration
import com.unity3d.ads.core.data.datasource.GetAuidData
import com.unity3d.ads.core.data.datasource.GetIdfiData
import com.unity3d.ads.core.data.datasource.PreservingByteStringPreferenceMigration
import com.unity3d.ads.core.data.model.ByteStringSerializer
import com.unity3d.ads.core.data.model.UniversalRequestStoreSerializer
import com.unity3d.ads.core.data.model.WebViewConfigurationStoreSerializer
import com.unity3d.ads.core.data.repository.DiagnosticEventRepository
import com.unity3d.ads.core.domain.SendDiagnosticEvent
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.SYSTEM_CRONET_FAILURE
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.SYSTEM_CRONET_SUCCESS
import com.unity3d.ads.core.extensions.toByteString
import com.unity3d.services.UnityAdsConstants
import com.unity3d.services.ads.measurements.MeasurementsService
import com.unity3d.services.ads.token.AsyncTokenStorage
import com.unity3d.services.ads.token.InMemoryAsyncTokenStorage
import com.unity3d.services.ads.token.InMemoryTokenStorage
import com.unity3d.services.ads.topics.TopicsService
import com.unity3d.services.core.device.StorageManager
import com.unity3d.services.core.device.StorageManager.StorageType
import com.unity3d.services.core.device.VolumeChange
import com.unity3d.services.core.device.VolumeChangeMonitor
import com.unity3d.services.core.di.ServiceProvider.CDN_CREATIVES_HOST
import com.unity3d.services.core.di.ServiceProvider.CDN_CREATIVES_PORT
import com.unity3d.services.core.di.ServiceProvider.DATA_STORE_AUID
import com.unity3d.services.core.di.ServiceProvider.DATA_STORE_GATEWAY_CACHE
import com.unity3d.services.core.di.ServiceProvider.DATA_STORE_GL_INFO
import com.unity3d.services.core.di.ServiceProvider.DATA_STORE_IAP_TRANSACTION
import com.unity3d.services.core.di.ServiceProvider.DATA_STORE_IDFI
import com.unity3d.services.core.di.ServiceProvider.DATA_STORE_NATIVE_CONFIG
import com.unity3d.services.core.di.ServiceProvider.DATA_STORE_PRIVACY
import com.unity3d.services.core.di.ServiceProvider.DATA_STORE_PRIVACY_FSM
import com.unity3d.services.core.di.ServiceProvider.DATA_STORE_UNIVERSAL_REQUEST
import com.unity3d.services.core.di.ServiceProvider.DATA_STORE_WEBVIEW_CONFIG
import com.unity3d.services.core.di.ServiceProvider.DEFAULT_DISPATCHER
import com.unity3d.services.core.di.ServiceProvider.GATEWAY_HOST
import com.unity3d.services.core.di.ServiceProvider.GATEWAY_PORT
import com.unity3d.services.core.di.ServiceProvider.HTTP_CACHE_DISK_SIZE
import com.unity3d.services.core.di.ServiceProvider.HTTP_CLIENT_FETCH_TIMEOUT
import com.unity3d.services.core.di.ServiceProvider.IO_DISPATCHER
import com.unity3d.services.core.di.ServiceProvider.MAIN_DISPATCHER
import com.unity3d.services.core.di.ServiceProvider.NAMED_GET_TOKEN_SCOPE
import com.unity3d.services.core.di.ServiceProvider.NAMED_INIT_SCOPE
import com.unity3d.services.core.di.ServiceProvider.NAMED_LOAD_SCOPE
import com.unity3d.services.core.di.ServiceProvider.NAMED_PUBLIC_JOB
import com.unity3d.services.core.di.ServiceProvider.NAMED_SDK
import com.unity3d.services.core.di.ServiceProvider.NAMED_SHOW_SCOPE
import com.unity3d.services.core.di.ServiceProvider.NAMED_TRANSACTION_SCOPE
import com.unity3d.services.core.di.ServiceProvider.PREF_AUID
import com.unity3d.services.core.di.ServiceProvider.PREF_DEFAULT
import com.unity3d.services.core.di.ServiceProvider.PREF_GL_INFO
import com.unity3d.services.core.di.ServiceProvider.PREF_IDFI
import com.unity3d.services.core.domain.ISDKDispatchers
import com.unity3d.services.core.domain.SDKDispatchers
import com.unity3d.services.core.domain.task.ConfigFileFromLocalStorage
import com.unity3d.services.core.misc.JsonStorage
import com.unity3d.services.core.network.core.CronetClient
import com.unity3d.services.core.network.core.HttpClient
import com.unity3d.services.core.network.core.LegacyHttpClient
import com.unity3d.services.core.network.core.OkHttp3Client
import com.unity3d.services.core.preferences.AndroidPreferences
import com.unity3d.services.core.properties.ClientProperties
import com.unity3d.services.core.request.metrics.SDKMetrics
import com.unity3d.services.core.request.metrics.SDKMetricsSender
import com.unity3d.services.core.webview.bridge.SharedInstances
import gateway.v1.NativeConfigurationOuterClass.AdOperationsConfiguration
import gateway.v1.NativeConfigurationOuterClass.NativeConfiguration
import gateway.v1.NativeConfigurationOuterClass.RequestPolicy
import gateway.v1.NativeConfigurationOuterClass.RequestRetryPolicy
import gateway.v1.NativeConfigurationOuterClass.RequestTimeoutPolicy
import gateway.v1.adOperationsConfiguration
import gateway.v1.diagnosticEventsConfiguration
import gateway.v1.featureFlags
import gateway.v1.nativeConfiguration
import gateway.v1.requestPolicy
import gateway.v1.requestRetryPolicy
import gateway.v1.requestTimeoutPolicy
import kotlinx.coroutines.CoroutineDispatcher
import kotlinx.coroutines.CoroutineExceptionHandler
import kotlinx.coroutines.CoroutineName
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.Job
import kotlinx.coroutines.SupervisorJob
import kotlinx.coroutines.runBlocking
import kotlinx.coroutines.suspendCancellableCoroutine
import kotlinx.coroutines.withTimeoutOrNull
import okhttp3.OkHttpClient
import org.chromium.net.CronetEngine
import org.koin.core.annotation.ComponentScan
import org.koin.core.annotation.Factory
import org.koin.core.annotation.Module
import org.koin.core.annotation.Named
import org.koin.core.annotation.Single
import org.koin.dsl.koinApplication
import org.koin.ksp.generated.com_unity3d_services_core_di_KoinModule
import java.util.*
import kotlin.coroutines.resume
import kotlin.time.DurationUnit
import kotlin.time.ExperimentalTime
import kotlin.time.TimeSource

@Module
@ComponentScan("com.unity3d")
class KoinModule {
    @Single
    fun androidContext(): Context = ClientProperties.getApplicationContext()

    @Single
    @Named(MAIN_DISPATCHER)
    fun mainDispatcher(): CoroutineDispatcher = Dispatchers.Main

    @Single
    @Named(DEFAULT_DISPATCHER)
    fun defaultDispatcher(): CoroutineDispatcher = Dispatchers.Default

    @Single
    @Named(IO_DISPATCHER)
    fun ioDispatcher(): CoroutineDispatcher = Dispatchers.IO

    @Single
    fun sdkDispatchers(): ISDKDispatchers = SDKDispatchers()

    @Single
    fun sdkMetrics(): SDKMetricsSender = SDKMetrics.getInstance()

    @Factory
    @Named(NAMED_INIT_SCOPE)
    fun initCoroutineScope(
        dispatchers: ISDKDispatchers,
        @Named(NAMED_SDK) errorHandler: CoroutineExceptionHandler,
        @Named(NAMED_PUBLIC_JOB) parentJob: Job,
    ): CoroutineScope = CoroutineScope(parentJob + dispatchers.default + CoroutineName(NAMED_INIT_SCOPE) + errorHandler)

    @Factory
    @Named(NAMED_LOAD_SCOPE)
    fun loadCoroutineScope(
        dispatchers: ISDKDispatchers,
        @Named(NAMED_SDK) errorHandler: CoroutineExceptionHandler,
        @Named(NAMED_PUBLIC_JOB) parentJob: Job,
    ): CoroutineScope = CoroutineScope(parentJob + dispatchers.default + CoroutineName(NAMED_LOAD_SCOPE) + errorHandler)

    @Factory
    @Named(NAMED_SHOW_SCOPE)
    fun showCoroutineScope(
        dispatchers: ISDKDispatchers,
        @Named(NAMED_SDK) errorHandler: CoroutineExceptionHandler,
        @Named(NAMED_PUBLIC_JOB) parentJob: Job,
    ): CoroutineScope = CoroutineScope(parentJob + dispatchers.default + CoroutineName(NAMED_SHOW_SCOPE) + errorHandler)

    @Factory
    @Named(NAMED_TRANSACTION_SCOPE)
    fun transactionCoroutineScope(
        dispatchers: ISDKDispatchers,
        @Named(NAMED_SDK) errorHandler: CoroutineExceptionHandler,
        @Named(NAMED_PUBLIC_JOB) parentJob: Job,
    ): CoroutineScope = CoroutineScope(parentJob + dispatchers.main + CoroutineName(NAMED_TRANSACTION_SCOPE) + errorHandler)

    @Factory
    @Named(NAMED_GET_TOKEN_SCOPE)
    fun getTokenCoroutineScope(
        dispatchers: ISDKDispatchers,
        @Named(NAMED_SDK) errorHandler: CoroutineExceptionHandler,
        @Named(NAMED_PUBLIC_JOB) parentJob: Job,
    ): CoroutineScope = CoroutineScope(parentJob + dispatchers.main + CoroutineName(NAMED_GET_TOKEN_SCOPE) + errorHandler)

    @Factory
    @Named(NAMED_PUBLIC_JOB)
    fun publicApiJob(
        diagnosticEventRepository: DiagnosticEventRepository,
    ): Job = Job().apply {
        invokeOnCompletion {
            diagnosticEventRepository.flush()
        }
    }

    @Single
    @Named(PREF_IDFI)
    fun idfiDataMigration(context: Context): DataMigration<ByteStringStore> = PreservingByteStringPreferenceMigration(
        context,
        AndroidStaticDeviceInfoDataSource.PREF_KEY_INSTALLINFO,
        AndroidStaticDeviceInfoDataSource.PREF_KEY_IDFI,
        GetIdfiData()
    )

    @Single
    @Named(PREF_AUID)
    fun auidDataMigration(context: Context): DataMigration<ByteStringStore> =
        ForcefulPreservingByteStringPreferenceMigration(
            context,
            AndroidStaticDeviceInfoDataSource.PREF_KEY_SUPERSONIC,
            AndroidStaticDeviceInfoDataSource.PREF_KEY_AUID,
            GetAuidData()
        )

    @Single
    @Named(PREF_DEFAULT)
    fun defaultByteStringMigration(): DataMigration<ByteStringStore> = DefaultByteStringMigration(
        AndroidStaticDeviceInfoDataSource.PREF_KEY_INSTALLINFO,
        AndroidStaticDeviceInfoDataSource.PREF_KEY_IDFI,
        GetIdfiData()
    )

    @Single
    @Named(DATA_STORE_GATEWAY_CACHE)
    fun gatewayDataStore(
        context: Context,
        @Named(IO_DISPATCHER) dispatcher: CoroutineDispatcher,
    ): DataStore<ByteStringStore> = provideByteStringDataStore(context, dispatcher, DATA_STORE_GATEWAY_CACHE)

    @Single
    @Named(DATA_STORE_PRIVACY)
    fun privacyDataStore(
        context: Context,
        @Named(IO_DISPATCHER) dispatcher: CoroutineDispatcher,
    ): DataStore<ByteStringStore> = provideByteStringDataStore(context, dispatcher, DATA_STORE_PRIVACY)

    @Single
    @Named(DATA_STORE_PRIVACY_FSM)
    fun privacyFsmDataStore(
        context: Context,
        @Named(IO_DISPATCHER) dispatcher: CoroutineDispatcher,
    ): DataStore<ByteStringStore> = provideByteStringDataStore(context, dispatcher, DATA_STORE_PRIVACY_FSM)

    @Single
    @Named(DATA_STORE_NATIVE_CONFIG)
    fun nativeConfigurationDataStore(
        context: Context,
        @Named(IO_DISPATCHER) dispatcher: CoroutineDispatcher,
    ): DataStore<ByteStringStore> = provideByteStringDataStore(context, dispatcher, DATA_STORE_NATIVE_CONFIG)

    @Single
    @Named(DATA_STORE_IDFI)
    fun idfiDataStore(
        context: Context,
        @Named(IO_DISPATCHER) dispatcher: CoroutineDispatcher,
        @Named(PREF_IDFI) idfiMigration: DataMigration<ByteStringStore>,
        @Named(PREF_DEFAULT) defaultIdfi: DataMigration<ByteStringStore>
    ): DataStore<ByteStringStore> = DataStoreFactory.create(
        serializer = ByteStringSerializer(),
        produceFile = { context.dataStoreFile(DATA_STORE_IDFI) },
        migrations = listOf(idfiMigration, defaultIdfi),
        corruptionHandler = ReplaceFileCorruptionHandler(
            produceNewData = {
                val idfi = UUID.randomUUID()
                AndroidPreferences.setString(
                    "unityads-installinfo",
                    "unityads-idfi",
                    idfi.toString()
                ) // legacy support
                ByteStringStore.newBuilder().setData(idfi.toByteString()).build()
            }
        ),
        scope = CoroutineScope(dispatcher + SupervisorJob())
    )

    @Single
    @Named(DATA_STORE_AUID)
    fun auidDataStore(
        context: Context,
        @Named(IO_DISPATCHER) dispatcher: CoroutineDispatcher,
        @Named(PREF_AUID) auidMigration: DataMigration<ByteStringStore>,
    ): DataStore<ByteStringStore> = DataStoreFactory.create(
        serializer = ByteStringSerializer(),
        produceFile = { context.dataStoreFile(DATA_STORE_AUID) },
        migrations = listOf(auidMigration),
        corruptionHandler = ReplaceFileCorruptionHandler(
            produceNewData = {
                ByteStringStore.newBuilder().setData(ByteString.empty()).build()
            }
        ),
        scope = CoroutineScope(dispatcher + SupervisorJob())
    )

    @Single
    @Named(DATA_STORE_GL_INFO)
    fun glInfoDataStore(
        context: Context,
        @Named(IO_DISPATCHER) dispatcher: CoroutineDispatcher,
        @Named(PREF_GL_INFO) fetchGLInfo: DataMigration<ByteStringStore>
    ): DataStore<ByteStringStore> = DataStoreFactory.create(
        serializer = ByteStringSerializer(),
        produceFile = { context.dataStoreFile(DATA_STORE_GL_INFO) },
        migrations = listOf(fetchGLInfo),
        scope = CoroutineScope(dispatcher + SupervisorJob())
    )

    @Single
    @Named(DATA_STORE_UNIVERSAL_REQUEST)
    fun universalRequestDataStore(
        context: Context,
        @Named(IO_DISPATCHER) dispatcher: CoroutineDispatcher
    ): DataStore<UniversalRequestStoreOuterClass.UniversalRequestStore> = DataStoreFactory.create(
        serializer = UniversalRequestStoreSerializer(),
        produceFile = { context.dataStoreFile(DATA_STORE_UNIVERSAL_REQUEST) },
        corruptionHandler = null,
        scope = CoroutineScope(dispatcher + SupervisorJob())
    )

    @Single
    @Named(DATA_STORE_IAP_TRANSACTION)
    fun iapTransactionDataStore(
        context: Context,
        @Named(IO_DISPATCHER) dispatcher: CoroutineDispatcher
    ): DataStore<ByteStringStore> = provideByteStringDataStore(context, dispatcher, DATA_STORE_IAP_TRANSACTION)


    @Single
    @Named(DATA_STORE_WEBVIEW_CONFIG)
    fun webViewConfigurationDataStore(
        context: Context,
        @Named(IO_DISPATCHER) dispatcher: CoroutineDispatcher
    ): DataStore<WebviewConfigurationStore.WebViewConfigurationStore> = DataStoreFactory.create(
        serializer = WebViewConfigurationStoreSerializer(),
        produceFile = { context.dataStoreFile(DATA_STORE_WEBVIEW_CONFIG) },
        corruptionHandler = null,
        scope = CoroutineScope(dispatcher + SupervisorJob())
    )

    @Single
    fun asyncTokenStorage(
        tokenStorage: InMemoryTokenStorage,
        sdkMetricsSender: SDKMetricsSender,
    ): AsyncTokenStorage = InMemoryAsyncTokenStorage(
        null,
        Handler(Looper.getMainLooper()),
        sdkMetricsSender,
        tokenStorage
    )

    @Single
    fun volumeChangeMonitor(
        volumeChange: VolumeChange,
    ): VolumeChangeMonitor = VolumeChangeMonitor(SharedInstances.webViewEventSender, volumeChange)

    @Single
    @Named("PUBLIC")
    fun publicJsonStorage(): JsonStorage = provideJsonStorage(StorageType.PUBLIC)

    @Single
    @Named("PRIVATE")
    fun privateJsonStorage(): JsonStorage = provideJsonStorage(StorageType.PRIVATE)

    @Single
    fun defaultNativeConfiguration(): NativeConfiguration = nativeConfiguration {
        adOperations = getDefaultAdOperations()
        initPolicy = getDefaultRequestPolicy()
        adPolicy = getDefaultRequestPolicy()
        otherPolicy = getDefaultRequestPolicy()
        operativeEventPolicy = getDefaultRequestPolicy()
        diagnosticEvents = diagnosticEventsConfiguration {
            enabled = true
            maxBatchSize = 10
            maxBatchIntervalMs = 30000
            ttmEnabled = false
        }
    }

    @Single
    @Named(DATA_STORE_GATEWAY_CACHE)
    fun gatewayCacheDataStore(@Named(DATA_STORE_GATEWAY_CACHE) dataStore: DataStore<ByteStringStore>) =
        provideByteStringDataSource(dataStore)

    @Single
    @Named(DATA_STORE_PRIVACY)
    fun privacyDataStore(@Named(DATA_STORE_PRIVACY) dataStore: DataStore<ByteStringStore>) =
        provideByteStringDataSource(dataStore)

    @Single
    @Named(DATA_STORE_IDFI)
    fun idfiDataStore(@Named(DATA_STORE_IDFI) dataStore: DataStore<ByteStringStore>) =
        provideByteStringDataSource(dataStore)

    @Single
    @Named(DATA_STORE_AUID)
    fun auidDataStore(@Named(DATA_STORE_AUID) dataStore: DataStore<ByteStringStore>) =
        provideByteStringDataSource(dataStore)

    @Single
    @Named(DATA_STORE_PRIVACY_FSM)
    fun privacyFsmDataStore(@Named(DATA_STORE_PRIVACY_FSM) dataStore: DataStore<ByteStringStore>) =
        provideByteStringDataSource(dataStore)

    @Single
    @Named(DATA_STORE_NATIVE_CONFIG)
    fun nativeConfigurationDataStore(@Named(DATA_STORE_NATIVE_CONFIG) dataStore: DataStore<ByteStringStore>) =
        provideByteStringDataSource(dataStore)

    @Single
    @Named(DATA_STORE_GL_INFO)
    fun glInfoDataStore(@Named(DATA_STORE_GL_INFO) dataStore: DataStore<ByteStringStore>) =
        provideByteStringDataSource(dataStore)

    @Single
    @Named(DATA_STORE_IAP_TRANSACTION)
    fun iapTransactionDataStore(@Named(DATA_STORE_IAP_TRANSACTION) dataStore: DataStore<ByteStringStore>) =
        provideByteStringDataSource(dataStore)

    @Single
    fun measurementService(
        context: Context,
        dispatchers: ISDKDispatchers,
    ): MeasurementsService = MeasurementsService(context, dispatchers, SharedInstances.webViewEventSender)

    @Single
    fun topicsService(
        context: Context,
        dispatchers: ISDKDispatchers
    ): TopicsService = TopicsService(context, dispatchers, SharedInstances.webViewEventSender)


    @OptIn(ExperimentalTime::class)
    @Single
    fun provideHttpClient(
        configFileFromLocalStorage: ConfigFileFromLocalStorage,
        alternativeFlowReader: AlternativeFlowReader,
        dispatchers: ISDKDispatchers,
        sendDiagnosticEvent: SendDiagnosticEvent,
        context: Context
    ): HttpClient = runBlocking {
        val isAlternativeFlowEnabled = alternativeFlowReader()
        if (isAlternativeFlowEnabled) {
            val startTime = TimeSource.Monotonic.markNow()
            val client = withTimeoutOrNull(HTTP_CLIENT_FETCH_TIMEOUT) {
                buildNetworkClient(context, dispatchers)
            }
            val diagnosticResult = if (client == null) SYSTEM_CRONET_FAILURE else SYSTEM_CRONET_SUCCESS
            sendDiagnosticEvent(diagnosticResult, startTime.elapsedNow().toDouble(DurationUnit.MILLISECONDS))
            client ?: OkHttp3Client(dispatchers, OkHttpClient())
        } else {
            val config = runBlocking {
                runCatching { configFileFromLocalStorage(ConfigFileFromLocalStorage.Params()) }.getOrNull()?.getOrNull()
            }
            if (config?.experiments?.isOkHttpEnabled == true) {
                OkHttp3Client(dispatchers, OkHttpClient())
            } else {
                LegacyHttpClient(dispatchers)
            }
        }
    }

    private suspend fun buildNetworkClient(
        context: Context,
        dispatchers: ISDKDispatchers
    ): HttpClient = suspendCancellableCoroutine { continuation ->
        CronetProviderInstaller.installProvider(context).addOnCompleteListener {
            if (it.isSuccessful) {
                val cronetEngine = CronetEngine.Builder(context)
                    .setStoragePath(context.filesDir.absolutePath)
                    .enableHttpCache(
                        CronetEngine.Builder.HTTP_CACHE_DISK,
                        HTTP_CACHE_DISK_SIZE
                    ) // HTTP_CACHE_DISK provides 0-RTT support
                    .enableQuic(true)
                    .addQuicHint(GATEWAY_HOST, GATEWAY_PORT, GATEWAY_PORT)
                    .addQuicHint(CDN_CREATIVES_HOST, CDN_CREATIVES_PORT, CDN_CREATIVES_PORT)
                    .build()
                continuation.resume(CronetClient(cronetEngine, dispatchers))
            } else {
                continuation.resume(OkHttp3Client(dispatchers, OkHttpClient()))
            }
        }
    }

    private fun provideJsonStorage(storageType: StorageType): JsonStorage {
        check(StorageManager.init(ClientProperties.getApplicationContext())) {
            "StorageManager failed to initialize"
        }
        return StorageManager.getStorage(storageType)
    }

    private fun provideByteStringDataSource(dataStore: DataStore<ByteStringStore>): ByteStringDataSource {
        return AndroidByteStringDataSource(dataStore)
    }

    private fun provideByteStringDataStore(
        context: Context,
        dispatcher: CoroutineDispatcher,
        dataStoreFile: String
    ): DataStore<ByteStringStore> {
        return DataStoreFactory.create(
            serializer = ByteStringSerializer(),
            produceFile = { context.dataStoreFile(dataStoreFile) },
            corruptionHandler = null,
            scope = CoroutineScope(dispatcher + SupervisorJob())
        )
    }

    private fun getDefaultAdOperations(): AdOperationsConfiguration {
        return adOperationsConfiguration {
            loadTimeoutMs = UnityAdsConstants.AdOperations.LOAD_TIMEOUT_MS
            showTimeoutMs = UnityAdsConstants.AdOperations.SHOW_TIMEOUT_MS
            getTokenTimeoutMs = UnityAdsConstants.AdOperations.GET_TOKEN_TIMEOUT_MS
        }
    }

    private fun getDefaultRequestPolicy(): RequestPolicy {
        return requestPolicy {
            retryPolicy = getDefaultRequestRetryPolicy()
            timeoutPolicy = getDefaultRequestTimeoutPolicy()
        }
    }

    private fun getDefaultRequestRetryPolicy(): RequestRetryPolicy {
        return requestRetryPolicy {
            maxDuration = UnityAdsConstants.RequestPolicy.RETRY_MAX_DURATION
            retryWaitBase = UnityAdsConstants.RequestPolicy.RETRY_WAIT_BASE
            retryJitterPct = UnityAdsConstants.RequestPolicy.RETRY_JITTER_PCT
            shouldStoreLocally = UnityAdsConstants.RequestPolicy.SHOULD_STORE_LOCALLY
            retryMaxInterval = UnityAdsConstants.RequestPolicy.RETRY_MAX_INTERVAL
            retryScalingFactor = UnityAdsConstants.RequestPolicy.RETRY_SCALING_FACTOR
        }
    }

    private fun getDefaultRequestTimeoutPolicy(): RequestTimeoutPolicy {
        return requestTimeoutPolicy {
            connectTimeoutMs = UnityAdsConstants.RequestPolicy.CONNECT_TIMEOUT_MS
            readTimeoutMs = UnityAdsConstants.RequestPolicy.READ_TIMEOUT_MS
            writeTimeoutMs = UnityAdsConstants.RequestPolicy.WRITE_TIMEOUT_MS
            overallTimeoutMs = UnityAdsConstants.RequestPolicy.OVERALL_TIMEOUT_MS
        }
    }

    companion object {
        val system = koinApplication {
            modules(com_unity3d_services_core_di_KoinModule)
        }
    }
}
