package com.unity3d.ads.adplayer

import android.os.Bundle
import android.view.KeyEvent
import android.webkit.WebView
import androidx.activity.ComponentActivity
import androidx.lifecycle.lifecycleScope
import kotlinx.coroutines.flow.filter
import kotlinx.coroutines.flow.launchIn
import kotlinx.coroutines.flow.onEach
import kotlinx.coroutines.launch

/**
 * An activity that displays a fullscreen webview.
 */
class FullScreenWebViewDisplay : ComponentActivity() {
    private var opportunityId = ""

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        opportunityId = intent.getStringExtra("opportunityId")
            ?: run {
                setResult(RESULT_CANCELED)
                finish()
                lifecycleScope.launch {
                    AndroidFullscreenWebViewAdPlayer.displayMessages.emit(DisplayMessage.DisplayError(opportunityId, "Opportunity ID not found"))
                }
                return
            }

        listenToAdPlayerEvents()
        lifecycleScope.launch {
            AndroidFullscreenWebViewAdPlayer.displayMessages.emit(DisplayMessage.WebViewInstanceRequest(opportunityId))
        }
    }

    private fun listenToAdPlayerEvents() {
        AndroidFullscreenWebViewAdPlayer.displayMessages
            .filter { it.opportunityId == opportunityId }
            .onEach {
                when (it) {
                    is DisplayMessage.DisplayFinishRequest -> finish()
                    is DisplayMessage.WebViewInstanceResponse -> loadWebView(it.webView)
                    else -> Unit
                }
            }
            .launchIn(lifecycleScope)
    }

    private fun loadWebView(webView: WebView) {
        lifecycleScope.launch {
            setContentView(webView)
            AndroidFullscreenWebViewAdPlayer.displayMessages.emit(DisplayMessage.DisplayReady(opportunityId))
        }
    }

    override fun onKeyDown(keyCode: Int, event: KeyEvent?): Boolean {
        return keyCode == KeyEvent.KEYCODE_BACK
    }

    override fun onResume() {
        super.onResume()

        lifecycleScope.launch {
            AndroidFullscreenWebViewAdPlayer.displayMessages.emit(DisplayMessage.VisibilityChanged(opportunityId, true))
        }
    }

    override fun onPause() {
        super.onPause()

        lifecycleScope.launch {
            AndroidFullscreenWebViewAdPlayer.displayMessages.emit(DisplayMessage.VisibilityChanged(opportunityId, false))
        }
    }

    override fun onWindowFocusChanged(hasFocus: Boolean) {
        super.onWindowFocusChanged(hasFocus)

        lifecycleScope.launch {
            AndroidFullscreenWebViewAdPlayer.displayMessages.emit(DisplayMessage.FocusChanged(opportunityId, hasFocus))
        }
    }
}