package com.twolight.android.loader;

import android.app.Activity;
import android.content.Context;
import android.net.Uri;
import android.support.annotation.DrawableRes;
import android.support.v4.app.Fragment;
import android.widget.ImageView;

import com.bumptech.glide.DrawableTypeRequest;
import com.bumptech.glide.Glide;
import com.bumptech.glide.RequestManager;
import com.bumptech.glide.load.DecodeFormat;
import com.bumptech.glide.load.engine.DiskCacheStrategy;
import com.bumptech.glide.request.target.Target;
import com.bumptech.glide.util.Util;
import com.twolight.android.loader.transformation.BlurTransformation;
import com.twolight.android.loader.transformation.CropCircleTransformation;
import com.twolight.android.loader.transformation.RoundedCornersTransformation;

import java.io.File;

/**
 * Created by twolight on 17/4/7.
 */

public class ImageLoader{

    public static Build with(Context context){
        return new Build(context);
    }

    public static Build with(Fragment fragment){
        return new Build(fragment.getContext());
    }

    public static Build with(Activity activity){
        return new Build(activity);
    }

    public static void pauseRequests(Context context){
        if(Util.isOnMainThread()){
            Glide.with(context.getApplicationContext()).pauseRequests();
        }

    }

    public static void resumeRequests(Context context){
        if(Util.isOnMainThread()){
            Glide.with(context.getApplicationContext()).resumeRequests();
        }
    }

    public static class Build{
        private Context mContext;

        private File mFile;
        private String mUrl;
        private Uri mUri;
        private @DrawableRes Integer mRes;

        private Blur mBlur;
        private Integer mCorner;
        private Config mConfig;
        private boolean mCircle;
        private @DrawableRes Integer mPlaceHolderRes;
        private @DrawableRes Integer mErrorRes;

        public Build(Context context){
            mContext = context;
        }

        public Build load(File file){
            mFile = file;
            return this;
        }

        public Build load(String url){
            mUrl = url;
            return this;
        }

        public Build load(Uri uri){
            mUri = uri;
            return this;
        }

        public Build load(@DrawableRes  Integer res){
            mRes = res;
            return this;
        }

        public Build error(@DrawableRes  Integer res){
            mErrorRes = res;
            return this;
        }

        public Build placeholder(@DrawableRes  Integer res){
            mPlaceHolderRes = res;
            return this;
        }

        public Build blur(Blur blur){
            mBlur = blur;
            return this;
        }

        public Build roundCorner(int roundCorner){
            mCorner = roundCorner;
            return this;
        }

        public Build circle(boolean circle){
            mCircle = circle;
            return this;
        }

        public Build reSize(int width,int height){
            return this;
        }

        public Build format(Config config){
            mConfig = config;
            return this;
        }

        public void into(ImageView imageView){
            if(mContext == null || imageView == null){
                return;
            }

            RequestManager requestManager = Glide.with(mContext);

            DrawableTypeRequest<?> drawableTypeRequest;

            if(mFile != null){
                drawableTypeRequest = requestManager.load(mFile);
            }else if(mUrl != null){
                drawableTypeRequest = requestManager.load(mUrl);
            }else if(mUri != null){
                drawableTypeRequest = requestManager.load(mUri);
            }else if(mRes != null){
                drawableTypeRequest = requestManager.load(mRes);
            }else{
                return;
            }

            if(mBlur != null){
                drawableTypeRequest.bitmapTransform(new BlurTransformation(mContext,mBlur.mRadius,mBlur.mSampling));
            }

            if(mCorner != null){
                drawableTypeRequest.bitmapTransform(new RoundedCornersTransformation(mContext,mCorner,0));
            }

            if(mCircle){
                drawableTypeRequest.bitmapTransform(new CropCircleTransformation(mContext));
            }

            if(mPlaceHolderRes != null){
                drawableTypeRequest.placeholder(mPlaceHolderRes);
            }

            if(mErrorRes != null) {
                drawableTypeRequest.error(mErrorRes);
            }

            drawableTypeRequest.dontAnimate();

            if(mConfig != null && mConfig == Config.ALPHA_8){
                drawableTypeRequest
                        .asBitmap()
                        .format(DecodeFormat.PREFER_ARGB_8888)
                        .asIs()
                        .override(Target.SIZE_ORIGINAL, Target.SIZE_ORIGINAL)
                        .into(imageView);
                return;
            }

            drawableTypeRequest
                    .diskCacheStrategy(DiskCacheStrategy.SOURCE)
                    .into(imageView);
        }
    }

    public static class Blur{
        private int mRadius;
        private int mSampling;

        public Blur(int radius, int sampling) {
            mRadius = radius;
            mSampling = sampling;
        }
    }

    public enum Config{
        ALPHA_8 (0),
        RGB_565 (1);

        final int nativeInt;

        Config(int i) {
            this.nativeInt = i;
        }
    }
}
