package com.twistpair.wave.thinclient.util;

import java.util.Iterator;
import java.util.Set;

import android.content.Context;
import android.content.Intent;
import android.content.pm.ApplicationInfo;
import android.content.pm.PackageManager;
import android.content.pm.PackageManager.NameNotFoundException;
import android.os.Bundle;
import android.util.SparseArray;
import android.view.MotionEvent;
import android.widget.TextView;
import android.widget.Toast;

import com.twistpair.wave.thinclient.logging.WtcLog;

public class WtcUtilsPlatform
{
    /**
     * @param context
     * @param resId resource id of the string to toast, or -1 for none
     */
    public static void toastLong(Context context, int resId)
    {
        toast(context, resId, Toast.LENGTH_LONG);
    }

    /**
     * @param context
     * @param text string to toast, or null/empty for none
     */
    public static void toastLong(Context context, String text)
    {
        toast(context, text, Toast.LENGTH_LONG);
    }

    /**
     * @param context
     * @param resId resource id of the string to toast, or -1 for none
     */
    public static void toastShort(Context context, int resId)
    {
        toast(context, resId, Toast.LENGTH_SHORT);
    }

    /**
     * @param context
     * @param text string to toast, or null/empty for none
     */
    public static void toastShort(Context context, String text)
    {
        toast(context, text, Toast.LENGTH_SHORT);
    }

    /**
     * @param context
     * @param resId resource id of the string to toast, or -1 for none
     * @param duration One of Toast.LENGTH_*
     */
    public static void toast(Context context, int resId, int duration)
    {
        if (resId != -1)
        {
            Toast.makeText(context, resId, duration).show();
        }
    }

    /**
     * @param context
     * @param text string to toast, or null/empty for none
     * @param duration One of Toast.LENGTH_*
     */
    public static void toast(Context context, String text, int duration)
    {
        if (!WtcString.isNullOrEmpty(text))
        {
            Toast.makeText(context, text, duration).show();
        }
    }

    /*
    public static void startActivityForResult(Activity activityFrom, Class<?> activityTo, Bundle extras, int requestCode)
    {
        Intent intent = new Intent(activityFrom, activityTo);
        if (extras != null)
        {
            intent.putExtras(extras);
        }
        activityFrom.startActivityForResult(intent, requestCode);
    }

    public static void startActivityForResult(Activity activityFrom, Class<?> activityTo, Intent intent, int requestCode)
    {
        startActivityForResult(activityFrom, activityTo, intent, requestCode);
    }

    public static void startActivity(Activity activityFrom, Class<?> activityTo, Bundle extras)
    {
        Intent intent = new Intent(activityFrom, activityTo);
        if (extras != null)
        {
            intent.putExtras(extras);
        }
        activityFrom.startActivity(intent);
    }

    public static void startActivityAndFinish(Activity activityFrom, Class<?> activityTo, Bundle extras)
    {
        startActivity(activityFrom, activityTo, extras);
        activityFrom.finish();
    }

    public static void startActivityAsFirstActivity(Activity activityFrom, Class<?> activityTo, Bundle extras)
    {
        Intent intent = new Intent(activityFrom, activityTo);
        if (extras != null)
        {
            intent.putExtras(extras);
        }
        intent.setFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
        activityFrom.startActivity(intent);
    }

    public static void startActivityAsFirstActivityAndFinish(Activity activityFrom, Class<?> activityTo, Bundle extras)
    {
        startActivityAsFirstActivity(activityFrom, activityTo, extras);
        activityFrom.finish();
    }
    */

    public static void toggleViewText(TextView view, boolean condition, int residTrue, int residFalse)
    {
        view.setText(condition ? residTrue : residFalse);
    }

    public static boolean equals(Class<?> a, Class<?> b)
    {
        return a.getCanonicalName().equals(b.getCanonicalName());
    }

    public static String getMetaDataString(Context context, String key)
    {
        PackageManager packageManager = context.getPackageManager();
        String packageName = context.getPackageName();

        ApplicationInfo applicationInfo = null;
        try
        {
            applicationInfo = packageManager.getApplicationInfo(packageName, PackageManager.GET_META_DATA);
        }
        catch (NameNotFoundException e)
        {
            e.printStackTrace();
            return null;
        }

        Bundle metaDataBundle = applicationInfo.metaData;

        String value = metaDataBundle.getString(key);
        return value;
    }

    public static String toString(Intent intent)
    {
        if (intent == null)
        {
            return "null";
        }

        StringBuilder sb = new StringBuilder();

        sb.append(intent.toString());

        Bundle bundle = intent.getExtras();
        sb.append(", extras=").append(toString(bundle));

        return sb.toString();
    }

    public static String toString(Bundle bundle)
    {
        if (bundle == null)
        {
            return "null";
        }

        StringBuilder sb = new StringBuilder();

        Set<String> keys = bundle.keySet();
        Iterator<String> it = keys.iterator();

        sb.append('{');
        while (it.hasNext())
        {
            String key = it.next();
            Object value = bundle.get(key);

            sb.append(WtcString.quote(key)).append('=').append(WtcString.quote(value));

            if (it.hasNext())
            {
                sb.append(", ");
            }
        }
        sb.append('}');

        return sb.toString();
    }

    /**
     * This and all supporting methods were shamelessly taken from:<br>
     *  https://github.com/android/platform_frameworks_base/blob/master/core/java/android/view/MotionEvent.java#L2980
     * @param tag
     * @param event
     */
    public static void logMotionEvent(String tag, String message, MotionEvent event)//, int filter)
    {
        int action = event.getAction();
        //if ((action & filter) == action)
        //{
        StringBuilder sb = new StringBuilder();

        if (!WtcString.isNullOrEmpty(message))
        {
            sb.append(message).append(' ');
        }

        sb.append("MotionEvent { action=").append(actionToString(action));

        final int pointerCount = event.getPointerCount();
        for (int i = 0; i < pointerCount; i++)
        {
            sb.append(", id[").append(i).append("]=").append(event.getPointerId(i));
            sb.append(", x[").append(i).append("]=").append(event.getX(i));
            sb.append(", y[").append(i).append("]=").append(event.getY(i));
            sb.append(", toolType[").append(i).append("]=").append(toolTypeToString(event.getToolType(i)));
        }

        sb.append(", buttonState=").append(buttonStateToString(event.getButtonState()));
        sb.append(", metaState=").append(metaStateToString(event.getMetaState()));
        sb.append(", flags=0x").append(Integer.toHexString(event.getFlags()));
        sb.append(", edgeFlags=0x").append(Integer.toHexString(event.getEdgeFlags()));
        sb.append(", pointerCount=").append(pointerCount);
        sb.append(", historySize=").append(event.getHistorySize());
        sb.append(", eventTime=").append(event.getEventTime());
        sb.append(", downTime=").append(event.getDownTime());
        sb.append(", deviceId=").append(event.getDeviceId());
        sb.append(", source=0x").append(Integer.toHexString(event.getSource()));
        sb.append(" }");

        WtcLog.info(tag, sb.toString());
        //}
    }

    /**
     * Returns a string that represents the symbolic name of the specified action
     * such as "ACTION_DOWN", "ACTION_POINTER_DOWN(3)" or an equivalent numeric constant
     * such as "35" if unknown.
     * Taken from https://github.com/android/platform_frameworks_base/blob/master/core/java/android/view/MotionEvent.java
     * @param action The action.
     * @return The symbolic name of the specified action.
     */
    public static String actionToString(int action)
    {
        switch (action)
        {
            case MotionEvent.ACTION_DOWN:
                return "ACTION_DOWN";
            case MotionEvent.ACTION_UP:
                return "ACTION_UP";
            case MotionEvent.ACTION_CANCEL:
                return "ACTION_CANCEL";
            case MotionEvent.ACTION_OUTSIDE:
                return "ACTION_OUTSIDE";
            case MotionEvent.ACTION_MOVE:
                return "ACTION_MOVE";
            case MotionEvent.ACTION_HOVER_MOVE:
                return "ACTION_HOVER_MOVE";
            case MotionEvent.ACTION_SCROLL:
                return "ACTION_SCROLL";
            case MotionEvent.ACTION_HOVER_ENTER:
                return "ACTION_HOVER_ENTER";
            case MotionEvent.ACTION_HOVER_EXIT:
                return "ACTION_HOVER_EXIT";
        }
        int index = (action & MotionEvent.ACTION_POINTER_INDEX_MASK) >> MotionEvent.ACTION_POINTER_INDEX_SHIFT;
        switch (action & MotionEvent.ACTION_MASK)
        {
            case MotionEvent.ACTION_POINTER_DOWN:
                return "ACTION_POINTER_DOWN(" + index + ")";
            case MotionEvent.ACTION_POINTER_UP:
                return "ACTION_POINTER_UP(" + index + ")";
            default:
                return Integer.toString(action);
        }
    }

    // Symbolic names of all tool types.
    private static final SparseArray<String> TOOL_TYPE_SYMBOLIC_NAMES = new SparseArray<String>();

    static
    {
        SparseArray<String> names = TOOL_TYPE_SYMBOLIC_NAMES;
        names.append(MotionEvent.TOOL_TYPE_UNKNOWN, "TOOL_TYPE_UNKNOWN");
        names.append(MotionEvent.TOOL_TYPE_FINGER, "TOOL_TYPE_FINGER");
        names.append(MotionEvent.TOOL_TYPE_STYLUS, "TOOL_TYPE_STYLUS");
        names.append(MotionEvent.TOOL_TYPE_MOUSE, "TOOL_TYPE_MOUSE");
        names.append(MotionEvent.TOOL_TYPE_ERASER, "TOOL_TYPE_ERASER");
    }

    /**
     * Returns a string that represents the symbolic name of the specified tool type
     * such as "TOOL_TYPE_FINGER" or an equivalent numeric constant such as "42" if unknown.
     * Taken from https://github.com/android/platform_frameworks_base/blob/master/core/java/android/view/MotionEvent.java
     * @param toolType The tool type.
     * @return The symbolic name of the specified tool type.
     */
    public static String toolTypeToString(int toolType)
    {
        String symbolicName = TOOL_TYPE_SYMBOLIC_NAMES.get(toolType);
        return symbolicName != null ? symbolicName : Integer.toString(toolType);
    }

    // Symbolic names of all button states in bit order from least significant
    // to most significant.
    private static final String[] BUTTON_SYMBOLIC_NAMES = new String[]
                                                        {
        "BUTTON_PRIMARY", //
        "BUTTON_SECONDARY", //
        "BUTTON_TERTIARY", //
        "BUTTON_BACK", //
        "BUTTON_FORWARD", //
        "0x00000020", //
        "0x00000040", //
        "0x00000080", //
        "0x00000100", //
        "0x00000200", //
        "0x00000400", //
        "0x00000800", //
        "0x00001000", //
        "0x00002000", //
        "0x00004000", //
        "0x00008000", //
        "0x00010000", //
        "0x00020000", //
        "0x00040000", //
        "0x00080000", //
        "0x00100000", //
        "0x00200000", //
        "0x00400000", //
        "0x00800000", //
        "0x01000000", //
        "0x02000000", //
        "0x04000000", //
        "0x08000000", //
        "0x10000000", //
        "0x20000000", //
        "0x40000000", //
        "0x80000000", //
                                                        };

    /**
     * Returns a string that represents the symbolic name of the specified combined
     * button state flags such as "0", "BUTTON_PRIMARY",
     * "BUTTON_PRIMARY|BUTTON_SECONDARY" or an equivalent numeric constant such as "0x10000000"
     * if unknown.
     * Taken from https://github.com/android/platform_frameworks_base/blob/master/core/java/android/view/MotionEvent.java
     * @param buttonState The button state.
     * @return The symbolic name of the specified combined button state flags.
     */
    public static String buttonStateToString(int buttonState)
    {
        if (buttonState == 0)
        {
            return "0";
        }
        StringBuilder result = null;
        int i = 0;
        while (buttonState != 0)
        {
            final boolean isSet = (buttonState & 1) != 0;
            buttonState >>>= 1; // unsigned shift!
            if (isSet)
            {
                final String name = BUTTON_SYMBOLIC_NAMES[i];
                if (result == null)
                {
                    if (buttonState == 0)
                    {
                        return name;
                    }
                    result = new StringBuilder(name);
                }
                else
                {
                    result.append('|');
                    result.append(name);
                }
            }
            i += 1;
        }
        return result.toString();
    }

    // Symbolic names of all metakeys in bit order from least significant to most significant.
    // Accordingly there are exactly 32 values in this table.
    private static final String[] META_SYMBOLIC_NAMES = new String[]
                                                      {
        "META_SHIFT_ON", //
        "META_ALT_ON", //
        "META_SYM_ON", //
        "META_FUNCTION_ON", //
        "META_ALT_LEFT_ON", //
        "META_ALT_RIGHT_ON", //
        "META_SHIFT_LEFT_ON", //
        "META_SHIFT_RIGHT_ON", //
        "META_CAP_LOCKED", //
        "META_ALT_LOCKED", //
        "META_SYM_LOCKED", //
        "0x00000800", //
        "META_CTRL_ON", //
        "META_CTRL_LEFT_ON", //
        "META_CTRL_RIGHT_ON", //
        "0x00008000", //
        "META_META_ON", //
        "META_META_LEFT_ON", //
        "META_META_RIGHT_ON", //
        "0x00080000", //
        "META_CAPS_LOCK_ON", //
        "META_NUM_LOCK_ON", //
        "META_SCROLL_LOCK_ON", //
        "0x00800000", //
        "0x01000000", //
        "0x02000000", //
        "0x04000000", //
        "0x08000000", //
        "0x10000000", //
        "0x20000000", //
        "0x40000000", //
        "0x80000000", //
                                                      };

    /**
     * Returns a string that represents the symbolic name of the specified combined meta
     * key modifier state flags such as "0", "META_SHIFT_ON",
     * "META_ALT_ON|META_SHIFT_ON" or an equivalent numeric constant such as "0x10000000"
     * if unknown.
     * Taken from https://github.com/android/platform_frameworks_base/blob/master/core/java/android/view/KeyEvent.java
     * @param metaState The meta state.
     * @return The symbolic name of the specified combined meta state flags.
     */
    public static String metaStateToString(int metaState)
    {
        if (metaState == 0)
        {
            return "0";
        }
        StringBuilder result = null;
        int i = 0;
        while (metaState != 0)
        {
            final boolean isSet = (metaState & 1) != 0;
            metaState >>>= 1; // unsigned shift!
            if (isSet)
            {
                final String name = META_SYMBOLIC_NAMES[i];
                if (result == null)
                {
                    if (metaState == 0)
                    {
                        return name;
                    }
                    result = new StringBuilder(name);
                }
                else
                {
                    result.append('|');
                    result.append(name);
                }
            }
            i += 1;
        }
        return result.toString();
    }
}
