package com.twistpair.wave.thinclient.protocol.headers;

import com.twistpair.wave.thinclient.protocol.WtcpConstants.WtcpMessageType;
import com.twistpair.wave.thinclient.util.IWtcMemoryStream;
import com.twistpair.wave.thinclient.util.WtcString;

public class WtcpMediaHeader extends IWtcpSubHeader
{
    public static final int SIZE                 = 2;
    public static final int SEQUENCE_NUMBER_BITS = 14;

    public int getOffset()
    {
        return WtcpHeader.SIZE;
    }

    public int getSize()
    {
        return SIZE;
    }

    public int getSequenceNumberBits()
    {
        return SEQUENCE_NUMBER_BITS;
    }

    public int getMessageType()
    {
        return WtcpMessageType.Media;
    }

    /// <summary>
    /// IsNotAudio              = 0b10000000 0b00000000 (mask 0x8000)
    /// Has16BitCrcAfterHeader  = 0b01000000 0b00000000 (mask 0x4000)
    /// SequenceNumber          = 0b00111111 0b11111111 (mask 0x3FFF)
    /// </summary>
    private int ushort0; // TODO:(pv) Make this mutable WtcUInt16

    /// <summary>
    /// IsNotAudio              = 0b10000000 0b00000000 (mask 0x8000)
    /// </summary>
    public boolean getIsNotAudio()
    {
        return ((ushort0 & 0x8000) >> 15) == 1;
    }

    public void setIsNotAudio(boolean value)
    {
        int mask = 0x8000;
        ushort0 = ((ushort0 & ~mask) | ((((value) ? 1 : 0) << 15) & mask));
    }

    /// <summary>
    /// Has16BitCrcAfterHeader  = 0b01000000 0b00000000 (mask 0x4000)
    /// </summary>
    public boolean getHas16BitCrcAfterHeader()
    {
        return ((ushort0 & 0x4000) >> 14) == 1;
    }

    public void setHas16BitCrcAfterHeader(boolean value)
    {
        int mask = 0x4000;
        ushort0 = ((ushort0 & ~mask) | ((((value) ? 1 : 0) << 14) & mask));
    }

    /// <summary>
    /// SequenceNumber          = 0b00111111 0b11111111 (mask 0x3FFF)
    /// </summary>
    public int getSequenceNumber()
    {
        return ((ushort0 & 0x3FFF) >> 0);
    }

    public void setSequenceNumber(int value)
    {
        int mask = 0x3FFF;
        ushort0 = ((ushort0 & ~mask) | ((value << 0) & mask));
    }

    public boolean getIsAudio()
    {
        return !getIsNotAudio();
    }

    public void setIsAudio(boolean value)
    {
        setIsNotAudio(!value);
    }

    /// <summary>
    /// Creates a new WtcpMediaHeader audio header (isNotAudio=false)
    /// </summary>
    public WtcpMediaHeader() //
    {
        this(false);
    }

    /// <summary>
    /// Creates a new WtcpMediaHeader with the given isNotAudio, has16BitCrcAfterHeader=false, and sequenceNumber=0
    /// </summary>
    /// <param name="isNotAudio"></param>
    public WtcpMediaHeader(boolean isNotAudio) //
    {
        this(isNotAudio, false, 0);
    }

    /// <summary>
    /// Creates a new WtcpMediaHeader with the given isNotAudio, has16BitCrcAfterHeader, and sequenceNumber
    /// </summary>
    /// <param name="isNotAudio"></param>
    /// <param name="has16BitCrcAfterHeader"></param>
    /// <param name="sequenceNumber"></param>
    public WtcpMediaHeader(boolean isNotAudio, boolean has16BitCrcAfterHeader, int sequenceNumber) //
    {
        super();
        setIsNotAudio(isNotAudio);
        setHas16BitCrcAfterHeader(has16BitCrcAfterHeader);
        setSequenceNumber(sequenceNumber);
    }

    /// <summary>
    /// Loads a WtcpMediaHeader from the given buffer
    /// </summary>
    /// <param name="buffer"></param>
    public WtcpMediaHeader(IWtcMemoryStream buffer) //
    {
        super(buffer);
    }

    public void dumpHostToNetworkOrder(IWtcMemoryStream buffer)
    {
        super.dumpHostToNetworkOrder(buffer);
        buffer.writeUInt16(ushort0);
    }

    public boolean loadNetworkToHostOrder(IWtcMemoryStream buffer)
    {
        if (!super.loadNetworkToHostOrder(buffer))
        {
            return false;
        }
        ushort0 = buffer.readUInt16();
        return true;
    }

    public void reset()
    {
        ushort0 = 0;
    }

    public String toString(char format)
    {
        format = Character.toLowerCase(format);
        StringBuffer sb = new StringBuffer();
        sb.append('{');
        switch (format)
        {
            case 'd':
                sb //
                   //.append("ushort0=0b").append(WtcString.toBitString(ushort0, 16)).append(" (") //
                .append("a=").append(getIsAudio()) //
                .append(",c=").append(getHas16BitCrcAfterHeader()) //
                .append(",s=").append(WtcString.formatNumber(getSequenceNumber(), 5)) //
                //.append(')') //
                ;
                break;
            default:
                sb //
                   //.append("ushort0=0b").append(WtcString.toBitString(ushort0, 16)).append(" (") //
                .append("a=").append(getIsAudio()) //
                .append(",c=").append(getHas16BitCrcAfterHeader()) //
                .append(",s=0x").append(WtcString.toHexString(getSequenceNumber(), 2))
                //.append(')')
                ;
                break;
        }
        sb.append('}');
        return sb.toString();
    }
}
