package com.twistpair.wave.thinclient.media;

import com.twistpair.wave.thinclient.logging.WtcLog;
import com.twistpair.wave.thinclient.media.WtcMediaDeviceSpeaker.IWtcMediaSpeakerStateListener;
import com.twistpair.wave.thinclient.util.WtcArrayBlockingQueue;
import com.twistpair.wave.thinclient.util.WtcArrayQueue;

public abstract class WtcMediaDeviceSpeaker
{
    private static final String TAG                           = WtcLog.TAG(WtcMediaDeviceSpeaker.class);

    public static final int     DEFAULT_AUTO_CLOSE_TIMEOUT_MS = 3000;

    public interface IWtcMediaSpeakerStateListener
    {
        /**
         * Here would be a good place for the listener to play a "on beep".
         */
        public void onSpeakerOpening();

        /**
         * Here would be a good place for the listener to play a "off beep".
         */
        public void onSpeakerClosed(boolean error);
    }

    public interface IWtcMediaSpeakerBufferListener
    {
        /**
         * @return true to allow the speaker to play the buffer; false to prevent the speaker from playing the buffer
         */
        public boolean onSpeakerBuffer();
    }

    private final WtcArrayBlockingQueue    mQueueEncoded;
    private final SparePool                mSparePool;

    protected WtcMediaCodec                mMediaDecoder;

    private IWtcMediaSpeakerStateListener  mListenerState;
    private IWtcMediaSpeakerBufferListener mListenerBuffer;

    // TODO:(pv) Fine tune this value via setAutoCloseTimeout, preferrably w/ a Debug screen/easter-egg
    private int                            mAutoCloseTimeout = DEFAULT_AUTO_CLOSE_TIMEOUT_MS;

    protected boolean                      mIgnoreRx         = true;

    protected WtcMediaDeviceSpeaker()
    {
        mQueueEncoded = new WtcArrayBlockingQueue("QueueSpeakerEncoded");
        mSparePool = new SparePool();
        setMediaDecoder(null);
    }

    public void setMediaDecoder(WtcMediaCodec mediaDecoder)
    {
        if (mediaDecoder == null)
        {
            // no media decoder defined defaults to raw byte copy
            mediaDecoder = new WtcMediaCodecRawCopy();
        }
        this.mMediaDecoder = mediaDecoder;
    }

    public void setStateListener(IWtcMediaSpeakerStateListener listener)
    {
        this.mListenerState = listener;
    }

    public void setBufferListener(IWtcMediaSpeakerBufferListener listener)
    {
        this.mListenerBuffer = listener;
    }

    public void maintenance(boolean clear)
    {
        mQueueEncoded.maintenance();
        mSparePool.maintenance(clear);
    }

    /**
     * Sets the duration of time to keep the speaker alive after a write before closing the speaker. 
     * @param timeout milliseconds
     */
    public void setAutoCloseTimeout(int timeout)
    {
        this.mAutoCloseTimeout = timeout;
    }

    public long getAutoCloseTimeout()
    {
        return mAutoCloseTimeout;
    }

    public boolean checkAutoCloseTimeout(long timeLastWrite)
    {
        if ((timeLastWrite + mAutoCloseTimeout) < System.currentTimeMillis())
        {
            WtcLog.warn(TAG, "Speaker inactivity timeout: timeLastWrite > " + mAutoCloseTimeout);
            return true;
        }
        return false;
    }

    public abstract void setVolume(int volume);

    public abstract int getVolume();

    public abstract boolean isOpen();

    public boolean isIgnoreRx()
    {
        return mIgnoreRx;
    }

    public void setIgnoreRx(boolean on)
    {
        this.mIgnoreRx = on;
    }

    protected void open() //
                    throws WtcMediaExceptionPlatform
    {
        IWtcMediaSpeakerStateListener listener = mListenerState;
        if (listener != null)
        {
            listener.onSpeakerOpening();
        }
        // Subclasses should add their logic here after calling this method...
    }

    /**
     * @param error true if the speaker is being closed due to an error, otherwise false
     */
    public void close(boolean error)
    {
        mQueueEncoded.clear();

        IWtcMediaSpeakerStateListener listener = mListenerState;
        if (listener != null)
        {
            listener.onSpeakerClosed(error);
        }
    }

    /**
     * See {@link IWtcMediaSpeakerStateListener#onSpeakerBuffer}
     * @return true to allow the speaker to play the buffer; false to prevent the speaker from playing the buffer
     */
    protected boolean onSpeakerBuffer()
    {
        boolean mayPlay = true;
        IWtcMediaSpeakerBufferListener listener = mListenerBuffer;
        if (listener != null)
        {
            mayPlay = listener.onSpeakerBuffer();
        }
        return mayPlay;
    }

    /**
     * Copy the encoded buffer to a queue for processing by another thread.
     * Another thread should call "dequeueDecoded(short[] bufferDecoded)" to get the *DECODED* buffer.
     * @param bufferEncoded
     * @param offset
     * @param length
     */
    protected void enqueueEncodedCopy(byte[] bufferEncoded, int offset, int length)
    {
        // Re-use any previously created spare, or create a new one:
        Spare spareEncoded = mSparePool.remove(length);
        System.arraycopy(bufferEncoded, offset, spareEncoded.buffer, 0, length);
        mQueueEncoded.add(spareEncoded);
    }

    /**
     * <p>Waits until an encoded buffer is queued, or a timeout occurs.
     * If a buffer is queued, it is dequeued and decoded in to memory pre-allocated by the caller.</p>
     * <p>Intentionally does *not* attempt to decode the buffer in a dedicated decoder thread (and adding the result to a queueDecoded).
     * It sounds tempting to do the actual decoding in a dedicated decoder thread, but there is little to no real net-gain in that.
     * If the decoding thread takes longer to decode the buffer than the playing thread takes to play the buffer,
     * then the playing thread will block waiting on the decoded queue, resulting in an extra context switch.
     * In the end it seem more efficient to do the decoding and playing in the same thread...<br>
     * ...as long as the total elapsed time is less than the buffer duration.<br>
     * If the total elapsed time is greater than the buffer duration then the decoder should not be used...period.</p>
     * @param bufferDecoded working buffer, pre-allocated by caller, of sufficient size to decode the encoded buffer in to
     * @return the number of *shorts* decoded, 0 for silence, or -1 if the queue timed out
     * @throws InterruptedException
     */
    protected int dequeueDecoded(short[] bufferDecoded) throws InterruptedException
    {
        int shortsDecoded = 0;

        if (onSpeakerBuffer())
        {
            Spare spareEncoded = (Spare) mQueueEncoded.poll(mAutoCloseTimeout);
            if (spareEncoded != null)
            {
                shortsDecoded = mMediaDecoder.decode(spareEncoded.buffer, spareEncoded.length, bufferDecoded);

                // Return the encoded buffer back to the spares queue for future re-use:
                mSparePool.add(spareEncoded);

                // TODO(pv): Add an event to send decoded buffer to listener.
            }
            else
            {
                shortsDecoded = -1;
            }
        }
        else
        {
            // Fill entire buffer with silence instead of dequeuing audio
            shortsDecoded = bufferDecoded.length;
            for (int i = 0; i < shortsDecoded; i++)
            {
                bufferDecoded[i] = 0;
            }
        }

        return shortsDecoded;
    }

    protected int dequeueDecoded(byte[] bufferDecoded) throws InterruptedException
    {
        // TODO:(pv) Better OOP way to get dequeueDecoded to work for both BlackBerry(bytes) and Android(shorts)
        return 0;
    }

    public void write(byte[] bufferEncoded, int offset, int length) //
                    throws WtcMediaExceptionPlatform
    {
        if (bufferEncoded == null)
        {
            return;
        }

        if (mIgnoreRx)
        {
            return;
        }

        if (!isOpen())
        {
            open();
        }

        enqueueEncodedCopy(bufferEncoded, offset, length);

        // TODO:(pv) Move as much non-platform-specific code from AndroidSpeaker in to this class that makes sense.
    }

    /**
     * Handles the edge-case situation where the length of a previously allocated buffer differs from what is being requested.<br>
     * If the requested length is less than the previously allocated length, then marks the end of the buffer.
     * If the requested length is greater than the previously allocated length, then allocates a new buffer.   
     */
    private static class Spare
    {
        private byte[] buffer = null;
        private int    length = 0;

        public Spare(int length)
        {
            setLength(length);
        }

        public void setLength(int length)
        {
            if (this.length < length)
            {
                // This should rarely/never happen; length should be constant
                //WtcLog.warn(TAG, "UNEXPECTED spare.length < length; new Spare(length=" + length + ")");
                this.buffer = new byte[length];
                this.length = length;
            }
        }
    }

    private static class SparePool
    {
        private final WtcArrayQueue queueSpares = new WtcArrayQueue("QueueSpeakerSpares");

        public void maintenance(boolean clear)
        {
            synchronized (queueSpares)
            {
                queueSpares.maintenance(clear);
            }
        }

        /**
         * Attempts to reuse memory from a queue of previously allocated memory; otherwise allocates new memory.
         * The intent is to save the CPU, memory, and garbage collection hit of perpetual allocations while streaming media.
         * @param length
         * @return
         */
        protected Spare remove(int length)
        {
            synchronized (queueSpares)
            {
                Spare spare;

                //WtcLog.info(TAG, "BEFORE queueSpares.size()=" + queueSpares.size());
                if (queueSpares.isEmpty())
                {
                    //WtcLog.info(TAG, "new Spare(length=" + length + ")");
                    spare = new Spare(length);
                }
                else
                {
                    //WtcLog.info(TAG, "queueSpares.remove()");
                    spare = (Spare) queueSpares.remove();
                    spare.setLength(length);
                }
                //WtcLog.info(TAG, "AFTER queueSpares.size()=" + queueSpares.size());

                return spare;
            }
        }

        protected void add(Spare spare)
        {
            synchronized (queueSpares)
            {
                //WtcLog.info(TAG, "queueSpares.add(spare)");
                queueSpares.add(spare);
            }
        }
    }
}
