package com.twistpair.wave.thinclient.logging;

import java.text.SimpleDateFormat;
import java.util.Date;

import android.os.Process;
import android.util.Log;

import com.twistpair.wave.thinclient.util.WtcString;

public class WtcLogPlatform
{
    public static final int       VERBOSE     = Log.VERBOSE;
    public static final int       FATAL       = 0;          // Log.ASSERT;
    public static final int       ERROR       = Log.ERROR;
    public static final int       WARN        = Log.WARN;
    public static final int       INFO        = Log.INFO;
    public static final int       DEBUG       = Log.DEBUG;

    private static final String[] LEVEL_NAMES = new String[]
                                              {
        "F", // 0
        "?", // 1
        "T", // 2
        "D", // 3
        "I", // 4
        "W", // 5
        "E", // 6
                                              };

    public static boolean isLoggable(String tag, int level)
    {
        return Log.isLoggable(tag, level);
    }

    /**
     * You can change the default level by setting a system property: 'setprop log.tag.&lt;YOUR_LOG_TAG&gt; &lt;LEVEL&gt;'
     * Where level is either VERBOSE, DEBUG, INFO, WARN, ERROR, ASSERT, or SUPPRESS.
     * SUPPRESS will turn off all logging for your tag.
     * You can also create a local.prop file that with the following in it:
     * 'log.tag.&lt;YOUR_LOG_TAG&gt;=&lt;LEVEL&gt;' and place that in /data/local.prop.
     * 
     * @param tag
     * @param level
     */
    public static void setTagLevel(String tag, int level)
    {
        // ignore
    }

    private static final SimpleDateFormat DATE_FORMAT = new SimpleDateFormat("MM-dd HH:mm:ss.SSS");

    /**
     * @param tag
     * @param level
     * @param msg
     * @param e
     * @return "MM-dd HH:mm:ss.SSS PID TID Level Tag Message Throwable"
     */
    public static String format(String tag, int level, String msg, Throwable e)
    {
        StringBuffer sb = new StringBuffer() //
        .append(DATE_FORMAT.format(new Date())) //
        .append(' ').append(WtcString.padNumber(Process.myPid(), ' ', 5)) //
        .append(' ').append(WtcString.padNumber(Process.myTid(), ' ', 5)) //
        .append(' ').append(LEVEL_NAMES[level]) //
        .append(' ').append(tag) //
        .append(' ').append(msg);
        if (e != null)
        {
            sb.append(": throwable=").append(Log.getStackTraceString(e));
        }
        return sb.toString();
    }

    /**
     * Prints a line to LogCat.
     * On Android, "System.out.println(...)" also prints to LogCat.
     * Do *NOT* "System.out.println(...)"; it would add a [near] duplicated line to LogCat. 
     * @param tag
     * @param level
     * @param msg
     * @param e
     * @return the resulting formatted string
     */
    public static String println(String tag, int level, String msg, Throwable e)
    {
        // LogCat does its own formatting (time, level, pid, tag, message).
        // Delay most custom formatting until *after* the message is printed to LogCat.

        // LogCat does not output the Thread ID; prepend msg with it here.
        StringBuffer sb = new StringBuffer();
        sb.append('T').append(Process.myTid()).append(' ').append(msg);

        // LogCat does not output the exception; append msg with it here. 
        if (e != null)
        {
            sb.append(": throwable=").append(Log.getStackTraceString(e));
            // null the exception so that format(...) [below] doesn't append it to msg a second time.
            e = null;
        }

        msg = sb.toString();

        // print to LogCat
        Log.println(level, tag, msg);

        // Now we can format the message for use by the caller 
        msg = format(tag, level, msg, e);

        // Again, do not "System.out.println(...)"; it would only add a [near] duplicate line to LogCat.

        return msg;
    }

    public static String getStackTraceString(Object caller, Throwable throwable)
    {
        return Log.getStackTraceString(throwable);
    }
}
